/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.dd.SipApplicationListeners;
import com.ericsson.ssa.sip.PathNode.Type;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.sip.Address;
import javax.servlet.sip.SipSessionEvent;
import javax.servlet.sip.SipSessionListener;
import javax.servlet.sip.URI;


public class SipSessionDialogImpl extends SipSessionImplBase
    implements Serializable {
    private static final long serialVersionUID = 3762535598732096310L;
    private transient ArrayList<SipSessionListener> sessionListeners = null;
    private Map<String, Object> sessionAttributeMap;
    private SipApplicationSessionImpl sipApplicationSession = null;
    private SipSessionManager sipSessionManager = null;
    private String handler = null;
    private DialogFragment dialog = null;
    private Type type = null;
    private String id; // final
    protected String description;
    private Address to; // Final
    private long creationDate = new Date().getTime(); // Final
    private long lastAccessedDate;
    private DialogSet dialogSet; // Final
    private boolean isDerived = false;
    private AtomicInteger cSeq = null;
    private URI remoteTarget = null;
    private boolean is1xxReliableOngoing = false;
    private boolean is1xxReliableSDP = false;
    private boolean updateOngoing = false;

    // JSR289
    private String routingRegion = null;
    private URI subscriberURI = null;
    private String linkedSipSessionId = null;

    /**
     * This boolean controls the values that getLocalParty() and
     * getRemoteParty() should return. It is false by default, which provides
     * the correct behaviour for UAC and proxy SipSessions. When an application
     * has decided to take the UAS role this attribute is set to true for the
     * SipSession, causing getLocalParty() and getRemoteParty() to respond
     * differently.
     */
    private boolean swapLocalRemote = false;

    public SipSessionDialogImpl(SipSessionManagerBase manager, DialogSet set,
        Address to, SipApplicationSessionImpl appSession, String handler,
        Type type) {
        super();
        sipSessionManager = manager;
        dialogSet = set;
        // remove to-tag
        this.to = (Address) ((AddressImpl) to).clone(false, false);
        ((AddressImpl) this.to).setReadOnly(true);
        sipApplicationSession = (SipApplicationSessionImpl) appSession;
        this.handler = handler;
        this.type = type;

        // if user agent then a cseq counter is needed, otherwise save memory...
        if (type.equals(Type.Caller) || type.equals(Type.Callee)) {
            cSeq = new AtomicInteger(SipFactoryImpl.START_CSEQ);
        }

        // important that id exist before adding to app session...
        id = createID().toString();
        sipApplicationSession.addSession(this);
        initListeners(sipApplicationSession);

        description = "SipSession with id " + manager.getApplicationId() + ":" +
            id;
    }

    /**
     * For serialization only.
     */
    public SipSessionDialogImpl() {
    }

    protected final void notifySessionCreated() {
        if (sessionListeners == null) {
            return;
        }

        for (Iterator<SipSessionListener> lIter = sessionListeners.iterator();
                lIter.hasNext();) {
            SipSessionListener listener = lIter.next();
            listener.sessionCreated(new SipSessionEvent(this));
        }
    }

    protected final void notifySessionDestroyed() {
        if (sessionListeners == null) {
            return;
        }

        for (Iterator<SipSessionListener> lIter = sessionListeners.iterator();
                lIter.hasNext();) {
            SipSessionListener listener = lIter.next();
            listener.sessionDestroyed(new SipSessionEvent(this));
        }
    }

    protected final void initListeners(
        SipApplicationSessionImpl sipApplicationSession) {
        SipApplicationListeners sipapplisteners = sipApplicationSession.getSipApplicationListeners();

        if (sipapplisteners != null) {
            setSipSessionListeners(sipapplisteners.getSipSessionListeners());
        }

        notifySessionCreated();
    }

    protected void setSipSessionListeners(
        ArrayList<SipSessionListener> sipSessionListeners) {
        this.sessionListeners = sipSessionListeners;
    }

    @Override
    protected void removeMeFromManager() {
        sipSessionManager.removeSipSession(this);
    }

    @Override
    protected Object getFromPFieldSessionAttribute(String name) {
        return getSessionAttributeMap(false).get(name);
    }

    @Override
    protected void addToPFieldSessionAttribute(String name, Object value) {
        getSessionAttributeMap(true).put(name, value);
    }

    @Override
    protected Collection<String> getFromPFieldSessionAttributeNames() {
        Collection<String> c = getSessionAttributeMap(false).keySet();

        if (c == null) {
            c = new ArrayList<String>(1);
        }

        c.add(SipFactoryImpl.REMOTE_TARGET);

        return c;
    }

    @Override
    protected void removeFromPFieldSessionAttribute(String name) {
        getSessionAttributeMap(false).remove(name);
    }

    private Map<String, Object> getSessionAttributeMap(boolean create) {
        if (sessionAttributeMap == null) {
            if (create) {
                sessionAttributeMap = new ConcurrentHashMap<String, Object>();
            } else {
                return Collections.emptyMap();
            }
        }

        return sessionAttributeMap;
    }

    @Override
    protected boolean getPField1xxReliableOngoing() {
        return is1xxReliableOngoing;
    }

    @Override
    protected boolean getPField1xxReliableSDP() {
        return is1xxReliableSDP;
    }

    @Override
    protected int incrementAndGetPFieldCSeq() {
        return cSeq.incrementAndGet();
    }

    @Override
    protected boolean hasPFieldCSeq() {
        return cSeq != null;
    }

    @Override
    protected void createPFieldCSeq() {
        cSeq = new AtomicInteger(SipFactoryImpl.START_CSEQ);
    }

    @Override
    protected long getPFieldCreationDate() {
        return creationDate;
    }

    @Override
    protected boolean getPFieldDerived() {
        return isDerived;
    }

    @Override
    protected DialogFragment getPFieldDialog() {
        return dialog;
    }

    @Override
    protected DialogSet getPFieldDialogSet() {
        return dialogSet;
    }

    @Override
    protected String getPFieldHandler() {
        return handler;
    }

    @Override
    protected String getPFieldId() {
        return id;
    }

    @Override
    protected long getPFieldLastAccessedDate() {
        return lastAccessedDate;
    }

    @Override
    protected String getPFieldLinkedSipSessionId() {
        return linkedSipSessionId;
    }

    @Override
    protected String getPFieldRoutingRegion() {
        return routingRegion;
    }

    @Override
    protected URI getPFieldRemoteTarget() {
        return remoteTarget;
    }

    @Override
    protected SipApplicationSessionImpl getPFieldSipApplicationSession() {
        return sipApplicationSession;
    }

    @Override
    protected SipSessionManager getPFieldSipSessionManagerField() {
        return sipSessionManager;
    }

    @Override
    protected URI getPFieldSubscriberURI() {
        return subscriberURI;
    }

    @Override
    protected boolean getPFieldSwapLocalRemote() {
        return swapLocalRemote;
    }

    @Override
    protected Address getPFieldTo() {
        return to;
    }

    @Override
    protected Type getPFieldType() {
        return type;
    }

    @Override
    protected boolean getPFieldUpdateOngoing() {
        return updateOngoing;
    }

    @Override
    protected void setPField1xxReliableOngoing(boolean is1xxReliableOngoing) {
        this.is1xxReliableOngoing = is1xxReliableOngoing;
    }

    @Override
    protected void setPField1xxReliableSDP(boolean is1xxReliableSDP) {
        this.is1xxReliableSDP = is1xxReliableSDP;
    }

    @Override
    protected void setPFieldCreationDate(long creationDate) {
        this.creationDate = creationDate;
    }

    @Override
    protected void setPFieldDerived(boolean isDerived) {
        this.isDerived = isDerived;
    }

    @Override
    protected void setPFieldDialog(DialogFragment dialog) {
        this.dialog = dialog;
    }

    @Override
    protected void setPFieldDialogSet(DialogSet dialogSet) {
        this.dialogSet = dialogSet;
    }

    @Override
    protected void setPFieldHandler(String handler) {
        this.handler = handler;
    }

    @Override
    protected void setPFieldId(String id) {
        this.id = id;
    }

    @Override
    protected void setPFieldLinkedSipSessionId(String linkedSipSessionId) {
        this.linkedSipSessionId = linkedSipSessionId;
    }

    @Override
    protected void setPFieldRemoteTarget(URI remoteContact) {
        this.remoteTarget = remoteContact;
    }

    @Override
    protected void setPFieldRoutingRegion(String routingRegion) {
        this.routingRegion = routingRegion;
    }

    @Override
    protected void setPFieldSipApplicationSession(
        SipApplicationSessionImpl sipApplicationSession) {
        this.sipApplicationSession = (SipApplicationSessionImpl) sipApplicationSession;
    }

    @Override
    protected void setPFieldSipSessionManager(
        SipSessionManager sipSessionManager) {
        this.sipSessionManager = sipSessionManager;
    }

    @Override
    protected void setPFieldSubscriberURI(URI subscriberURI) {
        this.subscriberURI = subscriberURI;
    }

    @Override
    protected void setPFieldSwapLocalRemote(boolean swapLocalRemote) {
        this.swapLocalRemote = swapLocalRemote;
    }

    @Override
    protected void setPFieldTo(Address to) {
        this.to = to;
    }

    @Override
    protected void setPFieldType(Type type) {
        this.type = type;
    }

    @Override
    protected void setPFieldUpdateOngoing(boolean updateOngoing) {
        this.updateOngoing = updateOngoing;
    }

    @Override
    protected void setPFieldLastAccessedDate(long lastAccessedDate) {
        this.lastAccessedDate = lastAccessedDate;
    }

    public void setShouldBePersisted() {
        sipApplicationSession.setSipSessionShouldBePersisted(this);
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeUTF(sipSessionManager.getApplicationId());
        out.writeUTF(sipApplicationSession.getId());

        Map<String, Object> sessionAttributes = getSessionAttributeMap(false);
        out.writeInt(sessionAttributes.size());

        for (Map.Entry<String, Object> entry : sessionAttributes.entrySet()) {
            out.writeUTF(entry.getKey());
            out.writeObject(entry.getValue());
        }

        out.writeUTF(handler);
        out.writeObject(dialog);
        out.writeObject(type);
        out.writeUTF(id);
        out.writeObject(to);
        out.writeLong(creationDate);
        out.writeLong(lastAccessedDate);
        out.writeObject(dialogSet);
        out.writeBoolean(isDerived);
        out.writeBoolean(swapLocalRemote);

        if (cSeq != null) {
            out.writeInt(cSeq.get());
        } else {
            out.writeInt(1);
        }

        // JSR289
        out.writeObject(subscriberURI);
        out.writeUTF(routingRegion);
    }

    private void readObject(ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        // Read the context name, get corresponding SipSessionManager, and use
        // it to look up the SipApplicationSession with the given id
        String appId = in.readUTF();
        sipSessionManager = SipSessionManagerBase.get(appId);

        String sasId = in.readUTF();
        sipApplicationSession = (SipApplicationSessionImpl) sipSessionManager.findSipApplicationSession(sasId);

        if (sipApplicationSession != null) {
            sipApplicationSession.addSession(this);
        }

        int size = in.readInt();

        if (size > 0) {
            sessionAttributeMap = new ConcurrentHashMap<String, Object>();

            for (int i = 0; i < size; i++) {
                sessionAttributeMap.put(in.readUTF(), in.readObject());
            }
        }

        handler = in.readUTF();
        dialog = (DialogFragment) in.readObject();
        type = (Type) in.readObject();
        id = in.readUTF();
        description = "SipSession with id " + appId + ":" + id;
        to = (Address) in.readObject();
        creationDate = in.readLong();
        lastAccessedDate = in.readLong();
        dialogSet = (DialogSet) in.readObject();
        isDerived = in.readBoolean();
        swapLocalRemote = in.readBoolean();

        if (type.equals(Type.Caller) || type.equals(Type.Callee)) {
            cSeq = new AtomicInteger(in.readInt());
        }

        // JSR289
        subscriberURI = (URI) in.readObject();
        routingRegion = in.readUTF();

        initListeners(sipApplicationSession);
    }

    public String toString() {
        return description;
    }
}
