/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.config.LayerHandler;
import com.ericsson.ssa.config.lease.Lease;
import com.ericsson.ssa.config.lease.LeaseExpiredException;
import com.ericsson.ssa.container.SipBindingResolver;
import com.ericsson.ssa.container.sim.ServletDispatcher;
import com.ericsson.ssa.sip.PathNode.Type;
import com.ericsson.ssa.sip.dns.TargetTuple;

import org.apache.catalina.session.SessionLock;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.sip.Address;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.SipServletMessage;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;
import javax.servlet.sip.SipSession;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.UAMode;
import javax.servlet.sip.URI;


/**
 * The SipSession implementation represents the SSA point-to-point SIP
 * relationship which is almost equal to a SIP dialog (it extends the dialog
 * definition, see specification). Important that this class will use getters
 * and setters internally since it should be possible to serialize the content
 * to external media.
 */
public abstract class SipSessionImplBase implements SipSessionBase {
    // --- Transient fields ---
    private static final long serialVersionUID = 3762535598732096310L;
    public final static String SIP_SESSIONID_DELIMITER = "##";
    private static final Logger logger = (Logger) Logger.getLogger(
            "SipContainer");
    private boolean isValid = true;
    private Config config = ConfigFactory.getConfig();
    private final SessionLock sessionLock = new SessionLock();
    private final Object ssLock = new Object();
    private final B2buaHelperPendingMessages pendingMessages = new B2buaHelperPendingMessages();
    private SipSession.State sessionState = State.INITIAL;

    protected SipSessionImplBase() {
    }

    public boolean isValid() {
        return isValid;
    }

    /**
     * Creates a unique id for this session
     *
     * @return the unique id
     */
    protected StringBuilder createID() {
        StringBuilder id = new StringBuilder(UUID.randomUUID().toString());
        id.append(SIP_SESSIONID_DELIMITER);

        if (Boolean.getBoolean("sip.module.use_new_config")) {
            try {
                Lease<TargetTuple[]> lease = SipBindingResolver.instance()
                                                               .lease(SipBindingResolver.PUBLIC_BINDING_CTX);
                System.out.println("SipSessionDialogImpl: \n" +
                    lease.getResource()[0]);
                id.append(lease.getResource()[0].getIP());
            } catch (LeaseExpiredException e) {
                // TODO logging
                e.printStackTrace();
            }
        } else {
            id.append(config.get(Constants.SIP_PUBLIC_HOST));
        }

        return id;
    }

    public long getCreationTime() {
        return getPFieldCreationDate();
    }

    public String getId() {
        return getPFieldId();
    }

    public long getLastAccessedTime() {
        // FIXME not implemented
        return -1;
    }

    public void invalidate() {
        // if called from SipServlet code
        invalidate(false);
    }

    public void invalidate(boolean hasTimedOut) {
        synchronized (ssLock) {
            if (isValid()) {
                isValid = false;

                if (getPFieldDialog() != null) {
                    getPFieldDialog().invalidate(hasTimedOut);
                    setPFieldDialog(null);
                } else if (getPFieldDialogSet() != null) {
                    // must be early dialog then...
                    getPFieldDialogSet().removeEarlyDialog();
                }

                // If the call from the SipApplicationSession then m_IsValid
                // is
                // false,
                // But if the call made directly on the session then
                // m_IsValid is
                // true.
                if (!getPFieldSipApplicationSession().isValid()) {
                    setPFieldSipApplicationSession(null);
                }

                removeMeFromManager();
                notifySessionDestroyed();
            } else {
                throw new IllegalStateException("The session is not valid.");
            }
        }
    }

    protected abstract void removeMeFromManager();

    public SipApplicationSession getApplicationSession() {
        return getPFieldSipApplicationSession();
    }

    public SipApplicationSessionImpl getApplicationSessionImpl() {
        return getPFieldSipApplicationSession();
    }

    public String getCallId() {
        return getPFieldDialogSet().getCallId();
    }

    public Address getLocalParty() {
        return getPFieldSwapLocalRemote() ? getTo()
                                          : getPFieldDialogSet().getFrom();
    }

    public Address getRemoteParty() {
        return getPFieldSwapLocalRemote() ? getPFieldDialogSet().getFrom()
                                          : getTo();
    }

    /**
     * Returns the from address for caller and to address for callee, otherwise
     * null
     *
     * @return the from address for caller and to address for callee, otherwise
     *         null
     */
    private Address getLocalSide() {
        Address add = null;

        if (getPFieldType().equals(Type.Caller)) {
            add = getPFieldDialogSet().getFrom();
        } else if (getPFieldType().equals(Type.Callee)) {
            add = getTo();
        } else {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Unable to decide if its caller or callee. Session id = " +
                    getId());
            }
        }

        return add;
    }

    /**
     * Returns the to address for caller and from address for callee, otherwise
     * null
     *
     * @return the to address for caller and from address for callee, otherwise
     *         null
     */
    private Address getOtherside() {
        Address add = null;

        if (getPFieldType().equals(Type.Caller)) {
            add = getTo();
        } else if (getPFieldType().equals(Type.Callee)) {
            add = getPFieldDialogSet().getFrom();
        } else {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Unable to decide if its caller or callee. Session id = " +
                    getId());
            }
        }

        return add;
    }

    /**
     * SSA 7.1.2 requestURI, Call-ID, From, To, CSeq and Route headers
     */
    public SipServletRequest createRequest(String method) {
        if ((getPFieldDialog() != null) && (getFromTag() != null) &&
                (getToTag() != null) && hasPFieldCSeq()) {
            Address from = (Address) ((AddressImpl) getLocalSide()).clone(true,
                    true);
            Address to = (Address) ((AddressImpl) getOtherside()).clone(true,
                    true);

            if ((from == null) || (to == null)) {
                throw new IllegalStateException();
            }

            SipServletRequestImpl req = new SipServletRequestImpl(method,
                    getRemoteTarget(), SipFactoryImpl.PROTOCOL_LINE);
            req.setDirection(getPFieldType());

            // set To
            Header toHeader = new SingleLineHeader(Header.TO, true);
            toHeader.setAddressValue(to, false);
            req.setHeader(toHeader);

            // set From
            Header fromHeader = new SingleLineHeader(Header.FROM, true);
            fromHeader.setAddressValue(from, false);
            req.setHeader(fromHeader);

            // set Max-Forwards
            Header maxForwardsHeader = new SingleLineHeader(Header.MAX_FORWARDS,
                    false);
            maxForwardsHeader.setValue("70", false);
            req.setHeader(maxForwardsHeader);

            // set CallID
            Header callIDHeader = new SingleLineHeader(Header.CALL_ID, true);
            callIDHeader.setValue(getCallId(), false);
            req.setHeader(callIDHeader);

            // set CSeq
            Header cSeqHeader = new SingleLineHeader(Header.CSEQ, true);
            cSeqHeader.setValue(Integer.toString(incrementAndGetPFieldCSeq()) +
                " " + req.getMethod(), false);
            req.setHeader(cSeqHeader);
            // update new request with session
            req.setSession(this);
            // update it with dialog
            req.setDialog(getPFieldDialog());
            // subsequent request
            req.setInitial(false);

            List<Layer> layers = LayerHandler.getInstance().getLayers();
            req._applicationStack.addAll(layers);
            req.setRole(getPFieldSipApplicationSession().getName());

            if (SipFactoryImpl.isDialogCreational(req.getMethod()) ||
                    "UPDATE".equals(req.getMethod())) {
                SessionManager.getInstance().addContact(req);
            }

            return req;
        } else {
            throw new IllegalStateException("Not allowed to create a request.");
        }
    }

    public void setHandler(String name) throws ServletException {
        ServletDispatcher servletDispatcher = getPFieldSipApplicationSession()
                                                  .getServletDispatcher();

        if (servletDispatcher != null) {
            if ((name == null) || !servletDispatcher.findServlet(name)) {
                throw new ServletException(
                    "Could not found the servlet to set it as a handler.");
            }
        } else {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Unexpected: Called setHandler, but the servlet dispatcher is null.");
            }
        }

        setPFieldHandler(name);
    }

    public String getHandler() {
        return getPFieldHandler();
    }

    public Object getAttribute(String name) {
        if (name.equals(SipFactoryImpl.REMOTE_TARGET)) {
            return getRemoteTarget();
        }

        if (!isValid()) {
            throw new IllegalStateException("The session is not valid.");
        }

        return getFromPFieldSessionAttribute(name);
    }

    public Enumeration<String> getAttributeNames() {
        if (!isValid()) {
            throw new IllegalStateException("The session is not valid.");
        }

        // Since Map doesn't have Enumeration as interface need to convert via
        // private class EnumerationConverter.
        Collection<String> c = getFromPFieldSessionAttributeNames();

        if (c == null) {
            c = new ArrayList<String>(1);
        }

        c.add(SipFactoryImpl.REMOTE_TARGET);

        return new EnumerationConverter<String>(c);
    }

    public void setAttribute(String name, Object attribute) {
        if (name.equals(SipFactoryImpl.REMOTE_TARGET)) {
            throw new IllegalStateException("reserved key.");
        }

        if (!isValid()) {
            throw new IllegalStateException("The session is not valid.");
        }

        addToPFieldSessionAttribute(name, attribute);
    }

    public void removeAttribute(String name) {
        if (!isValid()) {
            throw new IllegalStateException("The session is not valid.");
        }

        removeFromPFieldSessionAttribute(name);
    }

    /**
     * @return the to tag for this session.
     */
    public String getToTag() {
        return getTo().getParameter(AddressImpl.TAG_PARAM);
    }

    /**
     * @return true if no to tag has been set in this dialog otherwise false.
     */
    public boolean hasNoToTag() {
        return getPFieldDialog() == null;
    }

    /**
     * @return the from tag of this session.
     */
    public String getFromTag() {
        return getPFieldDialogSet().getFromTag();
    }

    public synchronized void setType(Type type) {
        setPFieldType(type);

        // if user agent then a cseq counter is needed, otherwise save memory...
        if (!hasPFieldCSeq() &&
                (getPFieldType().equals(Type.Caller) ||
                getPFieldType().equals(Type.Callee))) {
            createPFieldCSeq();
        }
    }

    /**
     * The session is associated with a dialog that must already be established
     * with a proper toTag. If this is the first message retrieving the session
     * the original session is returned otherwise a derived session is created
     * and returned.
     *
     * @param d
     *            the established dialog to associate the session with
     * @return the original or a derived session
     */
    private SipSessionImplBase getOriginalOrDerivedSession(DialogFragment d,
        String toTag) {
        if ((d.getToTag() != null) && (toTag != null) &&
                toTag.equals(d.getToTag())) {
            SipSessionImplBase s = null;

            if (getPFieldDialog() == null) {
                // double checked locking pattern
                synchronized (this) {
                    if (getPFieldDialog() == null) {
                        // return original session
                        s = this;
                        // set the dialog fragment in the session
                        setPFieldDialog(d);
                        // register the session in the dialog fragment...
                        getPFieldDialog().registerSession(s);

                        // ...and set toTag of session
                        AddressImpl toAddress = (AddressImpl) getTo();
                        toAddress.setReadOnly(false);
                        toAddress.setParameter(AddressImpl.TAG_PARAM, toTag);
                        toAddress.setReadOnly(true);
                    } else {
                        // return derived session
                        s = getSipSessionManager()
                                .createSipSession(getPFieldDialogSet(),
                                getTo(), getApplicationSessionImpl(),
                                getHandler(), getPFieldType());
                        s.setPFieldDerived(true);
                        // set the dialog fragment in the session
                        s.setPFieldDialog(d);
                        // register the session in the dialog fragment...
                        s.getPFieldDialog().registerSession(s);

                        // ...and set toTag of session
                        AddressImpl toAddress = (AddressImpl) s.getTo();
                        toAddress.setReadOnly(false);
                        toAddress.setParameter(AddressImpl.TAG_PARAM, toTag);
                        toAddress.setReadOnly(true);
                    }
                }
            } else {
                // return derived session
                s = getSipSessionManager()
                        .createSipSession(getPFieldDialogSet(), getTo(),
                        getApplicationSessionImpl(), getHandler(),
                        getPFieldType());
                s.setPFieldDerived(true);
                // set the dialog fragment in the session
                s.setPFieldDialog(d);
                // register the session in the dialog fragment
                s.getPFieldDialog().registerSession(s);

                // ...and set toTag of session
                AddressImpl toAddress = (AddressImpl) s.getTo();
                toAddress.setReadOnly(false);
                toAddress.setParameter(AddressImpl.TAG_PARAM, toTag);
                toAddress.setReadOnly(true);
            }

            return s;
        } else {
            throw new IllegalStateException(
                "DialogFragment must have same to-tag as message.");
        }
    }

    public SipSessionBase getOriginalOrDerivedSessionAndRegisterDialog(
        SipServletRequestImpl req, DialogFragment d) {
        // need to swap to and from for request
        String toTag = req.getFrom().getParameter(AddressImpl.TAG_PARAM);

        return getOriginalOrDerivedSessionAndRegisterDialog(req, d, toTag);
    }

    public SipSessionBase getOriginalOrDerivedSessionAndRegisterDialog(
        SipServletResponseImpl resp, DialogFragment d) {
        String toTag = resp.getTo().getParameter(AddressImpl.TAG_PARAM);

        return getOriginalOrDerivedSessionAndRegisterDialog(resp, d, toTag);
    }

    /**
     * Tries to register the DialogFragment with the toTag of the message. If
     * successful a original or derived session is returned. If unsuccessful a
     * confirmed DialogFragment with the toTag of the message is searched for.
     * If found the message is updated with this DialogFragment and a original
     * or derived session is returned. If no DialogFragment is found the
     * original DialogFragment is cloned and it is recursively sent to this
     * method.
     *
     * @param m
     *            the message with the toTag to be set in the DialogFragment
     * @param d
     *            the DialogFragment to register the session in.
     * @return an original or derived session
     */
    private SipSessionBase getOriginalOrDerivedSessionAndRegisterDialog(
        SipServletMessageImpl m, DialogFragment d, String toTag) {
        if (toTag != null) {
            SipSessionBase s = null;
            boolean isDialogCreational = SipFactoryImpl.isDialogCreational(m.getMethod());
            boolean success = d.tryToSetToTagAndRegisterDialog(toTag,
                    isDialogCreational);

            if (success) {
                s = getOriginalOrDerivedSession(d, toTag);
            } else {
                DialogFragment searchedDialog = getPFieldDialogSet()
                                                    .searchForDialog(toTag,
                        m.getFragmentId());

                if (searchedDialog != null) {
                    m.setDialog(searchedDialog);
                    s = getOriginalOrDerivedSession(searchedDialog, toTag);
                } else {
                    // clone the dialog
                    DialogFragment clone = (DialogFragment) d.clone();
                    // FIXME its a recursive call and a break condition must be
                    // added
                    s = getOriginalOrDerivedSessionAndRegisterDialog(m, clone,
                            toTag);
                }
            }

            return s;
        } else {
            throw new IllegalStateException(
                "DialogFragment must have same to-tag as message.");
        }
    }

    public void swapLocalRemote() {
        setPFieldSwapLocalRemote(true);
    }

    public boolean setUpdateOngoing() {
        if (getPFieldUpdateOngoing()) {
            return false;
        } else {
            setPFieldUpdateOngoing(true);

            return true;
        }
    }

    public void resetUpdateOngoing() {
        setPFieldUpdateOngoing(false);
    }

    // set to true the flag which specify that there is an 1xx Reliable
    // response ongoing and if an sdp is included (CR38 PRACK)
    //
    // @param sdp=true if an sdp is included in the 1xx rel response
    // @return true if there was no pending 1xx reliable response
    // false if there was pending 1xx reliable response
    public synchronized boolean set1xxReliableOngoing(boolean sdp) {
        if (is1xxReliableOngoing()) {
            return false;
        }

        setPField1xxReliableSDP(sdp);
        setPField1xxReliableOngoing(true);

        return true;
    }

    public void reset1xxReliable() {
        setPField1xxReliableOngoing(false);
        setPField1xxReliableSDP(false);
    }

    public int getFragmentId() {
        return (getPFieldDialog() != null) ? getPFieldDialog().getFragmentId()
                                           : (-1);
    }

    public SipSession.State getState() {
        synchronized (ssLock) {
            return sessionState;
        }
    }

    private void setState(SipSession.State state) {
        synchronized (ssLock) {
            sessionState = state;
        }
    }

    /**
     * Implements the logic for JSR289 <code>SipSession.State</code> This
     * method is called on every response and may update the state of the
     * SipSession.<br>
     *
     * JSR 289 6.2.1 Figure 6. The SipSession state machine.
     *
     * <pre>
     * INITIAL ---&gt; 100xx EARLY
     * INITIAL ---&gt; 200xx CONFIRMED
     * EARLY ---&gt; 3xx-6xx INITIAL
     * EARLY ---&gt; 2xx CONFIRMED
     * </pre>
     *
     * @param sipServletResponseImpl
     *           Current response.
     * @param ua
     *           UserAgent that is handling this dialog.
     */
    public void updateSipSessionState(SipServletResponse resp,
        PathNode.Type type) {
        //SipSessionBase sipSessionBase = sipServletResponseImpl.getSessionImpl();
        String method = resp.getRequest().getMethod();
        int status = resp.getStatus();

        // From JSR 289 "6.2.1 Relationship to SIP Dialogs"
        // The exception to the general rule
        // is that it does not apply to requests (e.g. BYE, CANCEL) that are
        // dialog terminating
        if (method.equals("BYE") || method.equals("CANCEL")) {
            setState(State.TERMINATED);

            return;
        }

        switch (getState()) {
        case INITIAL:

            // If the servlet acts as a UAC and sends a dialog creating request,
            // then the SipSession state tracks directly the SIP dialog state.
            // If the servlet acts as a UAS and receives a dialog creating
            // request
            // then the
            // SipSession state directly tracks the SIP dialog state.
            if (((status >= 100) && (status <= 199)) &&
                    SipFactoryImpl.isDialogCreational(method)) {
                setState(State.EARLY);
            } else if (((status >= 200) && (status <= 299)) &&
                    SipFactoryImpl.isDialogCreational(method)) {
                setState(State.CONFIRMED);
            }
            // 4. If the servlet acts as a UAS and receives a dialog creating
            // request, then the SipSession state directly tracks the SIP dialog
            // state. Unlike a UAC,
            // a non-2XX final response sent by the UAS in the EARLY or INITIAL
            // states
            // causes the SipSession state to go directly to the TERMINATED
            // state.
            else if ((type == Type.Callee) &&
                    ((status >= 300) && (status <= 699)) &&
                    SipFactoryImpl.isDialogCreational(method)) {
                setState(State.TERMINATED);
            }

            break;

        case EARLY:

            // 3. If the servlet acts as a UAC and sends a dialog creating
            // request,
            // then the SipSession state tracks directly the SIP dialog state
            // except that
            // non-2XX final responses received in the EARLY or INITIAL states
            // cause the SipSession state to return to the INITIAL state rather
            // than going to TERMINATED.
            // 5. If the servlet acts as a proxy for a dialog creating request
            // then
            // the SipSession state tracks the SIP dialog state except that
            // non-2XX
            // final responses received from downstream in the EARLY or INITIAL
            // states cause the SipSession state to return to INITIAL rather
            // than
            // going to TERMINATED.
            if (((type == Type.Caller) || (type == Type.Proxy)) &&
                    ((status >= 300) && (status <= 699)) &&
                    SipFactoryImpl.isDialogCreational(method)) {
                setState(State.INITIAL);
            }
            // 4. If the servlet acts as a UAS and receives a dialog creating
            // request, then the SipSession state directly tracks the SIP dialog
            // state. Unlike a UAC,
            // a non-2XX final response sent by the UAS in the EARLY or INITIAL
            // states
            // causes the SipSession state to go directly to the TERMINATED
            // state.
            else if ((type == Type.Callee) &&
                    ((status >= 300) && (status <= 699)) &&
                    SipFactoryImpl.isDialogCreational(method)) {
                setState(State.TERMINATED);
            } else if (((status >= 200) && (status <= 299)) &&
                    SipFactoryImpl.isDialogCreational(method)) {
                setState(State.CONFIRMED);
            }

            break;

        case CONFIRMED:
            break;

        case TERMINATED:
            break;
        }
    }

    /**
     * Implements the logic for JSR289 <code>SipSession.State</code> This
     * method is called on every request and may update the state of the
     * SipSession.<br>
     *
     * @param req
     */
    public void updateSipSessionState(SipServletRequest req) {
        String method = req.getMethod();

        // SUBSCRIPTION_STATE
        // From RFC 3265
        // A subscription is destroyed when a notifier sends a NOTIFY request
        // with
        // a "Subscription-State" of "terminated".
        if (method.equals("NOTIFY")) {
            String subState = req.getHeader("Subscription-State");

            if ("terminated".equalsIgnoreCase(subState)) {
                setState(State.TERMINATED);
            }
        }
        // From JSR 289 "6.2.1 Relationship to SIP Dialogs"
        // The exception to the general rule
        // is that it does not apply to requests (e.g. BYE, CANCEL) that are
        // dialog terminating
        else if (method.equals("BYE") || method.equals("CANCEL")) {
            setState(State.TERMINATED);
        }
    }

    public boolean isOngoingTransaction() {
        throw new RuntimeException(" Not yet implemented ");
    }

    public void setOutboundInterface(SipURI uri) {
        throw new RuntimeException(" Not yet implemented ");
    }

    /**
     * Gets the name of the instance.that currently owns this SipSession.
     *
     * @return The name of the instance that currently owns this SipSession
     */
    public String getCurrentOwnerInstanceName() {
        return getSipSessionManager().getInstanceName();
    }

    /**
     * get this session locked for foreground if the session is found to be
     * presently background locked; retry logic in a time-decay polling loop
     * waits for background lock to clear after 6 attempts (12.6 seconds) it
     * unlocks the session and acquires the foreground lock
     */
    protected boolean getSessionLockForForeground() {
        boolean result = false;

        // now lock the session
        long pollTime = 200L;
        int tryNumber = 0;
        int numTries = 7;
        boolean keepTrying = true;
        boolean lockResult = false;

        // try to lock up to numTries (i.e. 7) times
        // poll and wait starting with 200 ms
        while (keepTrying) {
            lockResult = lockForeground();

            if (lockResult) {
                keepTrying = false;
                result = true;

                break;
            }

            tryNumber++;

            if (tryNumber < (numTries - 1)) {
                pollTime = pollTime * 2L;
            } else {
                // unlock the background so we can take over
                // FIXME: need to log warning for this situation
                unlockBackground();
            }
        }

        return result;
    }

    /**
     * return whether this session is currently foreground locked
     */
    public synchronized boolean isForegroundLocked() {
        return sessionLock.isForegroundLocked();
    }

    /**
     * lock the session for foreground returns true if successful; false if
     * unsuccessful
     */
    public synchronized boolean lockBackground() {
        return sessionLock.lockBackground();
    }

    /**
     * lock the session for background returns true if successful; false if
     * unsuccessful
     */
    public synchronized boolean lockForeground() {
        return sessionLock.lockForeground();
    }

    /**
     * unlock the session completely irregardless of whether it was foreground
     * or background locked
     */
    public synchronized void unlockForegroundCompletely() {
        sessionLock.unlockForegroundCompletely();
    }

    /**
     * unlock the session from foreground
     */
    public synchronized void unlockForeground() {
        sessionLock.unlockForeground();
    }

    /**
     * unlock the session from background
     */
    public synchronized void unlockBackground() {
        sessionLock.unlockBackground();
    }

    /**
     * return the Session lock
     */
    public SessionLock getSessionLock() {
        return sessionLock;
    }

    /**
     * Returns true if this SipSession is replicable, false otherwise.
     *
     * @return true if this SipSession is replicable, false otherwise
     */
    public boolean isReplicable() {
        return getPFieldSipApplicationSession().isReplicable();
    }

    public URI getRemoteTarget() {
        return getPFieldRemoteTarget();
    }

    public void setRemoteTarget(URI contact) {
        setPFieldRemoteTarget(contact);
    }

    // Used by ApplicationDispatcher to set this before invoking of servlets
    public void setRegion(String region) {
        setPFieldRoutingRegion(region);
    }

    public String getRegion() {
        return getPFieldRoutingRegion();
    }

    // Used by ApplicationDispatcher to set this before invoking of servlets
    public void setSubscriberURI(URI subscriberURI) {
        setPFieldSubscriberURI(subscriberURI);
    }

    public URI getSubscriberURI() throws IllegalStateException {
        return getPFieldSubscriberURI();
    }

    // JSR289
    public String getLinkedSipSessionId() {
        return getPFieldLinkedSipSessionId();
    }

    public void setLinkedSipSessionId(String id) {
        setPFieldLinkedSipSessionId(id);
    }

    public SipSessionManager getSipSessionManager() {
        return getPFieldSipSessionManagerField();
    }

    public Address getTo() {
        return getPFieldTo();
    }

    public boolean isDerived() {
        return getPFieldDerived();
    }

    public boolean is1xxReliableOngoing() {
        return getPField1xxReliableOngoing();
    }

    public boolean is1xxReliableSDP() {
        return getPField1xxReliableSDP();
    }

    public List<SipServletMessage> getPendingMessages(UAMode mode) {
        return pendingMessages.get(mode);
    }

    // --- Persistable fields (PField) ---
    protected abstract String getPFieldHandler();

    protected abstract void setPFieldHandler(String name);

    protected abstract void setPFieldLinkedSipSessionId(String id);

    protected abstract String getPFieldLinkedSipSessionId();

    protected abstract void setPFieldSubscriberURI(URI subscriberURI);

    protected abstract URI getPFieldSubscriberURI();

    protected abstract URI getPFieldRemoteTarget();

    protected abstract void setPFieldRemoteTarget(URI contact);

    protected abstract Object getFromPFieldSessionAttribute(String name);

    protected abstract void addToPFieldSessionAttribute(String name,
        Object value);

    protected abstract Collection<String> getFromPFieldSessionAttributeNames();

    protected abstract void removeFromPFieldSessionAttribute(String name);

    protected abstract SipApplicationSessionImpl getPFieldSipApplicationSession();

    protected abstract void setPFieldSipApplicationSession(
        SipApplicationSessionImpl sipApplicationSession);

    protected abstract DialogFragment getPFieldDialog();

    protected abstract void setPFieldDialog(DialogFragment dialog);

    protected abstract long getPFieldCreationDate();

    protected abstract void setPFieldCreationDate(long creationDate);

    protected abstract long getPFieldLastAccessedDate();

    protected abstract void setPFieldLastAccessedDate(long lastAccessedDate);

    protected abstract DialogSet getPFieldDialogSet();

    protected abstract void setPFieldDialogSet(DialogSet dialogSet);

    protected abstract int incrementAndGetPFieldCSeq();

    protected abstract boolean hasPFieldCSeq();

    protected abstract void createPFieldCSeq();

    protected abstract void setPField1xxReliableSDP(boolean is1xxReliableSDP);

    protected abstract boolean getPField1xxReliableSDP();

    protected abstract boolean getPField1xxReliableOngoing();

    protected abstract void setPField1xxReliableOngoing(
        boolean is1xxReliableOngoing);

    protected abstract boolean getPFieldUpdateOngoing();

    protected abstract void setPFieldUpdateOngoing(boolean updateOngoing);

    protected abstract void setPFieldRoutingRegion(String routingRegion);

    protected abstract String getPFieldRoutingRegion();

    protected abstract boolean getPFieldSwapLocalRemote();

    protected abstract void setPFieldSwapLocalRemote(boolean swapLocalRemote);

    protected abstract Type getPFieldType();

    protected abstract void setPFieldType(Type type);

    protected abstract void setPFieldSipSessionManager(
        SipSessionManager sipSessionManager);

    protected abstract SipSessionManager getPFieldSipSessionManagerField();

    protected abstract void setPFieldId(String id);

    protected abstract void setPFieldTo(Address to);

    protected abstract Address getPFieldTo();

    protected abstract void setPFieldDerived(boolean isDerived);

    protected abstract boolean getPFieldDerived();

    protected abstract String getPFieldId();

    protected abstract void notifySessionDestroyed();
}
