/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.dns;

import org.xbill.DNS.Name;

import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class EnumUtil {
    public static final int MAX_DEPTH = 5;
    private static final char BACKSLASH = '\\';
    public static final String TRIVIAL_REGEXP = "^.*$";
    private static final Pattern POSIX_DIGIT_PATTERN = Pattern.compile("[:digit:]",
            Pattern.LITERAL);
    private static final String POSIX_DIGIT_REPLACEMENT = Matcher.quoteReplacement(
            "\\p{Digit}");

    public static String applyRegExp(String AUS, String regexpStr) {
        String[] regexp = EnumUtil.splitRegexp(EnumUtil.unescape(regexpStr));
        MatchResult matchResult = null;

        if (regexp[0].equals(EnumUtil.TRIVIAL_REGEXP)) {
            matchResult = new TrivialMatchResult(AUS);
        } else {
            Pattern pattern = Pattern.compile(EnumUtil.toAlmostPosix(regexp[0]));
            String substitution = regexp[1];
            Matcher matcher = pattern.matcher(AUS);

            if (matcher.find()) {
                matchResult = matcher;
            }
        }

        if (matchResult == null) {
            return null;
        }

        return EnumUtil.substitute(matchResult, regexp[1]);
    }

    public static String toAlmostPosix(String regexp) {
        return POSIX_DIGIT_PATTERN.matcher(regexp)
                                  .replaceAll(POSIX_DIGIT_REPLACEMENT);
    }

    public static boolean isEmpty(Name name) {
        return Name.root.equals(name) /*
         * this does not work... ||
         * Name.empty.equals(name);
         */;
    }

    public static String unescape(String str) {
        // TODO: Make a new state instead of this test
        if (str.indexOf(BACKSLASH) < 0) {
            return str;
        }

        State state = State.NORMAL;
        StringBuilder result = new StringBuilder(str.length());
        boolean escaped = false;

        for (int i = 0; i < str.length(); i++) {
            char ch = str.charAt(i);

            switch (state) {
            case NORMAL:

                if (ch == BACKSLASH) {
                    state = State.ESCAPE;
                } else {
                    result.append(ch);
                }

                break;

            case ESCAPE:

                if (Character.isDigit(ch)) {
                    state = State.DIGIT1;
                } else {
                    result.append(ch);
                    state = State.NORMAL;
                }

                break;

            case DIGIT1:

                if (Character.isDigit(ch)) {
                    state = State.DIGIT2;
                } else {
                    throw new IllegalArgumentException(str);
                }

                break;

            case DIGIT2:

                if (Character.isDigit(ch)) {
                    result.append((char) Integer.parseInt(str.substring(i - 2,
                                i + 1)));
                    state = State.NORMAL;
                } else {
                    throw new IllegalArgumentException(str);
                }

                break;
            }
        }

        return result.toString();
    }

    public static String toDomain(String telNumber, String enumTopDomain) {
        if (telNumber.charAt(0) != '+') {
            throw new IllegalArgumentException(
                "This is not a global phone number:" + telNumber);
        }

        StringBuilder domain = new StringBuilder((2 * telNumber.length()) + 10);

        for (int i = telNumber.length() - 1; i > 0; i--) {
            char ch = telNumber.charAt(i);

            if (Character.isDigit(ch)) {
                domain.append(ch);
                domain.append('.');
            }
        }

        domain.append(enumTopDomain);

        return domain.toString();
    }

    public static String substitute(MatchResult matcher, String substitution) {
        StringBuilder result = new StringBuilder();
        boolean backslash = false;

        for (int i = 0; i < substitution.length(); i++) {
            char ch = substitution.charAt(i);

            if (backslash) {
                if (Character.isDigit(ch)) {
                    result.append(matcher.group(Character.digit(ch, 10)));
                } else {
                    result.append(BACKSLASH).append(ch);
                }

                backslash = false;
            } else {
                if (ch == BACKSLASH) {
                    backslash = true;
                } else {
                    result.append(ch);
                }
            }
        }

        if (backslash) {
            result.append(BACKSLASH);
        }

        return result.toString();
    }

    public static String[] splitRegexp(String regexp) {
        String[] result = new String[3];
        int index = 0;
        boolean backslash = false;
        char delimiter = regexp.charAt(0);
        int start = 1;

        for (int i = 1; i < regexp.length(); i++) {
            char ch = regexp.charAt(i);

            if (backslash) {
                backslash = false;
            } else if (ch == BACKSLASH) {
                backslash = true;
            } else if (ch == delimiter) {
                if (index < 2) {
                    result[index++] = regexp.substring(start, i);
                    start = i + 1;
                } else {
                    throw new IllegalArgumentException(regexp);
                }
            }
        }

        if (index != 2) {
            throw new IllegalArgumentException(regexp);
        }

        result[2] = regexp.substring(start);

        return result;
    }

    public static String getAUSValue(String input) {
        StringBuilder sb = new StringBuilder();
        sb.append('+');

        for (int i = 0; i < input.length(); i++) {
            char ch = input.charAt(i);

            if (Character.isDigit(ch)) {
                sb.append(ch);
            }
        }

        return sb.toString();
    }
    enum State {NORMAL,
        ESCAPE,
        DIGIT1,
        DIGIT2;
    }
}
