/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.dns;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;


/**
 * @reviewed ejoelbi 2006-oct-23
 */
public class TargetFailedRegister extends TimerTask {
    private ConcurrentHashMap<String, Long> myHashMap = new ConcurrentHashMap<String, Long>();

    // private final int myKeepValue=15; // in secs
    private final int myTimerInterval = 60; // in

    // secs
    private Timer myTimer;
    private Logger logger = (Logger) Logger.getLogger("SipContainer");

    /**
     * This object is used to control access to myTimer.<br>
     * myTimer variable is accessed from more then one Thread (SIP worker Thread
     * and Timer internal Thread).<br>
     * So the access to the variable should be synchronized.
     */
    private final Object lock = new Object();

    public TargetFailedRegister() {
    }

    public void registerFailed(TargetTuple target, int expireValue) {
        // String failedEntryKey = protocolName+":"+ip+":"+port;
        String failedEntryKey = target.key();

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "registerFailed :" + failedEntryKey);
        }

        // save current timestamp
        // previous entry gets replaced if any
        myHashMap.put(failedEntryKey,
            new Long(new java.util.Date().getTime() + (expireValue * 1000)));

        // The only purpose of the timer is not to keep
        // old entrys in the hashmap (i.e tuples not beeing retried)
        synchronized (lock) {
            if (myTimer == null) {
                myTimer = new Timer();
                myTimer.scheduleAtFixedRate(this, 0, myTimerInterval * 1000);
            }
        }
    }

    // public boolean isFailed(String protocolName, String ip, int port)
    public boolean isFailed(TargetTuple target) {
        // String failedEntryKey = protocolName+":"+ip+":"+port;
        String failedEntryKey = target.key();

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "check if target failed:" + failedEntryKey);
        }

        boolean targetIsFailed = false;
        Object o = myHashMap.get(failedEntryKey);

        if (o != null) {
            long expireAt = ((Long) o).longValue();
            long now = new java.util.Date().getTime();

            if (expireAt > now) {
                targetIsFailed = true;
            } else {
                myHashMap.remove(o);
            }
        }

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "check if target failed:" + targetIsFailed);
        }

        return targetIsFailed;
    }

    public void debugRegister() {
        logger.log(Level.FINE, "____________________________");
        logger.log(Level.FINE, "Debug status in failed targets register\n");

        java.util.Enumeration<Long> e = myHashMap.elements();
        java.util.Enumeration<String> f = myHashMap.keys();

        while (e.hasMoreElements() && f.hasMoreElements()) {
            logger.log(Level.FINE,
                f.nextElement() + " timestamp:" + e.nextElement());
        }

        logger.log(Level.FINE, "____________________________");
    }

    public void clean() {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Cleaning the TargetFailedRegister");
        }

        java.util.Enumeration<Long> e = myHashMap.elements();
        java.util.Enumeration<String> f = myHashMap.keys();

        while (e.hasMoreElements()) {
            Long entry = e.nextElement();
            long now = new java.util.Date().getTime();

            if (entry.longValue() <= now) {
                String key = f.nextElement();

                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "removing " + key + " timestamp:" + entry.longValue());
                }

                myHashMap.remove(key);
            }
        }

        // if no failed entries left we may cancel the timer
        // is started again if new entries is added
        if (myHashMap.size() == 0) {
            synchronized (lock) {
                myTimer.cancel();
                myTimer = null;

                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "Cancel the TargetFailedRegister timer");
                }
            }
        }
    }

    public void run() {
        clean();
    }
}
