/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.dns;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;


public class TargetTuple implements Externalizable {
    /**
     *
     */
    private static final long serialVersionUID = 3257289110686283316L;
    private static final int UNDEFINED_PORTNO = -1;
    private String myIp;
    private int myPort;
    private SipTransports myProtocol;

    public TargetTuple() {
    }

    public TargetTuple(SipTransports transportProtocol,
        InetSocketAddress socketAddress) {
        myProtocol = transportProtocol;
        myIp = socketAddress.getAddress().getHostAddress();
        myPort = socketAddress.getPort();
    }

    public TargetTuple(SipTransports transportProtocol, String ip, int port) {
        myProtocol = transportProtocol;
        myIp = ip;
        myPort = port;
    }

    public void writeExternal(ObjectOutput output) throws IOException {
        output.writeUTF(myIp);
        output.writeInt(myPort);
        output.writeObject(myProtocol);
    }

    public void readExternal(ObjectInput input) throws IOException {
        myIp = input.readUTF();
        myPort = input.readInt();

        try {
            myProtocol = (SipTransports) input.readObject();
        } catch (ClassNotFoundException ignore) {
        }
    }

    public String getIP() {
        return myIp;
    }

    public InetSocketAddress getSocketAddress() {
        try {
            return new InetSocketAddress(InetAddress.getByName(myIp), myPort);
        } catch (UnknownHostException ignored) {
        }

        return null;
    }

    public InetAddress getAddress() {
        try {
            return InetAddress.getByName(myIp);
        } catch (UnknownHostException ignored) {
        }

        return null;
    }

    public int getPort() {
        return myPort;
    }

    public SipTransports getProtocol() {
        return myProtocol;
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object org) {
        if (!(org instanceof TargetTuple)) {
            return false;
        }

        TargetTuple tt = (TargetTuple) org;

        if (myPort != tt.myPort) {
            return false;
        }

        if ((myIp != null) && !myIp.equals(tt.myIp)) {
            return false;
        }

        if ((myProtocol != null) && !myProtocol.equals(tt.myProtocol)) {
            return false;
        }

        return true;
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return myIp.hashCode() ^ myPort;
    }

    public String key() {
        return myProtocol.name() + ":" + myIp + ":" + myPort;
    }

    public String toString() {
        String protocolString = myProtocol.name();

        return "________________________\n" + "\nip:" + myIp + "\nport:" +
        myPort + "\ntransportProtocol:" + protocolString +
        "\n______________________\n";
    }
}
