/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.sip.SipApplicationSessionImpl;

import java.io.Serializable;

import java.util.concurrent.ScheduledFuture;

import javax.servlet.sip.ServletTimer;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.TimerListener;


/**
 * Implementation of SSA interface <code>javax.servlet.sip.ServletTimer</code>.
 */
public abstract class ServletTimerImplBase implements ServletTimer, Runnable {
    /**
     * Whether this timer has been successfully cancelled. Used for debugging.
     */
    @SuppressWarnings("unused")
    private Boolean iscanceled = null;

    /**
     * Lock that prevents simultaneous execution. Note! Do NOT call methods in
     * the associated sip session within synchronization of this lock since there
     * may be a dead lock.
     */
    private final Object SIPP_APPLICATION_LOCK = new Object();

    // Empty constructor for serialization
    public ServletTimerImplBase() {
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Runnable#run()
     */
    public void run() {
        try {
            getPFieldListener().timeout(this);
        } finally {
            if (getPFieldIsRepeatingTimer()) {
                estimateNextExecution();
            } else {
                // this non-repeating timer is now "ready"
                // and should not be included in the list of active timers
                // The application may already have canceled() the timer though
                cancel(); // dont bother about return value....
            }
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#getApplicationSession()
     */
    public SipApplicationSession getApplicationSession() {
        return getPFieldAppSession();
    }

    /**
     * Setter to set what should be executed.
     *
     * @param f
     */
    public void setFuture(ScheduledFuture<?> f) {
        synchronized (SIPP_APPLICATION_LOCK) {
            setPFieldFuture(f);
        }
    }

    /**
     * Helper to calculate when next execution time is.
     *
     */
    private void estimateNextExecution() {
        synchronized (SIPP_APPLICATION_LOCK) {
            if (getPFieldFixedDelay()) {
                setPFieldScheduledExecutionTime(getPFieldPeriod() +
                    System.currentTimeMillis());
            } else {
                if (getPFieldFirstExecution() == 0) {
                    // save timestamp of first execution
                    setPFieldFirstExecution(getPFieldScheduledExecutionTime());
                }

                setPFieldScheduledExecutionTime(getPFieldFirstExecution() +
                    (incrementAndGetPFieldNumInvocations() * getPFieldPeriod()));
            }
        }
    }

    /**
     * Getter for delay property.
     *
     * @return
     */
    public long getDelay() {
        return getPFieldDelay();
    }

    /**
     * Getter for period property
     *
     * @return
     */
    public long getPeriod() {
        return getPFieldPeriod();
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#getInfo()
     */
    public Serializable getInfo() {
        return getPFieldInfo();
    }

    /**
     * Gets the unique id of this timer.
     *
     * @return The unique id of this timer
     */
    public String getId() {
        return getPFieldId();
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#scheduledExecutionTime()
     */
    public long scheduledExecutionTime() {
        synchronized (SIPP_APPLICATION_LOCK) {
            return getPFieldScheduledExecutionTime();
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#cancel()
     */
    public void cancel() {
        cancel(false);
    }

    /**
     *
     * Cancel this timer, possibly by also interrupting the thread (from the
     * thread pool) running the task. Note that interupting the thread may have
     * undesired consequences.
     *
     * @param mayInterruptIfRunning
     */
    public void cancel(boolean mayInterruptIfRunning) {
        SipApplicationSessionImpl appSessionToCancelThisTimersFrom = null;

        synchronized (SIPP_APPLICATION_LOCK) {
            ScheduledFuture<?> future = getPFieldFuture();

            if (future != null) {
                // need to force cancel to get rid of
                // the task which is currently scheduled
                boolean res = future.cancel(mayInterruptIfRunning);
                // used for debugging/optimizeIt purpose
                // kan be kept in production code since object should
                // be due for gc anyway....
                iscanceled = new Boolean(res);
                appSessionToCancelThisTimersFrom = getPFieldAppSession();
                setPFieldFuture(null);
            }
        }

        if (appSessionToCancelThisTimersFrom != null) {
            appSessionToCancelThisTimersFrom.cancelServletTimer(this);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String fullyQualifiedId = getFullyQualifiedId();

        if (fullyQualifiedId != null) {
            sb.append(fullyQualifiedId).append('\n');
        }

        sb.append("Info = ").append(getPFieldInfo()).append('\n');
        sb.append("Scheduled execution time = ")
          .append(getPFieldScheduledExecutionTime()).append('\n');
        sb.append("Time now = ").append(System.currentTimeMillis()).append('\n');
        sb.append("SipApplicationSession = ").append(getPFieldAppSession())
          .append('\n');
        sb.append("ScheduledFuture = ").append(getPFieldFuture()).append('\n');
        sb.append("Delay = ").append(getPFieldDelay()).append('\n');

        return sb.toString();
    }

    /**
     * Predicate for persistence property
     *
     * @return
     */
    public boolean isPersistent() {
        return getPFieldPersistent();
    }

    /**
     * Setter for persistence property
     *
     * @param p
     */
    public void setPersistent(boolean p) {
        setPFieldPersistent(p);
    }

    /* To be implemented */
    public String getID() {
        throw new RuntimeException("Not yet implemented");
    }

    /**
     * Predicate for isRepeating property.
     *
     * @return
     */
    boolean isRepeatingTimer() {
        return getPFieldIsRepeatingTimer();
    }

    /**
     * Predicate for fixed delay property
     *
     * @return
     */
    boolean isFixedDelay() {
        return getPFieldFixedDelay();
    }

    /**
     * Getter for listener object
     *
     * @return
     */
    TimerListener getListener() {
        return getPFieldListener();
    }

    // --- Persistable fields
    protected abstract String getPFieldId();

    protected abstract ScheduledFuture<?> getPFieldFuture();

    protected abstract void setPFieldFuture(ScheduledFuture<?> future);

    protected abstract Serializable getPFieldInfo();

    protected abstract SipApplicationSessionImpl getPFieldAppSession();

    protected abstract long incrementAndGetPFieldNumInvocations();

    protected abstract long getPFieldFirstExecution();

    protected abstract void setPFieldFirstExecution(long firstExecutionTime);

    protected abstract long getPFieldScheduledExecutionTime();

    protected abstract void setPFieldScheduledExecutionTime(
        long scheduledExecutionTime);

    protected abstract long getPFieldPeriod();

    protected abstract long getPFieldDelay();

    protected abstract boolean getPFieldFixedDelay();

    protected abstract boolean getPFieldPersistent();

    protected abstract void setPFieldPersistent(boolean p);

    protected abstract TimerListener getPFieldListener();

    protected abstract boolean getPFieldIsRepeatingTimer();

    protected abstract String getFullyQualifiedId();
}
