/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.container.SipContainerThreadPool;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipSessionManager;

import java.io.Serializable;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import javax.servlet.sip.ServletTimer;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.TimerListener;
import javax.servlet.sip.TimerService;


/**
 * @reviewed ejoelbi 2007-jan-17
 */
public class TimerServiceImpl implements TimerService, TimerListener {
    private static TimerServiceImpl _instance = new TimerServiceImpl();
    private static Logger logger = Logger.getLogger("SipContainer");
    private SipContainerThreadPool _schedPool = null;

    private TimerServiceImpl() {
    }

    public static TimerServiceImpl getInstance() {
        return _instance;
    }

    public ServletTimer createTimer(SipApplicationSession appSession,
        long delay, boolean isPersistent, Serializable info) {
        SipApplicationSessionImpl as = (SipApplicationSessionImpl) appSession;
        SipSessionManager ssm = as.getSipSessionManager();

        if ((null == ssm) || (as.isValid() == false)) {
            throw new IllegalStateException("Invalid SipApplicationSession");
        }

        TimerListener tl = as.getTimerListener();

        if (tl == null) {
            throw new IllegalStateException("No TimerListener is configured ");
        }

        ServletTimerImpl st = createTimer(ssm, as, tl, delay, isPersistent, info);

        return st;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.TimerService#createTimer(javax.servlet.sip.SipApplicationSession,
     *      long, long, boolean, boolean, java.io.Serializable)
     */
    public ServletTimer createTimer(SipApplicationSession appSession,
        long delay, long period, boolean fixedDelay, boolean isPersistent,
        Serializable info) {
        if (period < 1) {
            throw new IllegalArgumentException(
                "Period should be greater than 0");
        }

        SipApplicationSessionImpl as = (SipApplicationSessionImpl) appSession;
        SipSessionManager ssm = as.getSipSessionManager();

        if ((null == ssm) || (as.isValid() == false)) {
            throw new IllegalStateException("Invalid SipApplicationSession");
        }

        TimerListener tl = as.getTimerListener();

        if (tl == null) {
            throw new IllegalStateException("No TimerListener is configured ");
        }

        ServletTimerImpl st = createTimer(ssm, as, tl, delay, period,
                fixedDelay, isPersistent, info);

        return st;
    }

    public ServletTimerImpl createTimer(TimerListener listener, long delay,
        boolean isPersistent, Serializable info) {
        return createTimer(null, null, listener, delay, isPersistent, info);
    }

    public ServletTimerImpl createTimer(TimerListener listener, long delay,
        long period, boolean fixedDelay, boolean isPersistent, Serializable info) {
        return createTimer(null, null, listener, delay, period, fixedDelay,
            isPersistent, info);
    }

    private ServletTimerImpl createTimer(SipSessionManager ssm,
        SipApplicationSessionImpl sas, TimerListener listener, long delay,
        boolean isPersistent, Serializable info) {
        setupScheduller(); // TODO only for test

        ServletTimerImpl st = null;

        if (ssm != null) {
            st = ssm.createServletTimer(sas, info, delay, listener);
        } else {
            st = new ServletTimerImpl(info, delay, listener);
        }

        ScheduledFuture<?> f = _schedPool.schedule(st, delay,
                TimeUnit.MILLISECONDS);
        st.setFuture(f);

        if (isPersistent) {
            // TODO - implement persistance
        }

        return st;
    }

    private ServletTimerImpl createTimer(SipSessionManager ssm,
        SipApplicationSessionImpl sas, TimerListener listener, long delay,
        long period, boolean fixedDelay, boolean isPersistent, Serializable info) {
        setupScheduller(); // TODO only for test

        ServletTimerImpl st = null;

        if (ssm != null) {
            st = ssm.createServletTimer(sas, info, delay, fixedDelay, period,
                    listener);
        } else {
            st = new ServletTimerImpl(info, delay, fixedDelay, period, listener);
        }

        ScheduledFuture<?> f = null;

        if (fixedDelay) {
            f = _schedPool.scheduleWithFixedDelay(st, delay, period,
                    TimeUnit.MILLISECONDS);
        } else {
            f = _schedPool.scheduleAtFixedRate(st, delay, period,
                    TimeUnit.MILLISECONDS);
        }

        st.setFuture(f);

        if (isPersistent) {
            // TODO - implement persistance
        }

        return st;
    }

    public void deserializeTimer(ServletTimerImpl timer, long initDelay) {
        final TimerListener l = timer.getListener();
        final ServletTimerImpl st = timer;
        setupScheduller();

        ScheduledFuture<?> f = null;

        if (!st.isRepeatingTimer()) {
            f = _schedPool.schedule(st, initDelay, TimeUnit.MILLISECONDS);
        } else if (st.isFixedDelay()) {
            f = _schedPool.scheduleWithFixedDelay(st, initDelay,
                    st.getPeriod(), TimeUnit.MILLISECONDS);
        } else {
            f = _schedPool.scheduleAtFixedRate(st, initDelay, st.getPeriod(),
                    TimeUnit.MILLISECONDS);
        }

        st.setFuture(f);
    }

    private void setupScheduller() {
        // Could be configured in dispatcher.xml so that TM is unaware of NM
        if (_schedPool == null) {
            _schedPool = SipContainerThreadPool.getInstance();

            if (_schedPool == null) {
                throw new IllegalStateException(
                    "Failed to get access to the ThreadPool");
            }

            createTimer((TimerListener) this, 10000, 10000, false, false, null);
        }
    }

    /*
     * // Only for unit test public void
     * setScheduledExecutorService(ScheduledExecutorService schedPool) {
     * _schedPool = schedPool; }
     */
    public void timeout(ServletTimer timer) {
        // logger.log(Level.FINE, "Purging Thread pool");
        _schedPool.purge();
    }
}
