/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.transaction;

import com.ericsson.ssa.container.NetworkManager;
import com.ericsson.ssa.container.startup.PerformanceMBeanListener;
import com.ericsson.ssa.sip.Dispatcher;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.MultiLineHeader;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.ViaImpl;
import com.ericsson.ssa.sip.timer.ServletTimerImpl;
import static com.ericsson.ssa.sip.transaction.TransactionState.CALLING;
import static com.ericsson.ssa.sip.transaction.TransactionState.COMPLETED;
import static com.ericsson.ssa.sip.transaction.TransactionState.ESTABLISHED;
import static com.ericsson.ssa.sip.transaction.TransactionState.PROCEEDING;
import static com.ericsson.ssa.sip.transaction.TransactionState.TERMINATED;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerA;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerB;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerC;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerD;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerRemoveTransaction;

import java.util.ListIterator;
import java.util.logging.Level;

import javax.servlet.sip.ServletTimer;


/**
 * @author ekrigro TODO To change the template for this generated type comment
 *         go to Window - Preferences - Java - Code Style - Code Templates
 */
public class InviteClientTransaction extends ClientTransaction {
    private ServletTimerImpl _timerA; // T1 -> T1*2 ->
    private ServletTimerImpl _timerB; // 64*T1
    private ServletTimerImpl _timerC; // >3min, proxy
    private ServletTimerImpl _timerD; // 32s for UDP 0s
                                      // TCP
    private ServletTimerImpl _timerRemoveTransaction; // 64*T1
    private SipServletRequestImpl _ack = null;
    private boolean _has101To199Response = false;

    /**
     * @param state
     * @param req
     */
    public InviteClientTransaction(String id, SipServletRequestImpl req) {
        super(id, CALLING, req);
        _request.saveRetransmissionApplicationStack();

        // if proxying start Timer C...
        if (!_request.isContactIndicated()) {
            _timerC = _timerService.createTimer(this, 200000, false, TimerC);
        }

        // start timer B
        _timerB = _timerService.createTimer(this, 64 * T1, false, TimerB);

        if (!_reliableTransport) {
            // Start timer A
            _timerA = _timerService.createTimer(this, T1, false, TimerA);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Dispatcher#dispatch(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public synchronized boolean handle(SipServletResponseImpl resp) {
        // For every response got from the stack have to associate it to the req
        resp.setRequest(_request);

        int status = resp.getStatus() / 100;
        int xx = resp.getStatus() % 100;

        switch (_state) {
        case CALLING:

            if (status == 1) {
                if (_timerA != null) {
                    _timerA.cancel();
                    _timerA = null;
                }

                if (_timerB != null) {
                    _timerB.cancel();
                    _timerB = null;
                }

                resetTimerC(xx);
                _state = PROCEEDING;
            } else if (status == 2) {
                toEstablished(resp);
            } else {
                toCompleted(resp); // 3xx to 6xx
            }

            break;

        case PROCEEDING:

            if (status == 1) {
                resetTimerC(xx);

                break;
            } else if (status == 2) {
                toEstablished(resp);
            } else {
                toCompleted(resp); // 3xx to 6xx
            }

            break;

        case COMPLETED: // 3XX - 6XX for UDP resent ACK

            if ((status >= 2) && (status <= 6)) {
                // it's acceptable to block the transaction
                // during the ack transmission
                createHopAck(resp);
            }

            return true; // Stop chain

        case ESTABLISHED:

            // could get 2xx from same or different end-points,
            // multiple 2xx responses may arraive
            break;

        case TERMINATED:
            break; // Should not get any response from TU in this state

        default:
            _log.log(Level.FINE, "IllegalState in ICT = " + _state);
        }

        return false; // Continue up the chain
    }

    public void timeout(ServletTimer timer) {
        TransactionTimer tt = (TransactionTimer) timer.getInfo();
        Dispatcher d = null;
        SipServletResponseImpl resp = null;
        SipServletRequestImpl req = null;

        switch (tt) {
        case TimerA:

            synchronized (this) {
                if (_state == CALLING) {
                    // Have to dirty cast in order not to reimplement hole
                    // structure
                    long delay = ((ServletTimerImpl) timer).getDelay();
                    // calculate next timer*2
                    delay *= 2;
                    // schedule new timer
                    _timerA = _timerService.createTimer(this, delay, false,
                            TimerA);
                    // resend the request
                    _request.restoreRetransmissionApplicationStack();
                    req = (SipServletRequestImpl) _request.clone();
                }
            }

            if (req != null) {
                // dispatch after synch block...
                d = req.popDispatcher();

                if (d != null) {
                    d.dispatch(req);
                }

                if (PerformanceMBeanListener.isEnabled()) {
                    updateLastAccessTimestamp();
                }
            }

            break;

        case TimerB:

            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE, "Timer B fired - terminating()");
            }

            synchronized (this) {
                resp = sendRequestTimeout();
                terminate();
            }

            // dispatch after synch block...
            d = resp.popDispatcher();

            if (d != null) {
                d.dispatch(resp);
            }

            break; // Time to do some GC

        case TimerC:

            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE, "Timer C fired - terminating()");
            }

            synchronized (this) {
                if ((_state == CALLING) || (_state == PROCEEDING)) {
                    if (_has101To199Response) {
                        req = sendCancel();
                    } else {
                        resp = sendRequestTimeout();
                        terminate();
                    }
                }
            }

            // dispatch after synch block...
            if (_has101To199Response) {
                // lets CANCEL this branch...
                if (req != null) {
                    d = req.popDispatcher();

                    if (d != null) {
                        d.dispatch(req);
                    }
                }
            } else {
                // lets send 408 Request Timeout...
                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "Timer C fired - send 408 Request Timeout");
                }

                d = resp.popDispatcher();

                if (d != null) {
                    d.dispatch(resp);
                }
            }

            break; // Time to do some GC

        case TimerD:
            terminate();

            break;

        case TimerRemoveTransaction:

            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE,
                    "Timer RemoveTransaction fired - terminating()");
            }

            terminate();

            break;

        default:
            _log.log(Level.FINE, "IllegalTimer in ICT = " + tt);
        }
    }

    /**
     * If provisonal response (except 100) and proxying, Timer C will be reset
     *
     * @param xx
     *        the xx of the 1xx response status code
     */
    private void resetTimerC(int xx) {
        if (!_request.isContactIndicated() && (xx != 0)) {
            // reset Timer C
            if (_timerC != null) {
                _timerC.cancel();
            }

            _timerC = _timerService.createTimer(this, 200000, false, TimerC);
            _has101To199Response = true;
        }
    }

    private SipServletRequestImpl sendCancel() {
        Header via = _request.getRawHeader(Header.VIA);
        ListIterator<String> li = via.getValues();
        String topVia = li.next();

        if (topVia != null) {
            SipServletRequestImpl cancel = _request.createCancelImpl();

            // set topvia of original request to CANCEL
            Header viaOfCancel = new MultiLineHeader(Header.VIA, true);
            ViaImpl v = new ViaImpl(topVia);
            viaOfCancel.setValue(v.toString(), true);
            cancel.setHeader(viaOfCancel);

            return cancel;
        } else {
            _log.log(Level.SEVERE,
                "Can not send CANCEL because INVITE does not have via: " +
                _request.toString());

            return null;
        }
    }

    private SipServletResponseImpl sendRequestTimeout() {
        SipServletResponseImpl resp = _request.createTerminatingResponse(408);

        // POP via
        Header via = resp.getRawHeader(Header.VIA);
        via.setReadOnly(false);

        ListIterator<String> li = via.getValues();
        li.next();
        li.remove();
        via.setReadOnly(true);

        // set session...
        SipServletRequestImpl req = _request.getTransactionRequest();

        //    TR HH52078
        if ((req != null) && (resp != null)) {
            resp.setRequest(req);
            resp.setSession(req.getSessionImpl());
        }

        return resp;
    }

    /* TODO check synchronization */
    protected synchronized void terminate() {
        super.terminate();

        if (_timerA != null) {
            _timerA.cancel();
            _timerA = null;
        }

        if (_timerB != null) {
            _timerB.cancel();
            _timerB = null;
        }

        if (_timerC != null) {
            _timerC.cancel();
            _timerC = null;
        }

        if (_timerD != null) {
            _timerD.cancel();
            _timerD = null;
        }

        if (_timerRemoveTransaction != null) {
            _timerRemoveTransaction.cancel();
            _timerRemoveTransaction = null;
        }

        _state = TERMINATED;
    }

    private void toCompleted(SipServletResponseImpl resp) {
        // it's acceptable to block the transaction
        // during the ack transmission
        createHopAck(resp);

        if (_timerB != null) {
            _timerB.cancel();
            _timerB = null;
        }

        if (_timerC != null) {
            _timerC.cancel();
            _timerC = null;
        }

        if (!_reliableTransport) { // Start timer D

            if (PerformanceMBeanListener.isEnabled()) {
                updateLastAccessTimestamp();
            }

            if (_timerA != null) {
                _timerA.cancel();
                _timerA = null;
            }

            _state = COMPLETED;
            _timerD = _timerService.createTimer(this, 64 * T1, false, TimerD);
        } else {
            _state = TERMINATED; // TimerD = 0
            super.terminate();
        }
    }

    private void toEstablished(SipServletResponseImpl resp) {
        if (_timerB != null) {
            _timerB.cancel();
            _timerB = null;
        }

        if (!_reliableTransport) {
            if (PerformanceMBeanListener.isEnabled()) {
                updateLastAccessTimestamp();
            }

            if (_timerA != null) {
                _timerA.cancel();
                _timerA = null;
            }
        }

        // Start timer RemoveTransaction
        _timerRemoveTransaction = _timerService.createTimer(this, 64 * T1,
                false, TimerRemoveTransaction);
        _state = ESTABLISHED;
    }

    private void createHopAck(SipServletResponseImpl response) {
        if (_ack == null) {
            _ack = response.createHopAckImpl(_request);
        }

        // send it directly out on the network...
        NetworkManager.getInstance().dispatch(_ack);
    }
}
