/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.transaction;

import com.ericsson.ssa.container.startup.PerformanceMBeanListener;
import com.ericsson.ssa.sip.Dispatcher;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.timer.ServletTimerImpl;
import static com.ericsson.ssa.sip.transaction.TransactionState.COMPLETED;
import static com.ericsson.ssa.sip.transaction.TransactionState.PROCEEDING;
import static com.ericsson.ssa.sip.transaction.TransactionState.TERMINATED;
import static com.ericsson.ssa.sip.transaction.TransactionState.TRYING;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerE;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerF;
import static com.ericsson.ssa.sip.transaction.TransactionTimer.TimerK;

import java.util.ListIterator;
import java.util.logging.Level;

import javax.servlet.sip.ServletTimer;


/**
 * @author ekrigro TODO To change the template for this generated type comment
 *         go to Window - Preferences - Java - Code Style - Code Templates
 */
public class NonInviteClientTransaction extends ClientTransaction {
    private ServletTimerImpl _timerE; // T1 -> T1*2 ->
    private ServletTimerImpl _timerF; // 64*T1
    private ServletTimerImpl _timerK; // T4 for UDP 0s TCP

    /**
     * @param state
     * @param req
     */
    public NonInviteClientTransaction(String id, SipServletRequestImpl req) {
        super(id, TRYING, req);
        _request.saveRetransmissionApplicationStack();
        // start timer F
        _timerF = _timerService.createTimer(this, 64 * T1, false, TimerF);

        if (!_reliableTransport) {
            // Start timer E
            _timerE = _timerService.createTimer(this, T1, false, TimerE);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Dispatcher#dispatch(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public synchronized boolean handle(SipServletResponseImpl resp) {
        // For every response got from the stack have to associate it to the req
        resp.setRequest(_request);

        int status = resp.getStatus() / 100;

        switch (_state) {
        case TRYING:

            if (status == 1) {
                if (_timerE != null) {
                    _timerE.cancel();
                    _timerE = _timerService.createTimer(this, T2, false, TimerE);
                }

                _state = PROCEEDING;
            } else {
                toCompleted(resp); // 2xx to 6xx
            }

            break;

        case PROCEEDING:

            if (status == 1) {
                break;
            } else {
                toCompleted(resp); // 2xx to 6xx
            }

            break;

        case COMPLETED:
            break; // Should not get any response from TU in this state

        case TERMINATED:
            break; // Should not get any response from TU in this state

        default:
            _log.log(Level.FINE, "IllegalState in ICT = " + _state);
        }

        // response to CANCEL should not continue up the chain
        if (resp.getMethod().equals("CANCEL")) {
            return true;
        }

        return false; // Continue up the chain
    }

    public void timeout(ServletTimer timer) {
        TransactionTimer tt = (TransactionTimer) timer.getInfo();
        SipServletRequestImpl req = null;
        Dispatcher d = null;

        switch (tt) {
        case TimerE:

            synchronized (this) {
                if ((_state == TRYING) || (_state == PROCEEDING)) {
                    long delay = T2; // default to PROCEEDING delay

                    if (_state == TRYING) {
                        // Have to dirty cast in order not to reimplement hole
                        // structure
                        delay = ((ServletTimerImpl) timer).getDelay();
                        // calculate next timer*2 but less then T2 (4sec)
                        delay = ((delay * 2) <= T2) ? (delay * 2) : T2;
                    }

                    if (_timerE != null) {
                        _timerE.cancel();
                    }

                    // schedule new timer
                    _timerE = _timerService.createTimer(this, delay, false,
                            TimerE);

                    // resend the request
                    _request.restoreRetransmissionApplicationStack();
                    req = (SipServletRequestImpl) _request.clone();
                }
            }

            // dispatch after synch block...
            if (req != null) {
                d = req.popDispatcher();

                if (d != null) {
                    d.dispatch(req);
                }

                if (PerformanceMBeanListener.isEnabled()) {
                    updateLastAccessTimestamp();
                }
            }

            break;

        case TimerF:

            SipServletResponseImpl resp = null;

            synchronized (this) {
                if ((_state == TRYING) || (_state == PROCEEDING)) {
                    resp = _request.createTerminatingResponse(408);
                    req = _request.getTransactionRequest();

                    // TR HH52078
                    if ((req != null) && (resp != null)) {
                        resp.setRequest(req);
                        resp.setSession(req.getSessionImpl());
                    }
                }

                // lets remove this transaction before sending the 408 response
                terminate();
            }

            // dispatch after synch block...
            if (resp != null) {
                // POP via
                Header via = resp.getRawHeader(Header.VIA);
                via.setReadOnly(false);

                ListIterator<String> li = via.getValues();
                li.next();
                li.remove();
                via.setReadOnly(true);

                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "Timer F fired - send 408 Request Timeout " + _state +
                        " , response = " + resp.toDebugString());
                }

                d = resp.popDispatcher();

                if (d != null) {
                    d.dispatch(resp);
                }
            }

            break; // Time to do some GC

        case TimerK:
            terminate();

            break;

        default:
            _log.log(Level.FINE, "IllegalTimer in ICT = " + tt);
        }
    }

    /* TODO check synchronization */
    protected synchronized void terminate() {
        super.terminate();

        if (_timerE != null) {
            _timerE.cancel();
            _timerE = null;
        }

        if (_timerF != null) {
            _timerF.cancel();
            _timerF = null;
        }

        if (_timerK != null) {
            _timerK.cancel();
            _timerK = null;
        }

        _state = TERMINATED;
    }

    private void toCompleted(SipServletResponseImpl resp) {
        if (_timerF != null) {
            _timerF.cancel();
            _timerF = null;
        }

        if (!_reliableTransport) { // Start timer K

            if (PerformanceMBeanListener.isEnabled()) {
                updateLastAccessTimestamp();
            }

            if (_timerE != null) {
                _timerE.cancel();
                _timerE = null;
            }

            _state = COMPLETED;
            _timerK = _timerService.createTimer(this, T4, false, TimerK);
        } else {
            _state = TERMINATED; // TimerK = 0
            super.terminate();
        }
    }
}
