/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.admin.clbadmin.reader.impl;

import com.sun.enterprise.admin.util.JMXConnectorConfig;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.ClusterRef;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.HttpListener;
import com.sun.enterprise.config.serverbeans.HttpService;
import com.sun.enterprise.config.serverbeans.SipListener;
import com.sun.enterprise.config.serverbeans.SipService;
import com.sun.enterprise.config.serverbeans.PropertyResolver;
import com.sun.enterprise.config.serverbeans.Server;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.ServerRef;
import com.sun.enterprise.config.serverbeans.SystemProperty;
import com.sun.enterprise.util.RelativePathResolver;
import com.sun.enterprise.util.SystemPropertyConstants;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.util.i18n.StringManagerBase;

import com.sun.logging.ee.EELogDomains;

import org.jvnet.glassfish.comms.admin.clbadmin.reader.api.InstanceReader;
import org.jvnet.glassfish.comms.admin.clbadmin.reader.api.LbReaderException;
import org.jvnet.glassfish.comms.admin.clbadmin.transform.InstanceVisitor;
import org.jvnet.glassfish.comms.admin.clbadmin.transform.Visitor;

import java.net.InetAddress;
import java.net.UnknownHostException;

import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Provides instance information relavant to Load balancer tier.
 *
 * @author Vijaya Gadhamsetty
 */
public class InstanceReaderImpl implements InstanceReader {
    private static final StringManager _localStrMgr = StringManager.getManager(InstanceReaderImpl.class);

    // --- PRIVATE VARS -------
    ConfigContext _configCtx = null;
    ServerRef _serverRef = null;
    final private String HTTP_PROTO = "http://";
    final private String HTTPS_PROTO = "https://";

    final private String SIP_PROTO = "sip://";
    final private String SIPS_PROTO = "sips://";
    
    final private String SECURE_TRANSPORT = "tls";

    /**
     * Constructor
     */
    public InstanceReaderImpl(ConfigContext ctx, ServerRef sRef) {
        if ((ctx == null) || (sRef == null)) {
            String msg = _localStrMgr.getString("ConfigBeanAndNameNull");
            throw new IllegalArgumentException(msg);
        }

        _configCtx = ctx;
        _serverRef = sRef;
    }

    /**
     * Return server instance's name.
     *
     * @return String           instance' name
     */
    public String getName() throws LbReaderException {
        return _serverRef.getRef();
    }

    /**
     * Returns if the server is enabled in the load balancer or not.
     *
     * @return boolean          true if enabled in LB; false if disabled
     */
    public boolean getLbEnabled() throws LbReaderException {
        return _serverRef.isLbEnabled();
    }

    /**
     * This is used in quicescing. Timeouts after this interval and disables the
     * instance in the load balancer.
     *
     * @return String           Disable time out in minutes
     */
    public String getDisableTimeoutInMinutes() throws LbReaderException {
        return _serverRef.getDisableTimeoutInMinutes();
    }

    /**
     * Enlists both http and https listeners of this server instance
     * It will be form "http:<hostname>:<port> https:<hostname>:<port>"
     *
     * @return String   Listener(s) info.
     */
    public String getListeners() throws LbReaderException {
        String listenerStr = "";

        String sName = _serverRef.getRef();
        String hostName = getHostNameForServerInstance(_configCtx, sName);
        Config c = null;

        try {
            c = ServerHelper.getConfigForServer(_configCtx, sName);
        } catch (ConfigException ce) {
            String msg = _localStrMgr.getString("ConfigNotFound", sName);
            throw new LbReaderException(msg, ce);
        }

        // http-listeners
        HttpService httpSvc = c.getHttpService();
        HttpListener[] lstnrs = httpSvc.getHttpListener();

        for (int i = 0; i < lstnrs.length; i++) {
            if (lstnrs[i].isSecurityEnabled())
                listenerStr += HTTPS_PROTO;
            else
                listenerStr += HTTP_PROTO;

            // XXX actual hostname
            listenerStr += (hostName + ":");
            // resolve the port name
            listenerStr += resolvePort(lstnrs[i].getPort());
            // space between listener names
            listenerStr += " "; 
        }

        // sip-listeners
        SipService sipSvc = c.getSipService();
        SipListener[] sipListeners = sipSvc.getSipListener();

        for (int i = 0; i < sipListeners.length; i++) {
            if (sipListeners[i].getTransport().equals(SECURE_TRANSPORT))
                listenerStr += SIPS_PROTO;
            else
                listenerStr += SIP_PROTO;

            listenerStr += (hostName + ":");
            listenerStr += resolvePort(sipListeners[i].getPort());
            listenerStr += " ";
        }

        return listenerStr.trim();
    }

    private String resolvePort(String port) {

        // If it is system variable, resolve it
        if ((port != null) && (port.length() > 1) &&
                (port.charAt(0) == '$') && (port.charAt(1) == '{') &&
                (port.charAt(port.length() - 1) == '}')) {
            String portVar = port.substring(2, port.length() - 1);
            String sVar = null;
            PropertyResolver propResolver = null;

            try {
                propResolver = new PropertyResolver(_configCtx, _serverRef.getRef());
            } catch (ConfigException ce) {
                // ignore this exception
            }

            if (propResolver != null) {
                sVar = propResolver.getPropertyValue(portVar);
            }

            if (sVar != null) {
                return sVar;
            } else {
                return port;
            }
        } else {
            return port;
        }
    }

    // --- VISITOR IMPLEMENTATION ---
    public void accept(Visitor v) {
        InstanceVisitor iv = (InstanceVisitor) v;
        iv.visit(this);
    }

    private String getHostNameForServerInstance(ConfigContext ctx,
        String serverName) throws LbReaderException {
        try {
            JMXConnectorConfig info = ServerHelper.getJMXConnectorInfo(ctx,
                    serverName);
            String host = info.getHost();

            return host;
        } catch (Exception e) {
            String msg = _localStrMgr.getString("GetHostNameFailed");
            throw new LbReaderException(msg, e);
        }
    }

}
