/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core.sip;

import com.ericsson.ssa.container.callflow.CallflowResolver;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;

import org.jvnet.glassfish.comms.clb.core.CLBConstants;
import org.jvnet.glassfish.comms.clb.core.Controller;
import org.jvnet.glassfish.comms.clb.core.ControllerInitializer;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;

import java.util.logging.Level;

import javax.servlet.sip.SipServletResponse;


/**
 * This is the SipLoadBalancingManager that together with
 * {@link SipLoadBalancerManagerBackEnd} makes up the layers for the SIP part of the Converged Load
 * Balancer.
 * <p>
 * This part of the CLB is responsible for handling incoming requests/responses.
 * It is not present in pure back-end servers.
 *
 * It is a singleton and will be inserted into the SIP chain by the
 * {@link org.jvnet.glassfish.comms.startup.SipServiceListener} and
 * {@link com.ericsson.ssa.config.LayerHandler} at start-up.
 *
 * It is this class that is responsible for reading the configuration and
 * creating all the runtime objects for the front-end parts.
 * <p>
 * TODO (qbinjoe) Ensure that we share runtime objects (ServerCluster, ServerInstance,
 * etc.) between front-end and back-end layer.
 */
public class SipLoadBalancerManager extends ControllerInitializer
    implements Layer {
    private static LogUtil logger = new LogUtil(LogUtil.CLB_LOG_DOMAIN);
    private static final String CONTROLLER_IS_NULL = "org.jvnet.glassfish.comms.clb.core.sip.SipLoadBalancerManager.controller_null";
    private static final String IO_EXCEPTION = "org.jvnet.glassfish.comms.clb.core.sip.io_exception";
    private static SipLoadBalancerManager instance; // The singleton
                                                    // instance
    private Layer nextLayer; // The next layer above
    private Reporter reporter;

    /**
     * Create the instance
     */
    private SipLoadBalancerManager() {
    }

    /**
     * Gets the singleton instance. Used by the
     * {@link com.ericsson.ssa.config.LayerHandler} (via reflection) when
     * setting up the SIP and HTTP processing chains at start-up.
     *
     * @return the singleton instance
     */
    public static SipLoadBalancerManager getInstance() {
        if (instance == null) {
            logger.info("SipLoadBalancerManager created");
            instance = new SipLoadBalancerManager();
        }

        return instance;
    }

    /**
     * Called by {@link org.jvnet.glassfish.comms.startup.SipServiceListener} at
     * Lifecycle.AFTER_START_EVENT.
     */
    public void start() {
        logger.info("Starting SipLoadBalancerManager...");

        // Configure the single SIP request group
        try {
            getRequestGroup();
            logger.info("SipLoadBalancerManager started successfully.");
        } catch (IllegalStateException e) {
            logger.warning(e.getMessage());
        }
    }

    private SipRequestGroup getRequestGroup() throws IllegalStateException {
        // The controller might not have been initialized yet
        if (controller == null) {
            throw new IllegalStateException(CONTROLLER_IS_NULL);
        }

        SipRequestGroup sipRequestGroup = (SipRequestGroup) controller.getRequestGroup(null);
        sipRequestGroup.setLbmLayer(this);
        sipRequestGroup.setNextLayer(nextLayer);

        return sipRequestGroup;
    }

    public Controller createController() {
        Controller newController = super.createController();
        newController.setLBType(CLBConstants.SIP_CLB);

        return newController;
    }

    // --------------- Layer interface ----------------
    /**
     * Handle an incoming SIP request.
     *
     * @param req the received request
     * @see com.ericsson.ssa.sip.Layer#next(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void next(SipServletRequestImpl req) {
        try {
            SipRequestGroup sipRequestGroup = getRequestGroup();

            if (sipRequestGroup != null) {
                sipRequestGroup.handleIncomingRequest(req);
            } else {
                // No SIP request group just fall through this layer
                if (logger.isLoggable(Level.FINE)) {
                    logger.logMsg(Level.FINE,
                        "No request group, fall through; request: " + req);
                }

                LayerHelper.next(req, this, nextLayer);
            }
        } catch (IllegalStateException e) {
            sendErrorResponse(req, e);
        }
    }

    private void sendErrorResponse(SipServletRequestImpl req,
        IllegalStateException e) {
        logger.warning(e.getMessage());

        SipServletResponse resp = req.createResponse(503,
                "Could not server request due to that server is initializing");

        if (logger.isLoggable(Level.FINE)) {
            logger.logMsg(Level.FINE,
                "Could not server request:\n" + req + "Send error response:\n" +
                resp);
        }

        try {
            resp.send();
        } catch (IOException e1) {
            logger.warning(e1, IO_EXCEPTION);
        }
    }

    /**
     * Handle an incoming SIP response.
     *
     * @param resp the received request
     * @see com.ericsson.ssa.sip.Layer#next(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public void next(SipServletResponseImpl resp) {
        SipRequestGroup sipRequestGroup = getRequestGroup();

        if (sipRequestGroup != null) {
            sipRequestGroup.handleIncomingResponse(resp);
        } else {
            // No SIP request group just fall through this layer
            if (logger.isLoggable(Level.FINE)) {
                logger.logMsg(Level.FINE,
                    "No request group, fall through; request: " + resp);
            }

            LayerHelper.next(resp, this, nextLayer);
        }
    }

    /**
     * No handling of outgoing requests (handled by
     * {@link SipLoadBalancerManagerBackEnd}).
     *
     * @param req the request to be sent
     * @see com.ericsson.ssa.sip.Dispatcher#dispatch(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void dispatch(SipServletRequestImpl req) {
        if (logger.isLoggable(Level.FINE)) {
            logger.logMsg(Level.FINE, "Dispatch request: " + req);
        }

        req.popDispatcher().dispatch(req);
    }

    /**
     * No handling of outgoing responses (handled by
     * {@link SipLoadBalancerManagerBackEnd}).
     *
     * @param resp the response to be sent
     * @see com.ericsson.ssa.sip.Dispatcher#dispatch(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public void dispatch(SipServletResponseImpl resp) {
        if (logger.isLoggable(Level.FINE)) {
            logger.logMsg(Level.FINE, "Dispatch response: " + resp);
        }

        resp.popDispatcher().dispatch(resp);
    }

    /**
     * Register the next layer (used by
     * {@link com.ericsson.ssa.config.LayerHandler})
     *
     * @see com.ericsson.ssa.sip.Layer#registerNext(com.ericsson.ssa.sip.Layer)
     */
    public void registerNext(Layer layer) {
        nextLayer = layer;
    }
    
    public void setReporters(String reporters){
        reporter = CallflowResolver.getInstance().getReporter(reporters);
    }
    
    public Reporter getReporter(){
        return reporter;
    }
      
}
