/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core.sip;

import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import com.sun.grizzly.tcp.Response;

import org.jvnet.glassfish.comms.clb.core.CLBRuntimeException;
import org.jvnet.glassfish.comms.clb.core.ConsistentHashRequest;
import org.jvnet.glassfish.comms.clb.core.Controller;
import org.jvnet.glassfish.comms.clb.core.RequestGroup;
import org.jvnet.glassfish.comms.clb.core.Router;
import org.jvnet.glassfish.comms.clb.core.RouterFactory;
import org.jvnet.glassfish.comms.clb.core.ServerCluster;
import org.jvnet.glassfish.comms.clb.core.util.LoadbalancerUtil;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.List;


/**
 * This is the request group implementation for SIP.
 */
public class SipRequestGroup implements RequestGroup {
    private static LogUtil logger = new LogUtil(LogUtil.CLB_LOG_DOMAIN);
    private SipLoadBalancerIncomingHandler sipLoadBalancer;
    private Layer nextLayer;
    private Layer lbmLayer;
    private List<ServerCluster> clusters = new ArrayList<ServerCluster>();
    private Router stickyRouter;
    private Controller controller;

    /**
     * Create the request group.
     */
    public SipRequestGroup(Controller controller) {
        this.controller = controller;
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#addCluster(org.jvnet.glassfish.comms.clb.core.ServerCluster)
     */
    public void addCluster(ServerCluster cluster) {
        clusters.add(cluster);
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#deleteCluster(org.jvnet.glassfish.comms.clb.core.ServerCluster)
     */
    public void deleteCluster(ServerCluster cluster) {
        clusters.remove(cluster);
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#handleIncomingRequest(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void handleIncomingRequest(SipServletRequestImpl request) {
        Connection connection;
        controller.incrementRequestCount();

        try {
            ConsistentHashRequest chr = new ConsistentHashRequest(request, null);
            connection = sipLoadBalancer.handleIncomingRequest(chr, stickyRouter);

            if (connection != null) {
                // Alter connection and dispatch
                SipLoadBalancerManagerBackEnd.resetNetworkManager(request);
                request.setRemote(new TargetTuple(connection.getTransport(),
                        connection.getRemoteEndpoint().getHostName(),
                        connection.getRemoteEndpoint().getPort()));
                request.popDispatcher().dispatch(request);
            } else {
                // Continue on this server
                request.pushTransactionDispatcher(lbmLayer);
                request.pushApplicationDispatcher(lbmLayer);
                LayerHelper.next(request, lbmLayer, nextLayer);
            }
        } catch (SipRoutingException e) {
            logger.warning("Failed to route incoming request: " + e);

            SipServletResponseImpl errorResponse = request.createTerminatingResponse(503);

            if (errorResponse != null) {
                errorResponse.popDispatcher().dispatch(errorResponse);
            }
        } finally {
            controller.decrementRequestCount();
        }
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#handleIncomingResponse(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public void handleIncomingResponse(SipServletResponseImpl response) {
        Connection connection;
        controller.incrementRequestCount();

        try {
            connection = sipLoadBalancer.handleIncomingResponse(response);

            if (connection != null) {
                // Alter connection and dispatch
                SipLoadBalancerManagerBackEnd.resetNetworkManager(response);
                response.setRemote(new TargetTuple(connection.getTransport(),
                        connection.getRemoteEndpoint().getHostName(),
                        connection.getRemoteEndpoint().getPort()));
                response.popDispatcher().dispatch(response);
            } else {
                // Continue on this server
                LayerHelper.next(response, lbmLayer, nextLayer);
            }
        } catch (SipRoutingException e) {
            logger.warning("Failed to route incoming response: " + e);

            // Drop the response
        } finally {
            controller.decrementRequestCount();
        }
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#initializeRouter(java.lang.String)
     */
    public void initializeRouter(String lbpolicy) throws CLBRuntimeException {
        setupServerSelector();
        sipLoadBalancer = new SipLoadBalancerIncomingHandler(LoadbalancerUtil.getLocalSocket(
                    SipTransports.TCP_PROT));
    }

    private void setupServerSelector() throws CLBRuntimeException {
        stickyRouter = RouterFactory.createRouter(RouterFactory.LBPolicy_STICKY_CONSISTENT_HASH,
                clusters, false);
        stickyRouter.setController(controller);
        stickyRouter.initialize();
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#reinitializeRouter()
     */
    public void reinitializeRouter() throws CLBRuntimeException {
        setupServerSelector();
    }

    public void processDCRFileUpdateEvent() throws CLBRuntimeException {
        setupServerSelector();
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#serviceRequest(org.jvnet.glassfish.comms.clb.core.Request,
     *      org.jvnet.glassfish.comms.clb.core.Response)
     */
    public void serviceRequest(HttpRequest req, Response res) {
        throw new UnsupportedOperationException("Not relevant for SIP");
    }

    /**
     * Sets the next layer.
     *
     * @param nextLayer the nextLayer to set
     */
    public void setNextLayer(Layer nextLayer) {
        this.nextLayer = nextLayer;
    }

    /**
     * Sets the LBM layer that this request group belongs to.
     *
     * @param lbmLayer the lbmLayer to set
     */
    public void setLbmLayer(Layer lbmLayer) {
        this.lbmLayer = lbmLayer;
    }

    public void cleanup() {
        clusters = null;
        controller = null;
    }
}
