/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.proxy.http;

import com.sun.enterprise.web.connector.grizzly.ssl.SSLOutputWriter;
import com.sun.grizzly.ssl.SSLOutputBuffer;
import com.sun.enterprise.web.portunif.*;
import com.sun.enterprise.web.portunif.util.ProtocolInfo;

import com.sun.grizzly.http.SocketChannelOutputBuffer;
import com.sun.grizzly.tcp.Request;
import com.sun.grizzly.tcp.Response;
import com.sun.grizzly.util.http.MimeHeaders;

import org.jvnet.glassfish.comms.clb.proxy.HttpProxy;
import org.jvnet.glassfish.comms.clb.proxy.ProxyRequestHandler;
import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;
import org.jvnet.glassfish.comms.clb.proxy.config.LoadBalancerProxyConstants;
import org.jvnet.glassfish.comms.clb.proxy.http.util.ObjectManager;
import org.jvnet.glassfish.comms.clb.proxy.portunif.SailfinClbProxyProtocolInfo;

import java.io.IOException;
import java.nio.ByteBuffer;

import java.nio.channels.SelectionKey;
import java.util.Enumeration;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Redirect the request to the proper protocol, which can be http or https.
 *
 * @author
 */
public class LoadBalancerProxyHandler implements ProtocolHandler {

    /**
     * The protocols supported by this handler.
     */
    protected String[] protocols = {"lb/http", "lb/https"};
    private HttpProxy proxy;
    private Logger _logger = null;
    private ObjectManager objManager;
    private static ByteBuffer sslErrorBuffer = ByteBuffer.allocate(
            LoadBalancerProxyConstants.SSL_HEADER_SIZE);

    public LoadBalancerProxyHandler() {
        _logger = ProxyConfig.getInstance().getLogger();
        proxy = HttpProxy.getInstance();
        objManager = ObjectManager.getInstance();
    }

    /**
     * Redirect the request to the protocol defined in the
     * <code>protocolInfo</code>. Protocols supported are http and https.
     *
     * @param protocolInfo The protocol that needs to be redirected.
     */
    public void handle(ProtocolInfo protocolInfo) throws IOException {
        try {
            _handle(protocolInfo);
        } catch (Throwable t) {
            t.printStackTrace();
            throw new IOException(t.getMessage());
        }
    }

    private void _handle(ProtocolInfo protocolInfo) throws IOException {
        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE, "clb.proxy.http.protocol_handler_invoked", protocolInfo.key);
        }
        Object ptask = ((SailfinClbProxyProtocolInfo) protocolInfo).object;

        ProxyRequestHandler task = null;
        Endpoint endpoint = null;
        Request request = null;
        Response response = null;
        proxy.setSelectorThread(((SailfinClbProxyProtocolInfo) protocolInfo).selectorThread);

        if (ptask != null) {
            task = (ProxyRequestHandler) ptask;
            response = task.getResponse();
            request = task.getRequest();

            if (response.getStatus() != 200) {
                _logger.log(Level.SEVERE, "clb.proxy.http.handler_error_response");
                response.setHeader(LoadBalancerProxyConstants.HTTP_CONNECTION_HEADER,
                        LoadBalancerProxyConstants.HTTP_CONNECTION_CLOSE_VALUE);
                if (task.isSecure()) {
                    sendSecureResponse(request, response);
                } else {
                    sendResponse(request, response);
                }
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.log(Level.FINE, "clb.proxy.http.handler_sent_error_response");
                }
                task.recycle();
                objManager.offerTask(task, protocolInfo.isSecure);
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.log(Level.FINE, "clb.proxy.http.handler_released_resources");
                }
                protocolInfo.keepAlive = false;

                return;
            }
            endpoint = task.getEndpoint();
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE,
                        "clb.proxy.handler.endpoint", endpoint);
            }
        } else {
            /*
             * Means its not a new request but there is more data to
             * read from an older socket
             */
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "clb.proxy.http.handler_more_data");
            }
        }

        /** Invoke the proxy API
         */
        /**
         * We have
         * the buffer where the headers are read completely
         * the key
         * the remote address is available through a ThreadLocal
         * object from the HA LB. This invocation is just like whats done
         * in WSTCPProtocolHandler
         */
        ((SailfinClbProxyProtocolInfo) protocolInfo).cacheHandler = proxy.doProxyHttp(task,
                protocolInfo.byteBuffer, protocolInfo.key);
        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE,
                    "clb.proxy.http.handler_doproxy_return",
                    ((SailfinClbProxyProtocolInfo) protocolInfo).cacheHandler);
        }

        /* If we return false here the grizzly 1.0 controller will cancel key and close
         * the channel. Its a weird scenario because we will have to wait until a
         * response is received from the backend, whateven async processing we
         * do with the backend is of little use becase this thread cannot return.
         * If ret is true it means that the current requets has been read completely and
         * any more bytes on the chanenl is a new request which has to go thro
         * the finder
         *
         */
        if (!((SailfinClbProxyProtocolInfo) protocolInfo).cacheHandler) {
            protocolInfo.mappedProtocols.remove(protocolInfo.key);

            if (task == null) {
                task = proxy.getConnectionManager().getServerEndpoint(protocolInfo.key);
            }

            if (task != null) {
                task.recycle();
                objManager.offerTask(task, protocolInfo.isSecure);
            } else {
                _logger.log(Level.SEVERE, "clb.proxy.http.handler_release_fail");
            }
            ((SailfinClbProxyProtocolInfo) protocolInfo).parkRequest = true;
        } else {
            ((SailfinClbProxyProtocolInfo) protocolInfo).parkRequest = false;
        }

        // need to see how we should handle keepAlive here
        // always keep alive for now, because we will cancel key through the
        // call back

        protocolInfo.keepAlive = true;
        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE,
                    "clb.proxy.http.handler_keep_alive" +
                    protocolInfo.keepAlive);
        }
    }

    //=========================================================
    /**
     * Returns an array of supported protocols.
     * @return an array of supported protocols.
     */
    public String[] getProtocols() {
        return protocols;
    }

    /**
     * Invoked when the SelectorThread is about to expire a SelectionKey.
     * @return true if the SelectorThread should expire the SelectionKey, false
     *              if not.
     */
    public boolean expireKey(SelectionKey key) {
        _logger.log(Level.SEVERE, "clb.proxy.http.handler_expire_key", key);
        proxy.expireKey(key);
        return true;
    }

    protected void sendResponse(Request request, Response response) {
        SocketChannelOutputBuffer outputBuffer = (SocketChannelOutputBuffer) response.getOutputBuffer();

        MimeHeaders headers = response.getMimeHeaders();
        headers.setValue(LoadBalancerProxyConstants.HTTP_CONNECTION_HEADER).
                setString(LoadBalancerProxyConstants.HTTP_CONNECTION_CLOSE_VALUE);

        // Build the response header
        outputBuffer.sendStatus();

        int size = headers.size();

        for (int i = 0; i < size; i++) {
            outputBuffer.sendHeader(headers.getName(i), headers.getValue(i));
        }

        outputBuffer.endHeaders();

        try {
            outputBuffer.endRequest();
            outputBuffer.flush();
            outputBuffer.commit();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }

    protected void sendSecureResponse(Request request, Response response) {
        /**
         * If we use the SSLOutputBuffer in 1.6 grizzly, which is what we have
         * to use because only that allows us to use the tcp.Response interface,
         * we would have a problem when we flush the response. This is because
         * the 1.6 SSLOutputWriter tries to cast the workerthread to a 1.6 object
         * which will fail because we are still executing in a 1.0  worker.
         * So, as of now the best way to send a response is from here.
         */
        SSLOutputBuffer outputBuffer = (SSLOutputBuffer) response.getOutputBuffer();
        sslErrorBuffer.clear();
        MimeHeaders headers = response.getMimeHeaders();
        Enumeration names = headers.names();
        write(LoadBalancerProxyConstants.HTTP_11, sslErrorBuffer);
        write(response.getStatus() + " ", sslErrorBuffer);
        write(response.getMessage(), sslErrorBuffer);
        write(LoadBalancerProxyConstants.CRLF, sslErrorBuffer);
        while (names.hasMoreElements()) {
            String headername = (String) names.nextElement();
            String headervalue = headers.getHeader(headername);
            write(headername + ": " + headervalue, sslErrorBuffer);
            write(LoadBalancerProxyConstants.CRLF, sslErrorBuffer);
        }
        write(LoadBalancerProxyConstants.CRLF, sslErrorBuffer);
        sslErrorBuffer.flip();
        try {
            SSLOutputWriter.flushChannel(outputBuffer.getChannel(),
                    sslErrorBuffer.slice());
        } catch (IOException ex) {
            _logger.log(Level.SEVERE, "clb.proxy.handler_ssl_error_failed", ex);
        }
    }

    /**
     * This method will write the contents of the specyfied String to the
     * output stream, without filtering. This method is meant to be used to
     * write the response header.
     *
     * @param s data to be written
     */
    protected void write(String s, ByteBuffer buffer) {
        if (s == null) {
            return;
        }
        byte[] b = s.getBytes();
        buffer.put(b);
    }

    private static final String dumpBuffer(ByteBuffer byteBuffer) {
        ByteBuffer dd = byteBuffer.duplicate();
        dd.flip();
        int length = dd.limit();
        byte[] dump = new byte[length];
        dd.get(dump, 0, length);
        return (new String(dump));
    }
}
