/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.proxy.outbound;

import com.sun.grizzly.CallbackHandler;
import com.sun.grizzly.ConnectorHandler;
import com.sun.grizzly.Context;
import com.sun.grizzly.Controller;
import com.sun.grizzly.IOEvent;
import com.sun.grizzly.filter.ReadFilter;
import com.sun.grizzly.util.OutputWriter;
import com.sun.grizzly.util.SSLOutputWriter;
import com.sun.grizzly.util.WorkerThread;

import org.jvnet.glassfish.comms.clb.proxy.HttpProxy;
import org.jvnet.glassfish.comms.clb.proxy.ProxyRequestHandler;
import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;

import javax.net.ssl.SSLEngine;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author rp138409
 */
public class DefaultCallBackHandler implements CallbackHandler<Context> {

    protected ConnectorHandler connectorHandler;
    protected Logger _logger = null;
    protected SelectionKey clientKey;
    protected SelectionKey serverKey;
    protected ReadFilter readFilter;
    protected ResponseHandler responseHandler;
    protected Endpoint ep;
    protected boolean keepAlive = false;
    protected ConnectionManager connectionManager;
    protected SSLEngine sslEngine;
    protected ByteBuffer outputBB;
    protected boolean isSecure = false;
    boolean parsed = false;

    /** Creates a new instance of DefaultCallBackHandler */
    public DefaultCallBackHandler(ConnectorHandler handler, ProxyRequestHandler task) {
        _logger = ProxyConfig.getInstance().getLogger();
        connectorHandler = handler;
        ep = task.getEndpoint();
        clientKey = task.getSelectionKey();
        keepAlive = task.getKeepAlive();
        connectionManager = HttpProxy.getInstance().getConnectionManager();
        sslEngine = task.getSSLEngine();
        outputBB = task.getOutputBB();
        isSecure = task.isSecure();
        readFilter = new ReadFilter();
        responseHandler = new ResponseHandler();
    }

    public void onWrite(IOEvent<Context> ioEvent) {
        throw new IllegalStateException("Invalid state");
    }

    private void dumpBuffer(ByteBuffer buf) {
        int pos = buf.position();
        int limit = buf.limit();
        StringBuffer buff = new StringBuffer(buf.limit());
        for (int i = 0; i < buf.limit(); i++) {
            buff.append((char) buf.get(i));
        }
        buf.position(pos);
        buf.limit(limit);
        _logger.log(Level.FINE, buff.toString());
    }

    /**
     * Called when the connection has been established.
     */
    public void onConnect(IOEvent<Context> ioEvent) {
        serverKey = ioEvent.attachment().getSelectionKey();
        try {
            connectorHandler.finishConnect(serverKey);
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }
        ioEvent.attachment().getController().registerKey(
                serverKey, SelectionKey.OP_READ, Controller.Protocol.TCP);
        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE, "clb.proxy.callback_serverkey" + serverKey);
            _logger.log(Level.FINE, "clb.proxy.callback.clientkey" + clientKey.channel());
            _logger.log(Level.FINE, "clb.proxy.callback.server_channel" + serverKey.channel());
        }
    }

    public void onRead(IOEvent<Context> ioEvent) {
        Context ctx = ioEvent.attachment();
        ByteBuffer byteBuffer = null;
        try {
            SocketChannel clientChannel = (SocketChannel) clientKey.channel();
            readFilter.execute(ctx);
            WorkerThread workerThread = (WorkerThread) Thread.currentThread();
            byteBuffer = workerThread.getByteBuffer();
            if (byteBuffer.position() == 0) {
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.log(Level.FINE, "clb.proxy.callback_algorithm_zero_bytes");
                }
                return;
            }
            if (!parsed) {
                while (!responseHandler.parse(byteBuffer)) {
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.log(Level.FINE, "clb.proxy.callback_algorithm_parse_false");
                    }
                    readFilter.execute(ctx);
                }
                parsed = true;
            } else {
                byteBuffer.flip();
            }
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "clb.proxy.callback.algorithm_finished",
                        clientChannel.socket().getRemoteSocketAddress());
                _logger.log(Level.FINE, "clb.proxy.callback_clientkey",
                        clientKey.channel());
                _logger.log(Level.FINE, "clb.proxy.callback.server_channel",
                        ctx.getSelectionKey().channel());
                _logger.log(Level.FINE, "Byte Buffer " + byteBuffer);
                try {
                    dumpBuffer(byteBuffer);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            long byteswritten = 0;
            if ((clientChannel != null) && (clientChannel.isConnected())) {
                if (isSecure) {
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.log(Level.FINE, "clb.proxy.callback.ssl_writer",
                                clientChannel + " " + byteBuffer + " " + outputBB + " " + sslEngine);
                    }
                    byteswritten = SSLOutputWriter.flushChannel(clientChannel,
                            byteBuffer, outputBB, sslEngine);
                } else {
                    byteswritten = OutputWriter.flushChannel(clientChannel, byteBuffer);
                }
                if (responseHandler.isTransferEncoding()) {
                    if (_logger.isLoggable(Level.FINE)){
                        _logger.log(Level.FINE, "Transfer encoding Byte buffer " + byteBuffer);
                    }
                    /**
                     * It is quite inefficient to search for termination in the whole
                     * bytebuffer, Needs to be modified
                     */
                    responseHandler.updateEnd((ByteBuffer) byteBuffer.flip());
                } else {
                    responseHandler.updateBytesWritten(byteswritten);
                }

            } else {
                _logger.log(Level.SEVERE,
                        "clb.proxy.callback_client_channel_closed");
                throw new IOException("Socket_channel_closed");
            }
            if (byteBuffer != null) {
                byteBuffer.clear();
            }
            if (responseHandler.hasRemaining()) {
                SelectionKey key = ctx.getSelectionKey();
                /**
                 * Something more has to be done here
                 */
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.log(Level.FINE,
                            "clb.proxy.callback.algorithm_more_data",
                            clientChannel.socket().getRemoteSocketAddress());
                }
                ctx.getSelectorHandler().register(key, SelectionKey.OP_READ);
            } else {
                connectionManager.removeClientEndpoint(clientKey);
                connectionManager.releaseConnection(this.connectorHandler);
                if (!keepAlive) {
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.log(Level.FINE,
                                "clb.proxy.callback.algorithm_cancelled_key",
                                clientChannel.socket().getRemoteSocketAddress());
                    }
                    connectionManager.cancelClientKey(clientKey);
                } else {
                    if (_logger.isLoggable(Level.FINE)) {
                        _logger.log(Level.FINE,
                                "clb.proxy.callback.algorithm_registered_key",
                                clientChannel.socket().getRemoteSocketAddress());
                    }
                    connectionManager.registerClientKey(clientKey);
                }
            }

        } catch (Throwable e) {
            ctx.getController().cancelKey(ctx.getSelectionKey());
            HttpProxy.getInstance().getConnectionManager().releaseConnection(this.connectorHandler);
            connectionManager.removeClientEndpoint(clientKey);
            _logger.log(Level.SEVERE, "clb.proxy.callback.exception", e);
        } finally {
            if (outputBB != null) {
                outputBB.clear();
            }
        }
    }
}
