/*
* The contents of this file are subject to the terms
* of the Common Development and Distribution License
* (the License).  You may not use this file except in
* compliance with the License.
*
* You can obtain a copy of the license at
* https://glassfish.dev.java.net/public/CDDLv1.0.html or
* glassfish/bootstrap/legal/CDDLv1.0.txt.
* See the License for the specific language governing
* permissions and limitations under the License.
*
* When distributing Covered Code, include this CDDL
* Header Notice in each file and include the License file
* at glassfish/bootstrap/legal/CDDLv1.0.txt.
* If applicable, add the following below the CDDL Header,
* with the fields enclosed by brackets [] replaced by
* you own identifying information:
* "Portions Copyrighted [year] [name of copyright owner]"
*
* Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
* Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
*/
package org.jvnet.glassfish.comms.admin.config;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigRuntimeException;
import com.ericsson.ssa.config.event.ConfigAddEvent;
import com.ericsson.ssa.config.event.ConfigEvent;
import com.ericsson.ssa.config.event.ConfigEventMulticaster;
import com.ericsson.ssa.config.event.ConfigRemoveEvent;
import com.ericsson.ssa.config.event.ConfigUpdateEvent;

import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;
import java.io.InputStream;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.prefs.AbstractPreferences;
import java.util.prefs.BackingStoreException;


/**
 * Replaces the original Config implementation with an in memory backing store.
 */
public class SimpleConfig implements Config {
    private static String FACTORY_DEFAULTS = "sip_container_config_defaults.xml";
    private static String ADDITIONAL_PROPERTY_NODE="ElementProperty";

    //The Preferences is not used as prefrences
    //It is used as a tree structured collection of
    //in memory properties.
    private SimplePreferences prefs = new SimplePreferences();

    //This is lagecy as the Preferences used now is not anymore shared
    //by multiple instances in a cluster. Still kept the concept as it doesn't
    //harm and may ease tracing in a cluster environement.
    private String serverSpecificNode;
    private ConfigEventMulticaster eventMulticaster;
    private LogUtil log = LogUtil.SIP_LOGGER;

    //Event and transaction  
    private ThreadLocal<List<ConfigEvent>> tlEventQueue = new ThreadLocal<List<ConfigEvent>>();
    private ThreadLocal<String> tlEventMask = new ThreadLocal<String>();
    private ThreadLocal<List<String>> tlEventTypes = new ThreadLocal<List<String>>();

    public SimpleConfig(String aServerSpecificNode) {
        this(aServerSpecificNode, null);
    }

    public SimpleConfig(String aServerSpecificNode, InputStream aFactoryDefaults) {
        serverSpecificNode = aServerSpecificNode;
        setFactoryDefaults(aFactoryDefaults);
    }

    //Config   
    public String get(String key) {
        String value = get(serverSpecificNode, key);
        
        if (value==null) {
            value = System.getProperty(key);
            if (log.isLoggable(Level.FINEST)) {
                log.logMsg(Level.FINEST, "Using extention mechanism: jvm-option, key ["+key+
                        "] -> "+value);
            }            
        }
        
        return value;
    }

    public String get(String node, String key) {
        String value = prefs.node(node).get(key, null);

        try {
            if (value == null && 
                    prefs.node(node).nodeExists(ADDITIONAL_PROPERTY_NODE)) {                
                value=prefs.node(node).node(ADDITIONAL_PROPERTY_NODE).get(key, null);
                if (log.isLoggable(Level.FINEST)) {
                    log.logMsg(Level.FINEST, "Using extention mechanism: additional property, key ["+key+
                            "] and node ["+node+"]-> "+value);
                }                
            }
        } catch (BackingStoreException ex) {
            //Ignore
        }
        
        return value;        
    }

    public String[] getChildNames(String node) {
        String[] names = null;

        try {
            names = prefs.node(node).childrenNames();
        } catch (BackingStoreException ex) {
            //ignore shouldn't happen!
            names = new String[] {  };
        }

        return names;
    }

    public void set(String key, String value) {
        set(serverSpecificNode, key, value);
    }

    public void set(String node, String key, String value) {
        prefs.node(node).put(key, value);
    }

    public void clear() {
        clear(serverSpecificNode);
    }

    public void clearAll() {
        clear();
    }

    public void clear(String node) {
        try {
            prefs.node(node).clear();
        } catch (BackingStoreException ex) {
            if (log.logWarning()) {
                log.warning(ex, "sip.integration.operation_failed_continue",
                    getClass().getName(), "clear");
            }
        }
    }

    public void remove(String node) {
        try {
            prefs.node(node).removeNode();
            prefs.flush();
        } catch (BackingStoreException ex) {
            if (log.logWarning()) {
                log.warning(ex, "sip.integration.operation_failed_continue",
                    getClass().getName(), "remove");
            }
        }
    }

    public void copyConfig(String fromNode, String toNode) {
        try {
            for (String key : prefs.node(fromNode).keys()) {
                String value = prefs.node(fromNode).get(key, null);
                prefs.node(toNode).put(key, value);
            }
        } catch (BackingStoreException ex) {
            if (log.logWarning()) {
                log.warning(ex, "sip.integration.operation_failed_continue",
                    getClass().getName(), "copyConfig");
            }
        }
    }

    //Event and transaction
    public void setEventMulticaster(ConfigEventMulticaster anEventMulticaster) {
        eventMulticaster = anEventMulticaster;
    }

    public void nodeAdded(String parentNode, String childNode) {
        List<ConfigEvent> currentEventQueue = tlEventQueue.get();

        if (currentEventQueue != null) {
            if (log.isLoggable(Level.FINEST)) {
                log.logMsg(Level.FINEST, "Capturing nodeAdded parentNode = " +
                    parentNode + ", childNode = " + childNode + ", eventMask = " +
                    tlEventMask.get() + ", eventTypes = " + tlEventTypes.get());
            }
            
            if ((tlEventTypes.get() == null) ||
                    tlEventTypes.get().contains(ConfigAddEvent.TYPE)) {
                if ((tlEventMask.get() == null) ||
                        parentNode.matches(tlEventMask.get())) {
                    ConfigAddEvent configEvent = new ConfigAddEvent(this,
                            parentNode, childNode);

                    if (!currentEventQueue.contains(configEvent)) {
                        currentEventQueue.add(configEvent);
                        if (log.isLoggable(Level.FINEST)) {
                            log.logMsg(Level.FINEST, "Captured event " +
                                configEvent.getType() + " : " +
                                configEvent.getNode() + " mask: " +
                                tlEventMask.get() + " types: " +
                                tlEventTypes.get());
                        }
                    }
                }
            }
        }
    }

    public void nodeChanged(String node) {
        List<ConfigEvent> currentEventQueue = tlEventQueue.get();

        if (currentEventQueue != null) {
            if (log.isLoggable(Level.FINEST)) {
                log.logMsg(Level.FINEST, "Capturing nodeChanged node = " + node +
                    ", eventMask = " + tlEventMask.get() + ", eventTypes = " +
                    tlEventTypes.get());
            }
            
            if ((tlEventTypes.get() == null) ||
                    tlEventTypes.get().contains(ConfigUpdateEvent.TYPE)) {
                if ((tlEventMask.get() == null) ||
                        node.matches(tlEventMask.get())) {
                    ConfigUpdateEvent configEvent = new ConfigUpdateEvent(this,
                            node);

                    if (!currentEventQueue.contains(configEvent)) {
                        currentEventQueue.add(configEvent);
                        if (log.isLoggable(Level.FINEST)) {
                            log.logMsg(Level.FINEST, "Captured event " +
                                configEvent.getType() + " : " +
                                configEvent.getNode() + " mask: " +
                                tlEventMask.get() + " types: " +
                                tlEventTypes.get());
                        }
                    } else {
                        if (log.isLoggable(Level.FINEST)) {
                            log.logMsg(Level.FINEST, "Skipped duplicate event " +
                                configEvent.getType() + " : " +
                                configEvent.getNode() + " mask: " +
                                tlEventMask.get() + " types: " +
                                tlEventTypes.get());
                        }
                    }
                }
            }
        }
    }

    public void nodeRemoved(String parentNode, String childNode) {
        List<ConfigEvent> currentEventQueue = tlEventQueue.get();

        if (currentEventQueue != null) {
            if (log.isLoggable(Level.FINEST)) {
                log.logMsg(Level.FINEST, "Capturing nodeRemoved parentNode = " +
                    parentNode + ", childNode = " + childNode + ", eventMask = " +
                    tlEventMask.get() + ", eventTypes = " + tlEventTypes.get());
            }
            
            if ((tlEventTypes.get() == null) ||
                    tlEventTypes.get().contains(ConfigRemoveEvent.TYPE)) {
                if ((tlEventMask.get() == null) ||
                        parentNode.matches(tlEventMask.get())) {
                    ConfigRemoveEvent configEvent = new ConfigRemoveEvent(this,
                            parentNode, childNode);

                    if (!currentEventQueue.contains(configEvent)) {
                        currentEventQueue.add(configEvent);
                        if (log.isLoggable(Level.FINEST)) {
                            log.logMsg(Level.FINEST, "Captured event " +
                                configEvent.getType() + " : " +
                                configEvent.getNode() + " mask: " +
                                tlEventMask.get() + " types: " +
                                tlEventTypes.get());
                        }
                    }
                }
            }
        }
    }

    public void captureEvents() {
        captureEvents(null, null);
    }

    public void captureEvents(String eventMask) {
        captureEvents(eventMask, null);
    }

    public void captureEvents(String[] eventTypes) {
        captureEvents(null, eventTypes);
    }

    public void captureEvents(String eventMask, String[] eventTypes) {
        tlEventQueue.set(new ArrayList<ConfigEvent>());
        tlEventMask.set(eventMask);

        if (eventTypes != null) {
            tlEventTypes.set(Arrays.asList(eventTypes));
        }
    }

    public void processEvents() {
        List<ConfigEvent> currentEventQueue = tlEventQueue.get();

        ConfigRuntimeException troubles = null;
        if (currentEventQueue != null) {
            try {
               for (ConfigEvent event : currentEventQueue) {
                    if (log.isLoggable(Level.FINEST)) {
                        log.logMsg(Level.FINEST, "processing event: [" +
                            event.getClass().getName() + ", " + event.getNode() +
                            "]");
                    }
                    try {
                        eventMulticaster.processEvent(event);
                    } catch(Exception e) {                        
                        //This catch may be to wide!
                        //Each event gets the opertunaty be handled and
                        //lokalize the effect of troublesome listeners.
                        if (troubles==null) {
                            troubles = new ConfigRuntimeException("Processing events failed for atleast one event.");
                        }
                        troubles.addCause(e);
                    }
                }
            } finally {
                currentEventQueue.clear();
                tlEventQueue.set(null);
            }
        }
        
        if (troubles!=null) {
            throw troubles;
        }
    }
    
    //Support    
    @Override
    public String toString() {
        StringBuffer buffer = new StringBuffer("Config[" + hashCode() + "]:  ");

        buffer.append(prefs.toString());

        return buffer.toString();
    }

    private void setFactoryDefaults(InputStream aFactoryDefaults) {
        InputStream factoryDefaults = aFactoryDefaults;

        if (factoryDefaults == null) {
            factoryDefaults = getClass().getResourceAsStream(FACTORY_DEFAULTS);
        }

        if (factoryDefaults != null) {
            Properties def = new Properties();

            try {
                def.loadFromXML(factoryDefaults);

                for (Object key : def.keySet()) {
                    String[] nodeAndPrefKey = ((String) key).split(":", 2);

                    if ((nodeAndPrefKey != null) &&
                            (nodeAndPrefKey.length == 2)) {
                        String nodePath = nodeAndPrefKey[0].replace('.', '/');
                        nodePath = "".equals(nodePath) ? "/" : nodePath;

                        String value = (String) def.get(key);

                        prefs.node(nodePath).put(nodeAndPrefKey[1], value);
                    } else {
                        if (log.logWarning()) {
                            log.warning("sip.integration.resource_incompatible_continue",
                                "'factory defaults'", key);
                        }
                    }
                }
            } catch (IOException ex) {
                if (log.logWarning()) {
                    log.warning(ex,
                        "sip.integration.resource_not_found_continue",
                        "'factory defaults'");
                }
            }
        } else {
            if (log.logWarning()) {
                log.warning("sip.integration.resource_not_found_continue",
                    "'factory defaults'");
            }
        }
    }

    //Custom Preferences that just is a in memory backing store.
    class SimplePreferences extends AbstractPreferences {
        HashMap<String, String> preferences = new HashMap<String, String>();
        HashMap<String, SimplePreferences> childeren = new HashMap<String, SimplePreferences>();

        public SimplePreferences() {
            super(null, "");
        }

        public SimplePreferences(SimplePreferences parent, String name) {
            super(parent, name);
        }

        protected void putSpi(String key, String value) {
            if (log.isLoggable(Level.FINEST)) {
            log.logMsg(Level.FINEST, "SimplePreferences[" + hashCode() +
                "] putSpi called: [" + key + ", " + value + "]");
            }
            
            String oldValue = preferences.get(key);

            preferences.put(key, value);

            if (((value != null) && !value.equals(oldValue)) ||
                    ((value == null) && (oldValue != null))) {
                nodeChanged(absolutePath());
            }
        }

        protected String getSpi(String key) {
            return preferences.get(key);
        }

        protected void removeSpi(String key) {
            preferences.remove(key);
        }

        protected void removeNodeSpi() throws BackingStoreException {
            preferences.clear();
            childeren.clear();

            if (parent() != null) {
                ((SimplePreferences) parent()).removeChild(this);
            }
        }

        protected String[] keysSpi() throws BackingStoreException {
            return preferences.keySet()
                              .toArray(new String[preferences.keySet().size()]);
        }

        protected String[] childrenNamesSpi() throws BackingStoreException {
            return childeren.keySet()
                            .toArray(new String[childeren.keySet().size()]);
        }

        protected AbstractPreferences childSpi(String name) {
            if (log.isLoggable(Level.FINEST)) {
                log.logMsg(Level.FINEST, "SimplePreferences[" + hashCode() +
                    "] childSpi called: [" + name + "]\n"+ toString());
            }
            SimplePreferences child = childeren.get(name);

            if (child == null) {
                child = new SimplePreferences(this, name);
                childeren.put(name, child);
            }

            nodeAdded(absolutePath(), child.absolutePath());

            return child;
        }

        protected void syncSpi() throws BackingStoreException {
        }

        protected void flushSpi() throws BackingStoreException {
        }

        private void removeChild(SimplePreferences child) {
            childeren.remove(child.name());

            nodeRemoved(absolutePath(), child.absolutePath());
        }

        public String toString() {
            StringBuffer buffer = new StringBuffer("SimplePreference[" +
                    hashCode() + "]: \n");

            toString(buffer, "");
            buffer.append("\n");

            return buffer.toString();
        }

        public String toString(StringBuffer buffer, String indent) {
            buffer.append(indent);
            buffer.append("<" + ((name().length() != 0) ? name() : "root") +
                ">");

            for (SimplePreferences child : childeren.values()) {
                buffer.append("\n");
                child.toString(buffer, indent + "  ");
            }

            if (!preferences.isEmpty()) {
                buffer.append("\n  ");
                buffer.append(indent);
                buffer.append(preferences.toString());
            }

            buffer.append("\n");
            buffer.append(indent);
            buffer.append("</" + ((name().length() != 0) ? name() : "root") +
                ">");

            return buffer.toString();
        }
    }
}
