/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.replication.dialogmgmt;

import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.DialogSet;
import com.ericsson.ssa.sip.PersistentDialogFragmentManager;
import com.sun.appserv.ha.spi.BackingStore;
import com.sun.appserv.ha.spi.BackingStoreException;
import com.sun.appserv.ha.spi.BackingStoreFactory;
import com.sun.appserv.ha.spi.BackingStoreRegistry;
import com.sun.appserv.ha.spi.SimpleMetadata;
import com.sun.ejb.containers.util.cache.BaseCache;
import com.sun.enterprise.ee.web.initialization.ServerConfigReader;
import com.sun.enterprise.ee.web.sessmgmt.EEPersistenceTypeResolver;
import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreFactory;
import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreImpl;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationReceiver;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationSender;
import com.sun.enterprise.ee.web.sessmgmt.PurgeCapable;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationHealthChecker;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationManager;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationMessageRouter;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationResponseRepository;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationStateQueryResponse;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import com.sun.enterprise.ee.web.sessmgmt.StorePool;
import com.sun.enterprise.ee.web.sessmgmt.StorePoolElement;
import com.sun.logging.LogDomains;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import org.apache.catalina.LifecycleException;
import org.apache.catalina.security.SecurityUtil;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * In-memory replicating DialogFragmentManager.
 * 
 * Get this singleton instance using {@link com.ericsson.ssa.sip.DialogFragmentManager#getInstance()}.
 */
public class ReplicationDialogFragmentManager
        extends PersistentDialogFragmentManager
        implements ReplicationManager, PurgeCapable {

    // Singleton class so app_id is always this constant.
    private static final String APP_ID = "DialogFragmentManager";

    private static final Logger _logger = Logger.getLogger(LogUtil.SIP_LOG_DOMAIN);
    protected static final String MODE_SIP = ReplicationState.MODE_SIP;
    final static String DUPLICATE_IDS_SEMANTICS_PROPERTY 
        = ReplicationState.DUPLICATE_IDS_SEMANTICS_PROPERTY; 
    final static String COMMAND_MAP = "commandmap";
    private static final Level TRACE_LEVEL = Level.FINE;
    protected static int _maxBaseCacheSize = 4096;
    protected static float _loadFactor = 0.75f;
    
    private static final Map dialogFragmentCommandMap = new HashMap();
   
    public final static String SAVE_COMMAND 
        = ReplicationState.SAVE_COMMAND;
     public final static String VALVE_SAVE_COMMAND 
        = ReplicationState.VALVE_SAVE_COMMAND;
    public final static String REMOVE_COMMAND 
        = ReplicationState.REMOVE_COMMAND;    
    public final static String UNDEPLOY_COMMAND 
        = ReplicationState.UNDEPLOY_COMMAND;
    public final static String REMOVE_EXPIRED_COMMAND 
        = ReplicationState.REMOVE_EXPIRED_COMMAND;
    public final static String UPDATE_LAST_ACCESS_TIME_COMMAND 
        = ReplicationState.UPDATE_LAST_ACCESS_TIME_COMMAND;
    public final static String MESSAGE_BROADCAST_QUERY 
        = ReplicationState.MESSAGE_BROADCAST_QUERY;
    public final static String MESSAGE_BROADCAST_LOAD_RECEIVED
        = ReplicationState.MESSAGE_BROADCAST_LOAD_RECEIVED;
    
    public final static String SAVE_DIALOG_FRAGMENT_COMMAND = "savedialogfragment";
    public final static String REMOVE_DIALOG_FRAGMENT_COMMAND = "removedialogfragment";    
    public final static String UPDATE_LAST_ACCESS_TIME_DIALOG_FRAGMENT_COMMAND 
        = "updatelastaccesstimedialogfragment";
    public final static String MESSAGE_BROADCAST_QUERY_DIALOG_FRAGMENT 
        = "broadcastfinddialogfragment"; 
 
    /**
     * the list of method names that are broadcasts
     */ 
    private static List broadcastMethods 
        = Arrays.asList(MESSAGE_BROADCAST_QUERY_DIALOG_FRAGMENT);    
    
    private StorePool dialogFragmentStorePool;

    /**
     * Our cache of replicated DialogFragment ReplicationState objects
     * keyed by id
     */
    protected BaseCache replicatedDialogFragments = new BaseCache();

    /**
     * Our cache of replicated DialogFragement ReplicationStateUpdate
     * objects
     * keyed by id
     */
    protected BaseCache replicatedDialogFragmentUpdates = null;
    
    /**
    * Our Replicator instance for DialogFragments (for SimpleMetadata)
    */
    private BackingStore dialogFragmentBackingStore = null; 
    
    protected String _passedInPersistenceType = null;
    protected boolean _duplicateIdsSemanticsAllowed = false;  
  
    private AtomicBoolean isReplicationInitialized = new AtomicBoolean(); //defaults to false
    private AtomicBoolean isReplicationStarted = new AtomicBoolean(); //defaults to false

    private final String instanceName;
   
    static {
        checkSessionCacheProperties();
        initializeCommandMap();
    }

    /**
     * Constructor, which registers the new instance with the
     * ReplicationMessageRouter.
     *
     * @see com.ericsson.ssa.sip.DialogFragmentManager#getInstance()
     */
    public ReplicationDialogFragmentManager() {
        super();
        instanceName = ReplicationUtil.getInstanceName();
    }

    protected static boolean checkSessionCacheProperties() {
        boolean result = false;

        try {
            Properties props = System.getProperties();
            String cacheSize = props.getProperty(
                    "HTTP_SESSION_CACHE_MAX_BASE_CACHE_SIZE");

            if (null != cacheSize) {
                _maxBaseCacheSize = (new Integer(cacheSize).intValue());
            }

            String loadFactor = props.getProperty(
                    "HTTP_SESSION_CACHE_MAX_BASE_LOAD_FACTOR");

            if (null != loadFactor) {
                _loadFactor = (new Float(loadFactor).floatValue());
            }
        } catch (Exception e) {
            //do nothing accept defaults
        }

        return result;
    }
    
    protected static void initializeCommandMap() {
        dialogFragmentCommandMap.put(SAVE_COMMAND, SAVE_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(VALVE_SAVE_COMMAND, SAVE_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(REMOVE_COMMAND, REMOVE_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(UPDATE_LAST_ACCESS_TIME_COMMAND, UPDATE_LAST_ACCESS_TIME_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(MESSAGE_BROADCAST_QUERY, MESSAGE_BROADCAST_QUERY_DIALOG_FRAGMENT);   
    }
    
    /**
     * Initialize ReplicationDialogFragmentManager replication services.
     *
     * @param passedInPersistenceType 
     * @param haStorePoolSize 
     * @param haStorePoolUpperSize 
     * @param haStorePoolPollTime 
     */
    public void initializeReplication(String passedInPersistenceType, int haStorePoolSize, 
                                      int haStorePoolUpperSize, int haStorePoolPollTime) {
        if (isInitialized.get() == false ) {
            IllegalStateException ex = new IllegalStateException("DialogFragmentManager must be initialized prior to initializing replication.");
            _logger.log(Level.SEVERE, null, ex);
            throw ex;
        }


        // Only initialize once.
        if (isReplicationInitialized.compareAndSet(false, true)) {
            //initialize replicated DialogFragment cache
            replicatedDialogFragments = new BaseCache();
            replicatedDialogFragments.init(_maxBaseCacheSize, _loadFactor,
                                            null);
        
            replicatedDialogFragmentUpdates = new BaseCache();
            replicatedDialogFragmentUpdates.init(_maxBaseCacheSize,
                                                 _loadFactor, null);

            setPassedInPersistenceType(passedInPersistenceType);
            StorePool dialogFragmentStorePool = 
                    new StorePool(haStorePoolSize, haStorePoolUpperSize, haStorePoolPollTime, 
                                  new DialogFragmentStoreFactory());
            setDialogFragmentStorePool(dialogFragmentStorePool);
            setDialogFragmentStore(new DialogFragmentStoreImpl());
        }
    }
    
    public void startReplication() {
        if (isStarted.get() == false ) {
            IllegalStateException ex = new IllegalStateException("DialogFragmentManager must be started prior to starting replication.");
            _logger.log(Level.SEVERE, null, ex);
            throw ex;
        }

        // Only start once.
        if (isReplicationStarted.compareAndSet(false, true)) {
            registerWithMessageRouter();
        }
    }

    public boolean isReplicationStarted() {
        return isReplicationStarted.get();
    }
    
    @Override
    protected void doStop() {
        super.doStop();        
        deregisterWithMessageRouter();
    }
    
     /**
     * Factory method to instansiate DialogFragments
     * @param ds
     * @return
     */
    public DialogFragment createDialogFragment(DialogSet ds) {
        return new HADialogFragment(ds);
    }
     
    
    private void registerWithMessageRouter() {
        ReplicationMessageRouter router = ReplicationMessageRouter.createInstance();
        if(router != null) {
            router.addReplicationManager(this.getApplicationId(), (ReplicationManager)this);
            router.registerBroadcastMethodList(broadcastMethods);
        }
    }  
    
    private void deregisterWithMessageRouter() {
        ReplicationMessageRouter router = ReplicationMessageRouter.createInstance();
        if(router != null) {
            router.removeReplicationManager(this.getApplicationId());
            // FIXME currently no means to deregister Broadcast methodlist.
        }
    }  
    
    public String getApplicationId() {
        return APP_ID;
    }
    
     /**
    * get the backingStore for DialogFragment
    */ 
    synchronized public BackingStore getDialogFragmentBackingStore() {
        if (dialogFragmentBackingStore == null) {
            dialogFragmentBackingStore = createBackingStore(dialogFragmentCommandMap); 
        }
        return dialogFragmentBackingStore;
    }
    
   /**
    * create and set the backing store
    * @param commandMap map used to translate commands
    */     
    BackingStore createBackingStore(Map commandMap) {
        BackingStoreFactory storeFactory = new JxtaBackingStoreFactory();
        BackingStoreRegistry backingStoreRegistry 
            = BackingStoreRegistry.getInstance();
        Properties inputEnv = backingStoreRegistry.getFactoryClassEnv(getPassedInPersistenceType());
        Properties env = (Properties)inputEnv.clone();
        env.put(DUPLICATE_IDS_SEMANTICS_PROPERTY, Boolean.valueOf(this.isDuplicateIdsSemanticsAllowed()));
        env.put(COMMAND_MAP, commandMap);
        BackingStore backingStore = null;
        try {
            backingStore = storeFactory.createBackingStore(
                        SimpleMetadata.class,     //type
                        this.getApplicationId(), //appid
                        env);
        } catch (BackingStoreException ex) {
            //deliberate no-op
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("backingStore: " + backingStore);
        }         
        if(backingStore != null) {
            if(backingStore instanceof JxtaBackingStoreImpl) {
                ((JxtaBackingStoreImpl)backingStore).setMode(MODE_SIP);
            }        
        }
        return backingStore;
    }        
    
    public String getPassedInPersistenceType() {
        return _passedInPersistenceType;
    }    
    
    public void setPassedInPersistenceType(String persistenceType) {
        _passedInPersistenceType = persistenceType;
    }    
    
    public boolean isDuplicateIdsSemanticsAllowed() {
        return _duplicateIdsSemanticsAllowed;
    }    
    
    public void setDuplicateIdsSemanticsAllowed(boolean value) {
        _duplicateIdsSemanticsAllowed = value;
    }   
    
    //DialogFragment cache methods

    /**
     * get the replicated DialogFragments cache
     */ 
    public BaseCache getReplicatedDialogFragments() {
        return replicatedDialogFragments;
    }
    
    /**
     * set the replicated DialogFragments cache
     * @param dialogFragmentTable
     */ 
    public void setReplicatedDialogFragments(BaseCache dialogFragmentTable) {
        replicatedDialogFragments = dialogFragmentTable;
    } 
    
    /**
     * Put dialogFragment State in DialogFragment replica cache
     * @param dialogFragmentState
     */    
    protected synchronized void putInDialogFragmentReplicationCache(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName()
                         + ">>putInDialogFragmentReplicationCache id: "
                         + dialogFragmentState.getId());
        } 
        if (dialogFragmentState == null) {
            return;
        }
        String id = (String)dialogFragmentState.getId();
        if(id == null) {
            return;
        }
        
        ReplicationState currentState = (ReplicationState)replicatedDialogFragments.get(id);
        if(_logger.isLoggable(Level.FINE)) {
            if(currentState != null) {
                _logger.fine("currentVersion: " + currentState.getVersion()
                             + " newVersion: " + dialogFragmentState.getVersion());
            }
        }
        if((currentState != null)
                && currentState.getVersion() > dialogFragmentState.getVersion()) {
            return;
        }
        replicatedDialogFragments.put(dialogFragmentState.getId(), dialogFragmentState);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE, "in " + this.getClass().getName()
                + ">>putInDialogFragmentReplicationCache complete id: "
                + dialogFragmentState.getId() + "[ver:" + dialogFragmentState.getVersion()
                + "]");
        }        
    }
    
    /**
     * get DialogFragment State from replica cache based on the id
     * @param id
     */    
    protected ReplicationState getFromDialogFragmentReplicationCache(String id) {
        if (replicatedDialogFragments == null) {
            return null;
        }
        ReplicationState returnState 
            = (ReplicationState)replicatedDialogFragments.get(id);
        if(returnState != null) {
            ReplicationStateUpdate returnStateUpdate
                = this.getFromDialogFragmentReplicationUpdateCache(id);
            //apply update if it exists and has higher version
            if(returnStateUpdate != null) {
                if(returnStateUpdate.getVersion() > returnState.getVersion()) {
                    returnState.setVersion(returnStateUpdate.getVersion());
                    returnState.setLastAccess(returnStateUpdate.getLastAccess());
                }
                this.removeFromDialogFragmentReplicationUpdateCache(returnStateUpdate.getId());
            }            
        }
        return returnState;
    }
    
    /**
     * remove dialogFragment State from replica cache based on the id 
     * @param id
     */    
    protected ReplicationState removeFromDialogFragmentReplicationCache(String id) {
        if(id == null) {
            return null;
        }
        if (replicatedDialogFragments == null) {
            return null;
        }
        removeFromDialogFragmentReplicationUpdateCache(id);
        return (ReplicationState)replicatedDialogFragments.remove(id);        
    }

    //DF updates cache methods
    
    /**
     * Put DialogFragment State updates in replica cache
     * @param dialogFragmentState
     */    
    protected synchronized void putInDialogFragmentReplicationUpdateCache(ReplicationStateUpdate dialogFragmentStateUpdate) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName()
                         + ">>putInDialogFragmentReplicationUpdateCache id: "
                         + dialogFragmentStateUpdate.getId());
        }
        if(dialogFragmentStateUpdate == null) {
            return;
        }
        String id = (String)dialogFragmentStateUpdate.getId();
        if(id == null) {
            return;
        }
        
        ReplicationStateUpdate currentStateUpdate = this.getFromDialogFragmentReplicationUpdateCache(id);
        if(_logger.isLoggable(Level.FINE)) {
            if(currentStateUpdate != null) {
                _logger.fine("currentVersion: "
                             + currentStateUpdate.getVersion()
                             + " newVersion: "
                             + dialogFragmentStateUpdate.getVersion());
            }
        }
        if((currentStateUpdate != null)
                && currentStateUpdate.getVersion() > dialogFragmentStateUpdate.getVersion()) {
            return;
        }
        replicatedDialogFragmentUpdates.put(dialogFragmentStateUpdate.getId(),
                                            dialogFragmentStateUpdate);
    }
    
    /**
     * get DialogFragment State update from replica cache based on the id
     * @param id
     */    
    protected ReplicationStateUpdate getFromDialogFragmentReplicationUpdateCache(String id) {    
        if( id == null || replicatedDialogFragmentUpdates == null) {
            return null;
        }
        return (ReplicationStateUpdate)replicatedDialogFragmentUpdates.get(id);
    }
    
    /**
     * remove dialogFragment State update from replica cache based on the id of
     * dialogFragmentStateUpdate
     * @param id
     */    
    protected void removeFromDialogFragmentReplicationUpdateCache(String id) {
        if(id == null || replicatedDialogFragmentUpdates == null) {
            return;
        }
        replicatedDialogFragmentUpdates.remove(id);
    }   
    
    /** Returns a store from the pool 
     * This method intializes the store with right parameters
     * @return returns DialogFragmentStoreImpl
     */
    public DialogFragmentStoreImpl getDialogFragmentStore() {
        if(_logger.isLoggable(Level.FINER)) {
            _logger.entering("ReplicationDialogFragmentManager", "getDialogFragmentStore");
        }
        if (dialogFragmentStorePool == null) {
            // Replication service not initialized yet.
            return null;
        }
        DialogFragmentStoreImpl store = null;
        try {
            store = (DialogFragmentStoreImpl) dialogFragmentStorePool.take();
            if(_logger.isLoggable(Level.FINEST)) {
                _logger.log(Level.FINEST,
                    "ReplicationDialogFragmentManager.getDialogFragmentStore returning   " + store);
            }
            return store;
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "getDialogFragmentStore", store);
        }
        return store;
    } 
    
    /** 
     *  Returns (puts) a store back to the pool
     */
    private void putDialogFragmentStore(StorePoolElement store) {    
        //( (DialogFragmentStoreImpl) store).setSipSessionManager(null);
        if (store != null && dialogFragmentStorePool != null) {
            try {
                dialogFragmentStorePool.put(store);
            } catch (InterruptedException ex1) {
                //FIXME: log this
                ex1.printStackTrace();
            }
        }
    }
    
    /** Returns the singleton store 
     * This method intializes the store with right parameters
     * @return returns DialogFragmentStoreImpl
     */
    public DialogFragmentStoreImpl getSingletonDialogFragmentStore() { 
        return (DialogFragmentStoreImpl)getDialogFragmentStore(); 
    }
    
    //begin send-side methods
    
    /**
     * Persists the given DialogFragment.
     *
     * @param dialogFragment The DialogFragment to be persisted
     */
    public void saveDialogFragment(DialogFragment dialogFragment)
            throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager", "saveDialogFragment");
        }        
        DialogFragmentStoreImpl store = null;
        try {
            store = getDialogFragmentStore();            
            store.save(dialogFragment);
        }
        finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "saveDialogFragment");
        }
    }
    
    /**
     * Removes the given DialogFragment from both the active cache
     * and the persistent session store of this session manager,
     *
     * @param dialogFragment The DialogFragment to be removed
     */
    public void removeDialogFragment(DialogFragment dialogFragment) {
        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager", "removeDialogFragment");
        }
        super.removeDialogFragmentFromCache(dialogFragment, false);
        DialogFragmentStoreImpl store = null;
        try {
            store = getDialogFragmentStore();   
            store.remove(dialogFragment.getDialogId());
        } catch (IOException ex) {
            _logger.log(
                Level.WARNING,
                "unable to remove DialogFragment:id = " + dialogFragment.getDialogId(),
                ex);
        } finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "removeDialogFragment");
        }
    } 
    
    /**
     * Look for a DialogFragment in the Store and, if found, restore
     * it in the Manager's list of active dialog fragments if appropriate.
     * The dialog fragment will be removed from the Store after swapping
     * in, but will not be added to the active dialog fragment list if it
     * is invalid or past its expiration.
     * @param id the id of the DialogFragment
     */
    protected DialogFragment swapInDialogFragment(String id) throws IOException {
        return swapInDialogFragment(id, null);
    }    
    
    /**
     * Look for a DialogFragment in the Store and, if found, restore
     * it in the Manager's list of active dialog fragments if appropriate.
     * The dialog fragment will be removed from the Store after swapping
     * in, but will not be added to the active dialog fragment list if it
     * is invalid or past its expiration.
     * @param id the id of the Dialog Fragment
     * @param version - the version of the Dialog Fragment
     */
    protected DialogFragment swapInDialogFragment(String id, String version) throws IOException {
        DialogFragmentStoreImpl store = null;
        try {
            store = this.getDialogFragmentStore();
            if (store == null)
                return null;

            DialogFragment dialogFragment = null;
            try {
                if (SecurityUtil.isPackageProtectionEnabled()){
                    try{
                        dialogFragment = (DialogFragment) AccessController.doPrivileged(new PrivilegedStoreLoadDialogFragment(id, version, store));
                    }catch(PrivilegedActionException ex){
                        Exception exception = ex.getException();
                        if (exception instanceof IOException){
                            throw (IOException)exception;
                        } else if (exception instanceof ClassNotFoundException) {
                            throw (ClassNotFoundException)exception;
                        }
                    }
                } else {
                    if (version != null) {
                         dialogFragment = store.load(id, version);
                    } else {
                         dialogFragment = store.load(id);
                    }
                }   
            } catch (ClassNotFoundException e) {
                IOException ex1 = 
                    (IOException) new IOException("Error during swapInSipApplicationSession: " + e.getMessage()).initCause(e);
                throw ex1;                 
            }

            if (dialogFragment == null)
                return (null); 
            registerDialogFragment(dialogFragment);
            return (dialogFragment);
        } finally {
            this.putDialogFragmentStore(store);
        }
    }        
    
    //start receiver side process methods
    
    //DialogFragment process methods

    /**
     * process the save of a DialogFragment
     * @param dialogFragmentState - contains the save command
     */     
    public void processSavedialogfragment(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN" + this.getClass().getName() + ">>processSavedialogfragment");
        }        
        this.putInDialogFragmentReplicationCache(dialogFragmentState);
    }    

    /**
     * process the removal of a DialogFragment
     * @param dialogFragmentState - contains the remove command
     */     
    public void processRemovedialogfragment(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN" + this.getClass().getName() + ">>processRemovedialogfragment");
        } 
        if(dialogFragmentState == null) {
            return;
        }
        replicatedDialogFragments.remove(dialogFragmentState.getId());
        removeFromDialogFragmentReplicationUpdateCache((String)dialogFragmentState.getId());      
    }

    /**
     * process the update of a DialogFragment
     * @param dialogFragmentState - contains the updatelastaccesstime command
     */    
    public void processUpdatelastaccesstimedialogfragment(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN" + this.getClass().getName() + ">>processUpdatelastaccesstimedialogfragment");
            _logger.fine("processUpdatelastaccesstimedialogfragment:version:" + dialogFragmentState.getVersion());
        }
        String id = (String)dialogFragmentState.getId();
        ReplicationState state = this.getFromDialogFragmentReplicationCache(id);
        if(state != null) {
            state.setLastAccess(dialogFragmentState.getLastAccess());
            state.setVersion(dialogFragmentState.getVersion());
            state.setContainerExtraParamsState(dialogFragmentState.getContainerExtraParamsState());
            this.putInDialogFragmentReplicationCache(state);
        } else {
            if(_logger.isLoggable(Level.FINE)) {                
                _logger.fine("processUpdatelastaccesstimedialogfragment: attempting to update a dialogframgnet not yet stored:id:" + dialogFragmentState.getId());
            }
            this.putInDialogFragmentReplicationUpdateCache(new ReplicationStateUpdate(id, dialogFragmentState.getVersion(), dialogFragmentState.getLastAccess()) );
        }        
    }
    
    /**
     * process the broadcastfinddialogfragment for DialogFragment
     * @param queryState
     */     
    public ReplicationState processBroadcastfinddialogfragment(ReplicationState queryState) {
        //complete query and send back response
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragment:instance: " + instanceName);
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragment:id=" + queryState.getId());                        
        } 
        
        //look in active cache
        HADialogFragment haDialogFragment = null;
        DialogFragment dialogFragment 
            = findDialogFragmentFromCacheOnly((String)queryState.getId());
        if(dialogFragment instanceof HADialogFragment) {
            haDialogFragment = (HADialogFragment)dialogFragment;
        }        
        
        //look in replica cache
        ReplicationState replicaState
            = getFromDialogFragmentReplicationCache((String)queryState.getId());
        ReplicationState returnState 
            = getBestDialogFragment(haDialogFragment, replicaState, queryState);          
        clearFromDialogFragmentManagerCache((String)queryState.getId(), false);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragment:returnState=" + returnState);
        }
        return returnState;
    }
    
    public void processBroadcastloadreceiveddialogfragment(ReplicationState queryState) {
        //load is acknowledged safe to remove replica now
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processbroadcastloadreceiveddialogfragment:instance: " + instanceName);
            _logger.fine("in " + this.getClass().getName() + ">>processbroadcastloadreceiveddialogfragment:id=" + queryState.getId());                        
        }
        if(queryState == null || queryState.getId() == null) {
            return;
        }
        String id = (String)queryState.getId();
        //remove active dialogfragment if present and not locked
        DialogFragment dialogFragment 
            = findDialogFragmentFromCacheOnly(id);
        if(dialogFragment != null 
                && !dialogFragment.isForegroundLocked()) {
            this.removeDialogFragmentFromCache(dialogFragment, false);
        } 
        //only safe to remove replica if we are not the replica partner
        //of the sending instance
        ReplicationHealthChecker healthChecker 
            = ReplicationHealthChecker.getInstance();
        String replicatedFromInstanceName 
            = healthChecker.getReshapeReplicateFromInstanceName(); 
        if(replicatedFromInstanceName != null && !replicatedFromInstanceName.equalsIgnoreCase(queryState.getInstanceName())) {           
            removeFromDialogFragmentReplicationCache(id);           
        }
    }   
    
    /**
     * find the best version of DialogFragment
     * removing any stale versions and return query result
     * @param activeDialogFragment DialogFragment 
     *      from active cache
     * @param replicaDialogFragment DialogFragment 
     *      from replica cache
     * @param queryVersion version requested in query (-1 means 
     *      version unaware
     */     
    private ReplicationState getBestDialogFragment(HADialogFragment activeDialogFragment, 
        ReplicationState replicaDialogFragment, ReplicationState queryState) {
        ReplicationState bestResult = null;
        long queryVersion = queryState.getVersion();
        
        //first check for none found in either active or replica caches
        if(replicaDialogFragment == null && activeDialogFragment == null) {
            //return nack
            return ReplicationState.createQueryResponseFrom(queryState, true);
        }
        
        //next check for artifacts found in both active and replica caches
        if(replicaDialogFragment != null && activeDialogFragment != null) { 
            //compare and remove the lesser version
            //keeping the higher version as (tentative) best
            if(replicaDialogFragment.getVersion() < activeDialogFragment.getVersion()) {
                //remove stale replica - work with active
                removeFromDialogFragmentReplicationCache((String)replicaDialogFragment.getId());
                //create appropriate response from active
                bestResult 
                    = createDialogFragmentResponseFrom(activeDialogFragment, queryState);
            } else {
                //remove stale active - work with replica
                clearFromDialogFragmentManagerCache(activeDialogFragment.getDialogId(), false);               
                //create appropriate response from replica
                bestResult 
                    = createDialogFragmentResponseFrom(replicaDialogFragment, queryState);                                
            }
        } else {
            //either replica or active is null and other is non-null
            //replica is null and active is not null
            if(replicaDialogFragment == null) {                
                //create appropriate response from active
                bestResult 
                    = createDialogFragmentResponseFrom(activeDialogFragment, queryState);                               
            } else {
                //active is null & replica is not null
                //create appropriate response from replica
                bestResult 
                    = createDialogFragmentResponseFrom(replicaDialogFragment, queryState);                                 
            }
        }
        return bestResult;
    }
    
    private ReplicationState createDialogFragmentResponseFrom(ReplicationState replicaDialogFragment, 
            ReplicationState queryState) {        
        //create appropriate response from replica
        ReplicationState result = null;
        long queryVersion = queryState.getVersion();        
        if(queryVersion != -1 && replicaDialogFragment.getVersion() < queryVersion) {
            //return nack & clear stale replica
            removeFromDialogFragmentReplicationCache((String)replicaDialogFragment.getId());
            result = ReplicationState.createQueryResponseFrom(queryState, true);
        } else {
            //return real response based on replica
            result = ReplicationState.createQueryResponseFrom(replicaDialogFragment); 
        } 
        return result;
    }

    private ReplicationState createDialogFragmentResponseFrom(HADialogFragment activeDialogFragment, 
            ReplicationState queryState) {     
        //create appropriate response from active
        ReplicationState result = null;
        long queryVersion = queryState.getVersion();        
        if(queryVersion != -1 && activeDialogFragment.getVersion() < queryVersion) {
            //return nack & clear stale active
            clearFromDialogFragmentManagerCache(activeDialogFragment.getDialogId(), false);
            result = ReplicationState.createQueryResponseFrom(queryState, true);
        } else {
            //return real response based on active DF
            try {
                result = createQueryResponseFrom(activeDialogFragment);
            } catch (IOException ioe) {
                _logger.warning("Failed load: Unable to serialize " +
                                activeDialogFragment);
                // We've been unable to serialize the given active
                // DialogFragment.
                // Clear it from the active cache and return a nack instead
                clearFromDialogFragmentManagerCache(
                    activeDialogFragment.getDialogId(), false);
                result = ReplicationState.createQueryResponseFrom(queryState,
                                                                  true);
            }
        }

        return result;
    }
    
    /**
     * Converts the given DialogFragment to a ReplicationState.
     *
     * @param DF The DialogFragment to be converted
     *
     * @return The ReplicationState corresponding to the given 
     * DialogFragment
     */ 
    private ReplicationState createQueryResponseFrom(
            HADialogFragment df) throws IOException {
        byte[] containerExtraParamState = null;
        DialogFragmentExtraParams containerExtraParams 
            = df.getExtraParameters();
        if(containerExtraParams != null) {
            try {
                containerExtraParamState
                    = ReplicationUtil.getByteArray(containerExtraParams);
            } catch (IOException ex) {
                ; //deliberate no-op
            }
        }         
        return new ReplicationState(
                MODE_SIP,
                df.getDialogId(),
                getApplicationId(),  
                df.getVersion(),
                0L,
                0L,
                null,
                null,
                null,
                ReplicationState.RETURN_BROADCAST_MSG_COMMAND,
                ReplicationUtil.getByteArray(df), 
                null,
                containerExtraParamState);
    }

    protected void clearFromDialogFragmentManagerCache(String id, boolean hasTimedOut) {
        DialogFragment df = findDialogFragmentFromCacheOnly(id);
        if(df != null) {                               
            this.removeDialogFragmentFromCache(df, hasTimedOut);
        } 
    }
    
    //end receiver side process methods
    
    //start expiration related methods
    
    void processExpiredReplicas() {
        //FIXME needs optimization
        //both scale of task and coupling of expiration logic
        processExpiredDialogFragmentReplicas();
    }
    
    void processExpiredDialogFragmentReplicas() {
        //FIXME 
        //code assumes that isExpired works for this type of ReplicationState
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processExpiredDialogFragmentReplicas");            
        }
        ArrayList expiredReplicas = new ArrayList(30);
        BaseCache replicasCache = this.getReplicatedDialogFragments();
        for(Iterator it = replicasCache.values(); it.hasNext();) {
            ReplicationState nextState = (ReplicationState)it.next();
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("in " + this.getClass().getName() + "nextState=" + nextState);            
            } 
            if(isExpiredDialogFragmentReplica(nextState)) {
                expiredReplicas.add(nextState);
            }
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processExpiredDialogFragmentReplicas:expiredReplicas.size=" + expiredReplicas.size());            
        }        
        for(int i=0; i<expiredReplicas.size(); i++) {
            ReplicationState nextState = 
                (ReplicationState)expiredReplicas.get(i);
            this.removeFromDialogFragmentReplicationCache((String)nextState.getId());
        }
    }
    
    private DialogFragmentExtraParams getDialogFragmentExtraParamsFrom(ReplicationState state) {
        if(state == null) {
            return null;
        }
        if(state.getDeserializedExtraParam() != null) {
            return (DialogFragmentExtraParams)state.getDeserializedExtraParam();
        } else {
            //deserialize and cache result in state
            byte[] extraParamsState = state.getContainerExtraParamsState();
            DialogFragmentExtraParams extraParams = null;
            if(extraParamsState != null) {
                try {
                    extraParams
                        = (DialogFragmentExtraParams)ReplicationState.getObjectValue(extraParamsState);
                } catch (Exception ex) {
                    _logger.warning("unable to deserialize DialogFragment extraParams for id " + state.getId() + ":" + ex);
                }
                state.setDeserializedExtraParam(extraParams);
            }            
            return extraParams;
        }
    }     
    
    protected boolean isExpiredDialogFragmentReplica(ReplicationState state) {
        DialogFragmentExtraParams extraParams
            = getDialogFragmentExtraParamsFrom(state);         
        if(extraParams == null) {
            return false;
        } else {
            return (extraParams.getExpirationTime() < System.currentTimeMillis());
        }
    }    
   
    //begin implement ReplicationManager
    
    private boolean isBroadcastState(ReplicationState state) {
        return(broadcastMethods.contains(state.getCommand()));
    }
                
    public void processMessage(ReplicationState message) {
        //handle broadcast methods
        if(isBroadcastState(message)) {
            processBroadcastMessage(message);
            return;
        }
        
        //handle non-void methods
        ReplicationStateQueryResponse queryResult = null;
        //do process non-void message (and cannot be response either)
        if(!message.isResponseState() && !message.isVoidMethodReturnState()) {
            //do non-void processing including sending response
            queryResult = this.doProcessQueryMessage(message);
            ReplicationState qResponse = queryResult.getState();
            if(qResponse != null) {
                //sourceInstanceName is preserved in the response
                ReplicationState response = 
                    ReplicationState.createResponseFrom(qResponse);
                if(_logger.isLoggable(Level.FINE)) {
                    _logger.fine("RepMgrBase:responseState=" + response);
                }                
                this.doSendResponse(response);
            }
            return;
        }
        //end do process non-void message        

        boolean isResponse = this.doProcessMessage(message);
        
    }
    
    //return true if message is processResponse
    public boolean doProcessMessage(ReplicationState message) {
        boolean result = false;
        String methodName = ReplicationUtil.getProcessMethodName(message);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>doProcessMessageName=" + methodName);
        }
        try {
            Class myClass = this.getClass();
            myClass.getMethod(
                methodName,
                    new Class[]{ message.getClass() }).invoke(
                        this, new Object[]{ message });            
        } catch (Throwable t) {
           _logger.log(
               Level.SEVERE,
               "Unable to process replication message with methodName=" +
               methodName, t);
        } 
        if(methodName.equals("processResponse")) {
            result = true;
        }
        return result;
    }    
    
    /**
    * send the response
    *
    * @param dialogFragmentState 
    *   The replication state response
    */    
    public void doSendResponse(ReplicationState dialogFragmentState) {
               
        JxtaReplicationReceiver replicationReceiver = 
                JxtaReplicationReceiver.createInstance();
        ReplicationState resultState = 
                replicationReceiver.sendReplicationStateResponse(dialogFragmentState); 
    }        
    
    public void processBroadcastMessage(ReplicationState message) {
        ReplicationStateQueryResponse response = this.doProcessQueryMessage(message);
        boolean isResponse = response.isResponse();
        ReplicationState responseState = response.getState();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:response=" + isResponse);
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseState=" + responseState);
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseStateTrunk=" + responseState.getTrunkState());
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseStateAttr=" + responseState.getState());
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseStateVer=" + responseState.getVersion());
        }

        //don't send a response to a response
        if(!isResponse) {
            //point-to-point response back to sender
            //System.out.println("processBroadcastMessage - need to send back result");
            doSendQueryResponse(responseState, message.getInstanceName());
        }
    }
    
    /**
    * send the response
    *
    * @param dialogFragmentState 
    *   The replication state response
    * @param instanceName  the name of the target instance
    */    
    public void doSendQueryResponse(ReplicationState dialogFragmentState, String instanceName) {
        JxtaReplicationSender replicationSender = 
                JxtaReplicationSender.createInstance();
        ReplicationState resultState = 
                replicationSender.sendReplicationStateQueryResponse(dialogFragmentState, instanceName); 
    }        
    
    //return true if message is processQueryResponse
    public ReplicationStateQueryResponse doProcessQueryMessage(ReplicationState message) {
        ReplicationState resultState = null;
        String methodName = ReplicationUtil.getProcessMethodName(message);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>doProcessQueryMessage:methodName=" + methodName);
            _logger.fine("in " + this.getClass().getName() + ">>doProcessQueryMessage:thisInstance=" + instanceName + "SASEreturnInstance=" + message.getInstanceName() );
        }         
        try {
            Class myClass = this.getClass();
            resultState = (ReplicationState) myClass.getMethod(
                methodName,
                    new Class[]{ message.getClass() }).invoke(
                        this, new Object[]{ message });            
        } catch (Throwable t) {
           _logger.log(
               Level.SEVERE,
               "Unable to process replication message with methodName=" +
               methodName, t);
        } 
        boolean isResponse = methodName.equals("processBroadcastresponse");
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>doProcessQueryMessage:resultState=" + resultState);
        }        
        //System.out.println("in " + this.getClass().getName() + ">>doProcessQueryMessage:resultState=" + resultState);
        return new ReplicationStateQueryResponse(resultState, isResponse);
    }
    
    public void processQueryMessage(ReplicationState message, String returnInstance) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processQueryMessage:returnInstance= " + returnInstance);
        }         
        ReplicationStateQueryResponse response = this.doProcessQueryMessage(message);
        boolean isResponse = response.isResponse();
        ReplicationState responseState = response.getState();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processQueryMessage:after doProcessQueryMessage:response=" + isResponse);
            _logger.fine("processQueryMessage:after doProcessQueryMessage:responseState=" + responseState);            
        }        
        //don't send a response to a response
        if(!isResponse && responseState != null) {
            //point-to-point response back to sender
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("processQueryMessage - need to send back result to " + returnInstance);            
            }             
            doSendQueryResponse(responseState, returnInstance);
        }
    } 
    
    public ReplicationState processBroadcastresponse(ReplicationState queryResponseState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastresponse:queryResponseState=" + queryResponseState);            
        }
        ReplicationResponseRepository.putFederatedEntry(queryResponseState);
        return queryResponseState;
    }    
    
    public void repair(long repairStartTime) {
       if (!isRepairDuringFailure()) {
           return;
       }        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("ReplicationDialogFragmentManager>>repair");                       
        }         

        if(ReplicationHealthChecker.isStopping()) {
            return;
        }

        Iterator it = dialogFragments.values().iterator();
        while (it.hasNext()) {
            DialogFragment fragment = (DialogFragment) it.next();
            //by virtue of being in the cache it is considered to 
            //but since invalidate is called before removal we must check            
            if(fragment.isValid()
                && (fragment.getDialogId() != null)
                && isDialogFragmentOlderThan(fragment, repairStartTime)) {
                if(fragment.lockBackground()) { 
                    try {
                        ((HADialogFragment)fragment).setReplicated(false);
                        ((HADialogFragment)fragment).setDirty(true);
                        saveDialogFragment(fragment);
                    } catch (IOException ex) {
                        _logger.warning("during repair unable to save DialogFragment:id = " + fragment.getDialogId() + ":" + ex);
                    } finally {
                        fragment.unlockBackground();
                    }
                }                                
	    }            
        }
    } 
    
    protected boolean isDialogFragmentOlderThan(DialogFragment fragment, long aTime) {
        // FIXME Uncomment when getLastAccessedTime is implemented.
        // return (fragment.getLastAccessedTime() <= aTime); 
        return false;
    }    
    
    public void repair(long repairStartTime, boolean checkForStopping) {
       if (!isRepairDuringFailure()) {
           return;
       }         
        //FIXME
    }
    
    public void purge(String owningInstanceName, long purgeStartTime) {
        List idsToRemove = new ArrayList();
        Iterator it = replicatedDialogFragments.values();
        while(it.hasNext()) {
            ReplicationState nextState 
                = (ReplicationState)it.next();
            byte[] extraParamsState
                = nextState.getContainerExtraParamsState();
            DialogFragmentExtraParams extraParams = null;
            if(extraParamsState != null) {
                try {
                extraParams
                    = (DialogFragmentExtraParams)ReplicationState.getObjectValue(extraParamsState);
                } catch (Exception ex) {
                    _logger.warning("unable to deserialize DialogFragment extraParams for id " + nextState.getId() + ":" + ex);
                }
            }
            if(extraParams != null) {
                String nextOwningInstance = extraParams.getCurrentOwnerInstanceName();
                if(nextOwningInstance != null 
                    && nextOwningInstance.equalsIgnoreCase(owningInstanceName)) {
                    idsToRemove.add((String)nextState.getId());
                }
            }
        }
        for(int i=0; i< idsToRemove.size(); i++) {
            removeFromDialogFragmentReplicationCache((String)idsToRemove.get(i));
        }
    }     

    public void respondToFailure(String failedInstanceName, boolean checkForStopping) {
        //FIXME Is there something specific to DialogFragments needed to be done here
        //     Following comment was copied from ReplicationDialogFragmentManager, doubt it applies to DialogFragment
        // only do timer wakeup if failedInstanceName was replication to us
        ReplicationHealthChecker healthChecker 
            = ReplicationHealthChecker.getInstance();
        String replicatedFromInstanceName 
            = healthChecker.getReshapeReplicateFromInstanceName();
        if(!failedInstanceName.equalsIgnoreCase(replicatedFromInstanceName)) {
            return;
        }
    }
  
    private void setDialogFragmentStorePool(StorePool dialogFragmentStorePool) {
        this.dialogFragmentStorePool = dialogFragmentStorePool;
    }

    public boolean isReplicationEnabled() {
        return isReplicationInitialized.get() && isReplicationStarted.get();
    }

    boolean isThirdPartyBackingStoreInUse() {
        BackingStore dialogFragmentBackingStore 
            = getDialogFragmentBackingStore();
        return (!(dialogFragmentBackingStore instanceof JxtaBackingStoreImpl));
    }
         
    private class ReplicationStateUpdate {
        private String _id = null;
        private long _lastAccess = 0L;
        private long _version = -1L;
        
        String getId() {
            return _id;
        }        
        
        long getLastAccess() {
            return _lastAccess;
        }
        
        long getVersion() {
            return _version;
        }       
        
        public ReplicationStateUpdate(String id, long version, long lastAccess) {
            _id = id;
            _version = version;
            _lastAccess = lastAccess;
        }
        
    } 
    
    private class PrivilegedStoreLoadDialogFragment
        implements PrivilegedExceptionAction {

        private String id;
        private String version;
        private DialogFragmentStoreImpl store;
            
        PrivilegedStoreLoadDialogFragment(String id, String version, DialogFragmentStoreImpl store) {     
            this.id = id;
            this.version = version;
            this.store = store;
        }

        public Object run() throws Exception{
           return store.load(id, version);
        }                       
    }
}
