/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import java.util.concurrent.ConcurrentHashMap;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ericsson.ssa.sip.SipApplicationSessionUtil;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import org.jvnet.glassfish.comms.replication.sessmgmt.ServletTimerExtraParams;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * A class that handles restoration and migration of ServletTimers and
 * SipApplicationSessions in response to their owning instance failing. 
 *
 * The entry point is registerServletTimerExtraParam(), which indicates that
 *  the specified timer must be restored wither on this instance or on a different
 *  instance. The call to registerServletTimerExtraParam() results in a creation of
 *  a JDK Timer task that contains enough information to restore / recreate the
 *  ServletTimer either on this instance or on a different instance. The timer task
 *  itself has a periodicity that is controlled by SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS.
 *  The only reason to execute the TimerTask periodically (untill it is cancelled), is
 *  to ensure that the ServletTimer is restoredf eventually in some instance.
 *
 * Also, in order to smooth out this restoration process, the initial delay of the
 *  TimerTasks is calculated as follows:
 *
 *  1. If the difference between "now" and the ServletTimer's nextExpirationTime is below
 *      SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS, then the TimerTask is scheduled to execute
 *      immediately (0 delay).
 *
 *  2. If the difference between "now" and the ServletTimer's nextExpirationTime is below
 *      SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS (but greater than SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS),
 *      then the TimerTask is scheduled to execute <b>AFTER</b> SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS
 *      plus some random time (that is between 0 and SERVLET_TIMER_RANDOM_MAX_IN_MILLIS).
 *
 *  In all cases, the TimerTask executes with a periodicity of SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS.
 *      When the TimerTask's run() method is called, the following is done:
 *      a) A check is made to determine the "actual" owner that must "own" this ServletTimer. This is
 *          done by calling DataCentricUtil.getInstanceNameFromConsistentHash()
 *      b) if the above checks indicates that this instance must "own" the ServletTimer, then
 *          processLoadAdvisoryServletTimer(eParam) is called to restore the ServletTimer in this
 *          instance. This method is assumed to eventually call unregisterServletTimerExtraParam
 *      c) if the check in (a) indicates that some other instance must "own" the ServletTimer, then
 *          sendLoadAdvisoryServletTimer(eParam) is called to instruct the other instance to "own"
 *          the ServletTimer. The other instance will load the ServletTimer and will eventually
 *          call unregisterServletTimerExtraParam
 *
 *  Eventually, once the ServletTimer is restored (either on this instance or on another instance),
 *      the call to unregisterServletTimerExtraParam will cause the TimerTask to be cancelled.
 *
*/
public class ActivationHelper {

    private SipTransactionPersistentManager sipTxMgr;
 
    private static final Logger _logger = Logger.getLogger(LogUtil.SIP_LOG_DOMAIN);

    private static final Level TRACE_LEVEL = Level.FINE;

    private Random random = new Random(System.currentTimeMillis());

    /**
     * ServletTimers that need to execute AFTER this threshold will result
     *  in creation of a TimerTask that will be scheduled this value plus
     *  a random time between zero and SERVLET_TIMER_RAMDOM_MAX_IN_MILLIS.
     */
    private static final long SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS = 5 * 60 * 60 * 1000; //5 hours

    /**
     * @See SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS.
     */
    private static final long SERVLET_TIMER_RANDOM_MAX_IN_MILLIS = 1 * 60 * 60 * 1000; //1 hour

    private static final long SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS = 2 * 60 * 1000; // 2 minutes

    /**
     * ServletTimers that need to execute before this threshold will result
     *  in creation of a TimerTask that will be scheduled immediately.
     */
    private static final long SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS = 6 * 60 * 1000; //6 minutes

    /**
     * Our cache of TimerTasks for which ServletTimers haven't been created.
     * keyed by TimerId
     */
    private ConcurrentHashMap<String, TimerMigrationTask> timerTaskCache =
        new ConcurrentHashMap<String, TimerMigrationTask>();
    
    private Timer migrationTimer = new Timer(true);

    /**
     * Constructor
     */
    ActivationHelper(SipTransactionPersistentManager sipTxMgr) {
        this.sipTxMgr = sipTxMgr;
    }

    public void registerServletTimerExtraParam(ServletTimerExtraParams eParam) {
        long timerScheduleTime = calculateServletTimerTaskScheduleTime(eParam);
        TimerMigrationTask task = new TimerMigrationTask(eParam);
        timerTaskCache.put(eParam.getId(), task);
        migrationTimer.scheduleAtFixedRate(task, timerScheduleTime,
            SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS);
    }   

    public boolean hasTimerTask(String timerId) {
        return timerTaskCache.contains(timerId);
    }

    public void unregisterServletTimerExtraParam(String timerId) {
        TimerMigrationTask task = timerTaskCache.remove(timerId);
        if (task != null) {
            task.cancel();
        }
    }

    long calculateServletTimerTaskScheduleTime(ServletTimerExtraParams eParam) {
        return this.calculateServletTimerTaskScheduleTime(eParam.getExpirationTime());
    }
    
    long calculateServletTimerTaskScheduleTime(long actualExpTime) {
        long now = System.currentTimeMillis();
        long timeDiff = actualExpTime - now;

        if (actualExpTime <= 0) {
            timeDiff = SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS + 1;
        }

        long scheduleDelay = 0;
        if (timeDiff <= SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS) {
            scheduleDelay = 0;
        } else if (timeDiff <= SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS) {
            //FIXME: Revisit to add some randomness
            scheduleDelay = timeDiff;
        } else {
            scheduleDelay = Math.abs(random.nextLong()) % SERVLET_TIMER_RANDOM_MAX_IN_MILLIS;
            scheduleDelay = SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS - scheduleDelay;
        }

        return scheduleDelay;
    }    

    private class TimerMigrationTask 
        extends TimerTask {

        private int runCount = 0;

        private ServletTimerExtraParams eParam;

        TimerMigrationTask(ServletTimerExtraParams extraParam) {
            this.eParam = extraParam;
        }

        public void run() {
            try {
                String myInstanceName = ReplicationUtil.getInstanceName();
                String timerId = eParam.getId();
                String actualOwnerName = 
                    SipApplicationSessionUtil.getActualServerInstance(
                        eParam.getParentSasId());
                if (myInstanceName.equals(actualOwnerName)) {
                    if (runCount > 0) {
                        _logger.log(TRACE_LEVEL, "Executing processLoadAdvisoryServletTimer() (for " + runCount + " time)");
                    }

                    //Note: processLoadAdvisoryServletTimer must eventually
                    //  call unregisterServletTimerExtraParam()
                    sipTxMgr.processLoadadvisoryservlettimer(eParam);
                } else {
                    if (runCount > 0) {
                        _logger.log(TRACE_LEVEL, "Sending sendLoadAdvisoryServletTimer() (for " + runCount + " time) to " + actualOwnerName);
                    }

                    //Note: The acutualOwner must eventually send an ACK that
                    //  must result in a call to unregisterServletTimerExtraParam()
                    sipTxMgr.sendLoadAdvisoryServletTimer(timerId, actualOwnerName);
                }
            } catch (Exception ex) {
                _logger.log(TRACE_LEVEL, "Got exception", ex);
            } finally {
                runCount++;
            }
        }
    }

}
