/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.config;

import com.ericsson.ssa.dd.ConvergedDescriptor;
import com.ericsson.ssa.dd.SipApplication;

import com.sun.enterprise.config.serverbeans.SipContainerAvailability;
import com.sun.enterprise.deployment.WebBundleDescriptor;
import com.sun.enterprise.deployment.runtime.web.SessionManager;
import com.sun.enterprise.web.Constants;
import com.sun.enterprise.web.SessionManagerConfigurationHelper;
import com.sun.enterprise.web.WebModule;
import com.sun.enterprise.web.WebModuleConfig;
import com.sun.enterprise.web.session.PersistenceType;

import org.jvnet.glassfish.comms.deployment.backend.SipBundleDescriptor;
import org.jvnet.glassfish.comms.deployment.backend.SipApplicationBase;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * Helper class for determining the persistence type, frequency, and scope
 * for the HTTP and SIP components of a converged application.
 *
 * @author jluehe
 */
public class ConvergedSessionManagerConfigurationHelper
    extends SessionManagerConfigurationHelper {
    private static final LogUtil SIP_LOGGER = new LogUtil(LogUtil.SIP_LOG_DOMAIN);
    private PersistenceType sipPersistenceType = PersistenceType.MEMORY;
    private String sipPersistenceFrequency = null;
    private String sipPersistenceScope = null;
    private ConvergedServerConfigLookup lookup = null;
    private SessionManager sipSmBean = null;

    /**
     * Constructor.
     */
    public ConvergedSessionManagerConfigurationHelper(WebModule ctx,
        SessionManager httpSmBean, SessionManager sipSmBean,
        WebBundleDescriptor wbd, WebModuleConfig wmInfo) {
        super(ctx, httpSmBean, wbd, wmInfo);
        lookup = new ConvergedServerConfigLookup();
        this.sipSmBean = sipSmBean;
    }

    /**
     * Determines the persistence type, frequency, and scope for the HTTP and
     * SIP components of the converged application that was passed to the
     * constructor, taking into account configuration information from both
     * domain.xml and the application's deployment descriptors.
     *
     * This method also ensures that if in-memory replication has been turned
     * on for SIP, it must also have been turned on for HTTP, and vice versa.
     * Otherwise, we fall back to memory-only for both SIP and HTTP.
     */
    public void initializeConfiguration() {
        // Determine persistence configuration for HTTP
        super.initializeConfiguration();

        // Determine persistence configuration for SIP
        boolean isSipAppDistributable = false;
        SipBundleDescriptor sbd = null;
        if (_wbd instanceof SipBundleDescriptor) {
            sbd = (SipBundleDescriptor) _wbd;
            SipApplicationBase sipApp = sbd.getSipApplication();

            if (sipApp != null) {
                isSipAppDistributable = ((SipApplication)sipApp).isDistributable();
            }
        }

        PersistenceType persistenceType = PersistenceType.MEMORY;
        String persistenceFrequency = null;
        String persistenceScope = null;

        boolean isAvailabilityEnabled = lookup.calculateSipAvailabilityEnabledFromConfig(_ctx);

        if (isAvailabilityEnabled) {
            // These are the global defaults if nothing is set at
            // domain.xml or sun-sip.xml
            persistenceType = PersistenceType.REPLICATED;
            persistenceFrequency = "sip-transaction";
            persistenceScope = "session";
        }

        // If domain.xml default exists, then use that
        PersistenceType domainPersistenceType = lookup.getSipPersistenceTypeFromConfig();

        if (domainPersistenceType != null) {
            persistenceType = domainPersistenceType;
            persistenceFrequency = lookup.getSipPersistenceFrequencyFromConfig();
            persistenceScope = lookup.getSipPersistenceScopeFromConfig();
        }

        // Consider persistence type, scope, and frequency from sun-sip.xml
        if (sipSmBean != null) {
            String pType = sipSmBean.getAttributeValue(
                SessionManager.PERSISTENCE_TYPE);
            persistenceType = PersistenceType.parseType(pType, persistenceType);

            String sipAppLevelPersistenceFrequency =
                getPersistenceFrequency(sipSmBean);
            if(sipAppLevelPersistenceFrequency != null) {
                persistenceFrequency = sipAppLevelPersistenceFrequency;
            }

            String sipAppLevelPersistenceScope =
                getPersistenceScope(sipSmBean);
            if(sipAppLevelPersistenceScope != null) {
                persistenceScope = sipAppLevelPersistenceScope;
            }
        }
        
        String frequency = null;
        String scope = null;

        // Deliberately leave frequency & scope null in memory-only case
        if (persistenceType != PersistenceType.MEMORY) {
            frequency = persistenceFrequency;
            scope = persistenceScope;
        }

        // If sip app is not distributable, and non-memory option
        // is attempted, log warning and set back to memory-only
        if (!isSipAppDistributable &&
                (persistenceType != PersistenceType.MEMORY)) {
            String sipAppName = getApplicationId(_ctx);

            // Suppress log warning for default-web-module
            // Log message only if availabilityenabled = true is attempted
            if (isAvailabilityEnabled &&
                    !sipAppName.equals(Constants.DEFAULT_WEB_MODULE_NAME) &&
                    !isSystemApp(sipAppName)) {
                SIP_LOGGER.warning("sipstack.invalid_session_manager_config",
                                   sipAppName, persistenceType.getType(),
                                   frequency, scope);
            }

            // Fall back to memory option
            persistenceType = PersistenceType.MEMORY;
            frequency = null;
            scope = null;
        }

        sipPersistenceType = persistenceType;
        sipPersistenceFrequency = frequency;
        sipPersistenceScope = scope;

        // Make sure that if the app is converged (i.e., contains both
        // a web.xml and sip.xml), and in-memory replication has been turned
        // on for SIP, it must also have been turned on for HTTP, and vice
        // versa.
        // Otherwise, we fall back to memory-only for both SIP and HTTP.
        if (sbd != null &&
                sbd.isConverged() &&
                (((sipPersistenceType == PersistenceType.REPLICATED) &&
                (_persistence != PersistenceType.REPLICATED)) ||
                ((sipPersistenceType != PersistenceType.REPLICATED) &&
                (_persistence == PersistenceType.REPLICATED)))) {
            SIP_LOGGER.warning("sipstack.inconsistent_persistence_type",
                               getApplicationId(_ctx));

            // Reset SIP persistence to memory-only
            sipPersistenceType = PersistenceType.MEMORY;
            sipPersistenceFrequency = null;
            sipPersistenceScope = null;

            // Reset HTTP persistence to memory-only
            _persistence = PersistenceType.MEMORY;
            _persistenceFrequency = null;
            _persistenceScope = null;
        }
    }

    public PersistenceType getSipPersistenceType() {
        checkInitialization();

        return sipPersistenceType;
    }

    public String getSipPersistenceFrequency() {
        checkInitialization();

        return sipPersistenceFrequency;
    }

    public String getSipPersistenceScope() {
        checkInitialization();

        return sipPersistenceScope;
    }

    /**
     * Gets the bean representing the sip-container-availability element
     * in domain.xml.
     *
     * @return The bean representing the sip-container-availability element
     * in domain.xml.
     */
    public SipContainerAvailability getSipContainerAvailability() {
        return lookup.getSipContainerAvailability();
    }
}
