/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.config;

import java.util.HashMap;

import javax.servlet.sip.SipSessionsUtil;


/**
 * Th SipSessionsUtilMap exposes a set of methods to hold the SipSessionsUtil instances
 * corresponding to an SipApplication or a ConvergedApplication. A HashMap is
 * used to map the SipSessionsUtil instances with the names of the corresponding
 * Sip/Converged Application
 *
 * This class is a singleton, and would have a single Map per instance of the
 * Application Server. As per JSR289, there would never be a need to retrive a
 * SipSessionsUtil of an Application which is deployed on another instance of the
 * Application Server.
 *
 * @author prasad subramanian
 */
public class SipSessionsUtilMap {
    /** Member variables */
    // SipSessionsUtilMap singleton object
    protected static SipSessionsUtilMap sipSessionsUtilMap = 
        new SipSessionsUtilMap();

    // HashMap where the SipSessionsUtil instance for each app is stored.
    private HashMap<String, SipSessionsUtil> ssuMap;

    /** Creates a new instance of SipSessionsUtilMap */
    private SipSessionsUtilMap() {
    }

    /** Public methods */
    public static SipSessionsUtilMap getInstance() {
        return sipSessionsUtilMap;
    }

    /**
     * Adds the instance of the SipSessionsUtil associated with an application
     * @param  appName The name of the application whose SipSessionsUtil instance is being
     *         stored
     * @param  sf The SipSessionsUtil object being added
     */
    public void addSipSessionsUtil(String appName, SipSessionsUtil ssu) {
        if (ssuMap == null) {
            ssuMap = new HashMap();
            ssuMap.put(appName, ssu);
        } else {
            ssuMap.put(appName, ssu);
        }
    }

    /**
     * Retrieves the SipSessionsUtil instance for a specific application
     * @param the name of the application whose SipSessionsUtil instance is to
     *        be retrieved.
     * @return the SipSessionsUtil object
     */
    public SipSessionsUtil getSipSessionsUtil(String appName) {
        return (SipSessionsUtil) ssuMap.get(appName);
    }

    /**
     * Removes the SipSessionsUtil instance for a particular
     * application
     * @param appName The name of the application whose SipSessionsUtil instance is to be
     *        removed from the HashMap
     * @return The SipFactory object that has been removed.
     */
    public SipSessionsUtil removeSipSessionsUtil(String appName) {
        return (SipSessionsUtil) ssuMap.remove(appName);
    }

    /**
     * Checks if the application has a SipSessionsUtil mapped
     * @param appName the name of the application for which the check is being
     *                made
     */
    public boolean isSipSessionsUtilMapped(String appName) {
        return ssuMap.containsKey(appName);
    }
}
