/*
* The contents of this file are subject to the terms
* of the Common Development and Distribution License
* (the License).  You may not use this file except in
* compliance with the License.
*
* You can obtain a copy of the license at
* https://glassfish.dev.java.net/public/CDDLv1.0.html or
* glassfish/bootstrap/legal/CDDLv1.0.txt.
* See the License for the specific language governing
* permissions and limitations under the License.
*
* When distributing Covered Code, include this CDDL
* Header Notice in each file and include the License file
* at glassfish/bootstrap/legal/CDDLv1.0.txt.
* If applicable, add the following below the CDDL Header,
* with the fields enclosed by brackets [] replaced by
* you own identifying information:
* "Portions Copyrighted [year] [name of copyright owner]"
*
* Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
* Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
*/
package com.ericsson.ssa.config.annotations;

import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.ConfigRuntimeException;
import com.ericsson.ssa.config.event.ConfigAddEvent;
import com.ericsson.ssa.config.event.ConfigChangeListener;
import com.ericsson.ssa.config.event.ConfigRemoveEvent;
import com.ericsson.ssa.config.event.ConfigUpdateEvent;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 *
 * @author elnelbo
 */
//TODO Optimization to reduce the number of listeners for a specific node.
//Now for each annotation possibly a listeners is registered.
//This can be done by seperating the Injector from the listeners and have the
// listerner call all associated injectors to do injection.
public class ConfigurationAnnotationIntrospector {
    private static final ConfigurationAnnotationIntrospector INSTANCE = 
            new ConfigurationAnnotationIntrospector();
    private Map<Object, List<ConfigInjection>> configurables = 
            new HashMap<Object, List<ConfigInjection>>();
    private LogUtil log = LogUtil.SIP_LOGGER;
    
    private ConfigurationAnnotationIntrospector() {        
    }
    
    public static ConfigurationAnnotationIntrospector instance() {
        return INSTANCE;
    }
    
    public void activateConfiguration(Object configurable) {
        if (configurable!=null && !configurables.containsKey(configurable)) {
            if (log.isLoggable(Level.FINE)) {
                log.logMsg(Level.FINE, "Activating Configuration for Object "+ 
                        configurable.getClass().getName()+":"+
                        configurable.hashCode());
            }
            List<ConfigInjection> configInjectors = 
                    new ArrayList<ConfigInjection>();
            configurables.put(configurable, configInjectors);
            
            for (Method method : configurable.getClass().getMethods()) {
                Configuration annotation = 
                        method.getAnnotation(Configuration.class);
                if (annotation!=null) {                    
                    if (log.isLoggable(Level.FINEST)) {
                        log.logMsg(Level.FINEST, "Annotated Method found:"+method);
                    }
                    String key = getCompatibleKey(annotation, method);
                    Class propertyType = getCompatibleArgument(method);
                    if (log.isLoggable(Level.FINEST)) {
                        log.logMsg(Level.FINEST, "Method is compatable");
                    }
                    
                    ConfigInjection configInjector = null;                
                    switch (annotation.update()) {
                    case STARTUP:
                        configInjector = new StartupConfigInjector(key, configurable, 
                                method, propertyType, annotation);
                        break;
                    case DYNAMIC:
                        configInjector = new DynamicConfigInjector(key, configurable, 
                                method, propertyType, annotation);
                        break;
                    }

                    configInjectors.add(configInjector);
                    configInjector.setup();
                    if (log.isLoggable(Level.FINE)) {
                        log.logMsg(Level.FINE, "Injector setup: "+configInjector);
                    }
                }
            }
        }
    }
    public void deactivateConfiguration(Object configurable) {
        List<ConfigInjection> configInjectors =
                configurables.remove(configurable);
        if (configInjectors!=null) {
            for (ConfigInjection configInjector : configInjectors) {
                configInjector.cleanup();
            }
        }
    }

    private String getCompatibleKey(Configuration annotation, Method method) {
            String key = annotation.key();
              
            //Key is not specified, take the property name from the setter
            //Note couldn't find a propery BeanUtil that does the trick easily.
            if (key.length()==0) {
                key = method.getName();
                if (key.startsWith("set") && key.length()>"set".length()) {
                    key = Character.toUpperCase(key.charAt("set".length())) +
                            key.substring("set".length()+1);
                } else {
                    if (log.logConfig()) {
                        log.config("sip.common.config_annotations_no_javabean_property_setter", 
                                method.getName());
                    }
                    throw new ConfigRuntimeException("sip.common.config_annotations_activation_failed_exception");
                }
            }
            
            return key;       
    }
    
    private Class getCompatibleArgument(Method method) {
        Class propertyType = null;
        Class[] paramTypes = method.getParameterTypes();
        if (paramTypes.length == 1) {
            try {
                if (!paramTypes[0].isPrimitive()) {
                    paramTypes[0].getConstructor(String.class);
                }
                propertyType = paramTypes[0];
            } catch (NoSuchMethodException ex) {
                if (log.logConfig()) {
                    log.config("sip.common.config_annotations_in_compatable_method_signature", 
                            method.getName(), 
                            "arguments type must have have constructure with string argument");
                }
                throw new ConfigRuntimeException("sip.common.config_annotations_activation_failed_exception");
            }
        } else {
            if (log.logConfig()) {
                log.config("sip.common.config_annotations_in_compatable_method_signature", 
                        method.getName(), 
                        "nr of arguments must be 1");
            }
            throw new ConfigRuntimeException("sip.common.config_annotations_activation_failed_exception");
        }
        return propertyType;
    }
    
    private interface ConfigInjection {
        void setup();
        void cleanup();
    }
    
    private class StartupConfigInjector implements ConfigInjection {
        protected Object configurable;
        protected Method method;
        protected Class propertyType;
        protected Configuration annotation;
        protected String key; 
        private String lastValue;

        public StartupConfigInjector(String aKey, Object aConfigurable, 
                Method aMethod,
                Class aPropertyType,
                Configuration anAnnotation) {
            key = aKey;
            configurable = aConfigurable;
            method = aMethod;
            propertyType = aPropertyType;
            annotation = anAnnotation;
            lastValue = null;
        }

        public synchronized void setup() {
            String value = getCurrentConfigValue();
            inject(value);
            lastValue = value;            
        }

        public synchronized void injectChange() {
            String value = getCurrentConfigValue();
            if (isValueChanged(value)) {
                inject(value);
                lastValue = value;
            }            
        }
        
        public synchronized void inject(String value) {            
            String resolvedValue = value;
            Object arg = null;
                
            if (value==null) {
                switch(annotation.usage()) {
                case FAIL:
                    if (log.logSevere()) {
                        log.severe("sip.common.config_annotations_FAIL_usage_on_missing_or_incompatible_value", 
                            method.getName(), key, annotation.node());
                    }
                    throw new ConfigRuntimeException("sip.common.config_annotations_FAIL_usage_exception");
                case WARN:
                    if (log.logWarning()) {
                        log.warning("sip.common.config_annotations_WARN_usage_on_missing_or_incompatible_value", 
                            method.getName(), key, annotation.node());
                    }
                    break;
                case IGNORE:
                    if (log.isLoggable(Level.FINEST)) {
                        log.log(Level.FINEST, "sip.common.config_annotations_IGNORE_usage_on_missing_or_incompatible_value", 
                            method.getName(), key, annotation.node());
                    }
                    break;
                case DEFAULT:
                    if (log.logConfig()) {
                        log.config("sip.common.config_annotations_DEFAULT_usage_on_missing_or_incompatible_value", 
                            method.getName(), key, annotation.node(), annotation.value());
                    }                    
                    resolvedValue = annotation.value();
                    break;
                }
            }               
            
            arg = resolveArg(resolvedValue);
            
            if (arg!=null) {
                try {
                    if (log.isLoggable(Level.FINEST)) {
                        log.logMsg(Level.FINEST, "Injecting arg "+arg+" in Method "+method);
                    }
                    method.invoke(configurable, arg);
                } catch (IllegalAccessException ex) {
                    if (log.logWarning()) {
                        log.warning(ex, "sip.common.config_annotations_injection_failed_exception_on_invoke", 
                            method.getName(), key, annotation.node());
                    }                    
                } catch (IllegalArgumentException ex) {
                    if (log.logWarning()) {
                        log.warning(ex, "sip.common.config_annotations_injection_failed_exception_on_invoke", 
                            method.getName(), key, annotation.node());
                    }                    
                } catch (InvocationTargetException ex) {
                    if (log.logWarning()) {
                        log.warning(ex, "sip.common.config_annotations_injection_failed_exception_on_invoke", 
                            method.getName(), key, annotation.node());
                    }                    
                }
            }
        }

        protected String getCurrentConfigValue() {
            String value = null;
            String node = annotation.node();
            node = stripTrailingSlash(node);
            
            if (node!=null && annotation.node().length() > 0) {
                value = ConfigFactory.getConfig().get(annotation.node(), key);
            } else {
                System.out.println("Getting "+key+" for server instance or jvm-option");
                value = ConfigFactory.getConfig().get(key);
            }

            return value;
        }
        
        protected String stripTrailingSlash(String str) {
            String stripped = str!=null && str.endsWith("/") ? 
                str.substring(str.length()-1): str;
            return stripped;
        }
        
        protected boolean isValueChanged(String aValue) {
            return lastValue!=null ? !lastValue.equals(aValue) :
                aValue!=null;            
        }

        protected Object resolveArg(String value) throws IllegalArgumentException, SecurityException {
            Object arg = null;
            boolean done = false;

            while (!done) {
                try {
                    Class argType = !propertyType.isPrimitive() ? 
                        propertyType :
                        getWrapperForPrimitiveArg();
                    if (argType!=Character.class) {
                        arg = argType.getConstructor(
                            String.class).newInstance(value);
                    } else {
                        //Special case for Character, who doesn't have a 
                        //constructor taking a String as argument.
                        //We take the first Character of the value.
                        arg = value.charAt(0);
                    }
                    done = true;
                } catch (Exception ex) {
                    switch(annotation.usage()) {
                    case FAIL:
                        done=true;
                        if (log.logSevere()) {
                            log.severe(ex, "sip.common.config_annotations_FAIL_usage_on_missing_or_incompatible_value", 
                                method.getName(), key, annotation.node(), value);
                        }
                        throw new ConfigRuntimeException("sip.common.config_annotations_FAIL_usage_exception");
                    case WARN:
                        done=true;
                        if (log.logWarning()) {
                            log.warning(ex, "sip.common.config_annotations_WARN_usage_on_missing_or_incompatible_value", 
                                method.getName(), key, annotation.node(), value);
                        }
                        break;
                    case IGNORE:
                        done=true;
                        if (log.isLoggable(Level.FINEST)) {
                            log.log(Level.FINEST, ex, "sip.common.config_annotations_IGNORE_usage_on_missing_or_incompatible_value", 
                                method.getName(), key, annotation.node(), value);
                        }
                        break;
                    case DEFAULT:
                        if (value!=annotation.value()) /* intentional ref compare */ {
                            if (log.logConfig()) {
                                log.config(ex, "sip.common.config_annotations_DEFAULT_usage_on_missing_or_incompatible_value", 
                                    method.getName(), key, annotation.node(), annotation.value(), value);
                            }                    
                            value = annotation.value();
                        } else {
                            done = true;
                        }
                        break;
                    }
                }
            }
            
            return arg;
        }

        private Class getWrapperForPrimitiveArg() {
            Class wrapper = null;
            
            if (propertyType==Boolean.TYPE) {
                wrapper = Boolean.class;
            } else if (propertyType==Byte.TYPE) {
                wrapper = Byte.class;
            } else if (propertyType==Short.TYPE) {
                wrapper = Short.class;
            } else if (propertyType==Integer.TYPE) {
                wrapper = Integer.class;
            } else if (propertyType==Long.TYPE) {
                wrapper = Long.class;
            } else if (propertyType==Float.TYPE) {
                wrapper = Float.class;
            } else if (propertyType==Double.TYPE) {
                wrapper = Double.class;
            } else if (propertyType==Character.TYPE) {
                wrapper = Character.class;
            }
            
            return wrapper;
        }
        
        public synchronized void cleanup() { 
            configurable = null;
            method = null;
            propertyType = null;
            annotation = null;
        }
        
        @Override
        public String toString() {
            StringBuffer buf = new StringBuffer("ConfigInjection: ");
            buf.append(getClass().getName());
            buf.append(" {key=");
            buf.append(key);
            buf.append(", node=");
            buf.append(annotation.node());
            buf.append(", method=");
            buf.append(method.getName());
            buf.append(", property type=");
            buf.append(propertyType.getName());
            buf.append("}");
            
            return buf.toString();
        }
    }
    
    private class DynamicConfigInjector extends StartupConfigInjector 
            implements ConfigChangeListener {
        
        private DynamicConfigInjector(String aKey, Object aConfigurable, 
                Method aMethod,
                Class aPropertyType,
                Configuration anAnnotation) {
            super(aKey, aConfigurable, aMethod, aPropertyType, anAnnotation);
        }

        @Override
        public synchronized void setup() {
            ConfigFactory.instance().registerConfigChangeListener(
                    stripTrailingSlash(annotation.node()), this);
            super.setup();
        }
        
        @Override
        public synchronized void cleanup() {
            ConfigFactory.instance().deregisterConfigChangeListener(this);
            super.cleanup();
        }        

        public void handleConfigEvent(ConfigAddEvent event) {
            injectChange();
        }

        public void handleConfigEvent(ConfigUpdateEvent event) {
            injectChange();
        }

        public void handleConfigEvent(ConfigRemoveEvent event) {
            injectChange();
        }        
    }    
}
