/*
 * SipBindingResolver.java
 *
 * Created on August 28, 2007, 3:39 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.config.event.ConfigAddEvent;
import com.ericsson.ssa.config.event.ConfigChangeListener;
import com.ericsson.ssa.config.event.ConfigRemoveEvent;
import com.ericsson.ssa.config.event.ConfigUpdateEvent;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;


/**
 *
 * @author ELNELBO
 */
public class SipBindingResolver {
    /*
     * A lot to tricky things still to resolve.
     * - Multithreading, Initialization, multiple users and the Config event thread.
     * - Transaction awareness of the ConfigEvent (add node, but prefs not yet set, but event sent)
     * - Expects valid prefs with defaults set.
     */
    private static final SipBindingResolver INSTANCE = new SipBindingResolver();
    private static final String CTX_PARENT = "/SipService/SipListener";
    private static final String CTX_PARENT_REGEXP = "/SipService/SipListener";
    private static final String CTX_REGEXP = "/SipService/SipListener/([^/]+)";
    private static final String SIP_CONTAINER = "/SipContainer";

    /** The context to be used for obtaining a lease for the public binding.
     */
    public static final String PUBLIC_BINDING_CTX = "PUBLIC_BINDING_CTX";
    private Config config = ConfigFactory.getConfig();
    private LogUtil log = LogUtil.SIP_LOGGER;
    private ParentCtxChangeListener parentCtxListener = new ParentCtxChangeListener();
    private PublicCtxChangeListener publicCtxListener = new PublicCtxChangeListener();
    private Map<String, SipBindingCtx> contexts = new HashMap<String, SipBindingCtx>();
    private Map<String, CtxChangeListener> ctxListeners = new HashMap<String, CtxChangeListener>();
    private List<SipBindingListener> sipBindingListeners = new ArrayList<SipBindingListener>();
    private String currentPublicHost = null;
    private Random rand = new Random();

    /** Creates a new instance of SipBindingResolver */
    public SipBindingResolver() {
    }

    public static SipBindingResolver instance() {
        return INSTANCE;
    }

    public void init() {
        ConfigFactory.instance()
                .registerConfigChangeListener(CTX_REGEXP, parentCtxListener);

        //Init the local bindings confgured
        for (String childName : config.getChildNames(CTX_PARENT)) {
            String ctx = getCtxForName(childName);
            contexts.put(ctx, new SipBindingCtx(this, ctx, resolveLocalTargetTuples(ctx)));
            CtxChangeListener ctxListener = new CtxChangeListener();
            ConfigFactory.instance()
                .registerConfigChangeListener(getCtxRegExp(ctx), ctxListener);
            ConfigFactory.instance()
                .registerConfigChangeListener(getCtxSslRegExp(ctx), ctxListener);
        }

        ConfigFactory.instance()
                .registerConfigChangeListener(SIP_CONTAINER, publicCtxListener);
        contexts.put(PUBLIC_BINDING_CTX, 
                new SipBindingCtx(this, PUBLIC_BINDING_CTX, resolvePublicTargetTuples(), false));
    }
    
    public String[] getLocalContexts() {
        List<String> ctxs = new ArrayList<String>(contexts.keySet());
        ctxs.remove(PUBLIC_BINDING_CTX);

        return ctxs.toArray(new String[ctxs.size()]);
    }
    
    public SipBindingCtx getPublicContext() {
        return contexts.get(PUBLIC_BINDING_CTX);
    } 


    public SipBindingCtx getContext(String ctx) {
        SipBindingCtx sipBindingCtx = contexts.get(ctx);

        return sipBindingCtx;
    }
    
    public boolean isStale(SipBindingCtx sipBindingCtx) {
        return sipBindingCtx==null || 
                !contexts.containsKey(sipBindingCtx.getContext()) ||
                contexts.get(sipBindingCtx.getContext())!=sipBindingCtx;
    }

    //utils
    public String getCurrentPublicHost() {
        if (currentPublicHost==null) {
            currentPublicHost = 
                    contexts.get(PUBLIC_BINDING_CTX).getTargetTuples()[0].getIP();
        }
        
        return currentPublicHost;
    }
    
    /**
     * Get an active SipBindingCtx for a specific protocol.
     * 
     * @param protocol
     * @return SipBindingCtx
     */
    public SipBindingCtx getActiveLocalContext(SipTransports protocol) {
        SipBindingCtx ctx = null;
        List<SipBindingCtx> ctxs = getLocalContextsForProtocol(protocol);
        if (!ctxs.isEmpty()) {
            ctx = ctxs.get(rand.nextInt(ctxs.size()));
        }
        return ctx;
    }
    
    public List<SipBindingCtx> getLocalContextsForProtocol(SipTransports protocol) {
        List<SipBindingCtx> ctxs = new ArrayList<SipBindingCtx>(1);
        for (Map.Entry<String, SipBindingCtx> e : contexts.entrySet()) {
            if (!PUBLIC_BINDING_CTX.equals(e.getKey())) {
                if (protocol==null || e.getValue().getTargetTupleForProtocol(protocol)!=null) {
                    ctxs.add(e.getValue());
                }
            }
        }
        
        return ctxs;
    }
    
    //SipBindingListeners
    public void registerSipBindingListener(SipBindingListener listener) {
        synchronized(sipBindingListeners) {
            sipBindingListeners.add(listener);
        }
    }

    public void deregisterSipBindingListener(SipBindingListener listener) {
        synchronized(sipBindingListeners) {
            sipBindingListeners.remove(listener);
        }
    }

    private void processNewSipBindingCtxAvalable(String context) {
        synchronized(sipBindingListeners) {
            for (SipBindingListener listener : sipBindingListeners) {
                listener.newSipBindingCtxAvaliable(context);
            }
        }
    }
    
    private void processSipBindingCtxUpdated(String context) {
        synchronized(sipBindingListeners) {
            for (SipBindingListener listener : sipBindingListeners) {
                listener.sipBindingCtxUpdated(context);
            }
        }
    }
    
    private void processSipBindingCtxRemove(String context) {
        synchronized(sipBindingListeners) {
            for (SipBindingListener listener : sipBindingListeners) {
                listener.sipBindingCtxRemoved(context);
            }
        }
    }    

    //ConfigChangeListeners    
    private String getCtxForName(final String name) {
        return CTX_PARENT + "/" + name;
    }

    private String getCtxRegExp(final String ctx) {
        return ctx;
    }
    
    private String getCtxSslRegExp(final String ctx) {
        return ctx+"/Ssl";
    }    

    //Resolvement
    private TargetTuple[] resolveLocalTargetTuples(String ctx) {
        String address = resolveLocalAddress(config.get(ctx, "Address"));
        int port = resolveLocalPort(config.get(ctx, "Port"));
        String transport = config.get(ctx, "Transport");
        List<SipTransports> transports = new ArrayList<SipTransports>();

        for (String prot : transport.split("_")) {
            try {
                transports.add(SipTransports.getTransport(prot));
            } catch (Exception ex) {
                //TODO logging!
                throw new IllegalArgumentException(ex);
            }
        }

        TargetTuple[] targetTuples = new TargetTuple[transports.size()];

        int i = 0;

        for (SipTransports st : transports) {
            targetTuples[i++] = new TargetTuple(st, address, port);
        }

        return targetTuples;
    }   

    private String resolveLocalAddress(String anLocalAddress) {
        String address = ((anLocalAddress != null) &&
            (anLocalAddress.length() > 0)) ? anLocalAddress
                                           : Constants.BIND_TO_ANY;

        return address;
    }

    private int resolveLocalPort(String aLocalPort) {
        int port = ((aLocalPort != null) && (aLocalPort.length() > 0))
            ? Integer.parseInt(aLocalPort)
            : Integer.parseInt(Constants.DEFAULT_SIP_PORT);

        return port;
    }

    private TargetTuple[] resolvePublicTargetTuples() {
        String address = resolvePublicAddress(config.get(SIP_CONTAINER,
                    "ExternalAddress"));
        int sipPort = resolvePublicPort(config.get(SIP_CONTAINER, "ExternalSipPort"), 
                Constants.DEFAULT_SIP_PORT);
        int sipsPort = resolvePublicPort(config.get(SIP_CONTAINER, "ExternalSipsPort"), 
                Constants.DEFAULT_SIPS_PORT);

        TargetTuple[] targetTuples = {
                new TargetTuple(SipTransports.UDP_PROT, address, sipPort),
                new TargetTuple(SipTransports.TCP_PROT, address, sipPort),
                new TargetTuple(SipTransports.TLS_PROT, address, sipsPort)
	};

        return targetTuples;
    }
    
    private String resolvePublicAddress(String anPublicAddress) {
        String address = ((anPublicAddress != null) &&
            (anPublicAddress.length() > 0)) ? anPublicAddress : getLanAddress();

        return address;
    }

    private int resolvePublicPort(String aPublicPort, String defaultPort) {
        int port = ((aPublicPort != null) && (aPublicPort.length() > 0))
            ? Integer.parseInt(aPublicPort)
            : Integer.parseInt(defaultPort);

        return port;
    }
    
    /**
     * getLanAddress() tries to find a configured IPv4 address of a physical
     * network interface. Some heuristics will be applied so that on TSP/Linux
     * one of the internal LAN addresses will most likely be returned. <br>
     * getLanAddress() always returns a correctly formatted dotted-decimal IP
     * address string; in the event of a machine with no configured IP addresses
     * it will return "0.0.0.0". <br>
     *
     * @return java.lang.String
     * @author erarafo
     */
    private String getLanAddress() {
        String lanAddress = "";
        String preferedPrefix = "172."; // TSP/Linux internal LAN address prefix,
                                        // preferred

        String avoidedPrefix = "127."; // Loopback interface, to be avoided
        String avoidedInterfacenamePrefix = "i4dd"; // VIP addresses, to be
                                                    // avoided

        String me = "Lan address resolve";

        if (log.isLoggable(Level.FINE)) {
            log.logMsg(Level.FINE,
                me +
                " try to find a configured IP address of a physical network interface");
        }

        int score = 0;
        int newscore;
        Enumeration<NetworkInterface> nifs = null;

        try {
            nifs = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException se) {
            if (log.logWarning()) {
                log.warning(se, "sip.stack.network.resolve_obtain_nw_interfaces_failed", me);
            }            
        }

        if (nifs != null) {
            while (nifs.hasMoreElements()) {
                NetworkInterface nif = nifs.nextElement();
                String ifName = nif.getName();

                if (log.isLoggable(Level.FINE)) {
                    log.logMsg(Level.FINE, me + " found interface: " + ifName);
                }

                int addrCount = 0;

                for (Enumeration<InetAddress> addrs = nif.getInetAddresses();
                        addrs.hasMoreElements();) {
                    InetAddress addr = addrs.nextElement();
                    addrCount++;

                    if (log.isLoggable(Level.FINE)) {
                        log.logMsg(Level.FINE,
                            me + " found address " + addr + " for interface " +
                            ifName);
                    }

                    String addrString = addr.getHostAddress();

                    if (addrString.startsWith(preferedPrefix)) {
                        newscore = 40; // 40: preferred address

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else if (addrString.startsWith(avoidedPrefix)) {
                        newscore = 10; // 10: avoid this address if possible

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else if (ifName.startsWith(avoidedInterfacenamePrefix)) {
                        newscore = 20; // 20: avoid this address if possible

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else if (addrString.indexOf(":") >= 0) {
                        newscore = 25; // score 25 for IPv6 address

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else {
                        newscore = 30; // 30: neither preferred nor avoided

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (log.isLoggable(Level.FINE)) {
                                log.logMsg(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    }
                }

                if (addrCount < 1) {
                    if (log.isLoggable(Level.INFO)) {
                        log.logMsg(Level.INFO,
                            me +
                            "found no configured IP addresses for interface " +
                            ifName);
                    }
                } else if (addrCount > 1) {
                    if (log.isLoggable(Level.INFO)) {
                        log.logMsg(Level.INFO,
                            me + "found " + addrCount +
                            " configured IP addresses for interface " + ifName);
                    }
                }
            }
        }

        if (lanAddress == null) {
            lanAddress = Constants.BIND_TO_ANY;
            if (log.logWarning()) {
                log.warning("sip.stack.network.resolve_only_unsuitable_ip_address_found",
                    me, score, lanAddress);                
            }
        } else if (score < 30) {
            if (log.logWarning()) {
                log.warning("sip.stack.network.resolve_only_unsuitable_ip_address_found",
                    me, score, lanAddress);                
            }
        }

        if (log.isLoggable(Level.FINE)) {
            log.logMsg(Level.FINE,
                "LanAddress: using " +
                ((score > 30) ? "preferred " : "") + "value '" + lanAddress +
                "'");
        }

        return lanAddress;
    }

    public String reportBindings() {
        StringBuffer buf = new StringBuffer("SipBindings: ");

        for (Map.Entry<String, SipBindingCtx> entries : contexts.entrySet()) {
            buf.append("\ncontext: '");
            buf.append(entries.getKey());
            buf.append("'.\n");

            for (TargetTuple tuple : entries.getValue().getTargetTuples()) {
                buf.append(tuple);
            }
        }

        return buf.toString();
    }

    class ParentCtxChangeListener implements ConfigChangeListener {
        /**
         * Prepares the new TargetTuples for the ctx.
         * registeres CtxChangeListener
         */
        public void handleConfigEvent(ConfigAddEvent event) {
            contexts.put(event.getNode(),
                new SipBindingCtx(SipBindingResolver.INSTANCE, event.getNode(), 
                resolveLocalTargetTuples(event.getNode())));

            CtxChangeListener listener = new CtxChangeListener();
            ctxListeners.put(event.getNode(), listener);
            ConfigFactory.instance().registerConfigChangeListener(getCtxRegExp(
                    event.getNode()), listener);
            ConfigFactory.instance().registerConfigChangeListener(getCtxSslRegExp(
                    event.getNode()), listener);
            
            processNewSipBindingCtxAvalable(event.getNode());
        }

        public void handleConfigEvent(ConfigUpdateEvent event) {
            //ignore
        }

        /**
         * Prepares the new TargetTuples for the ctx.
         * Cancels all associated leases.
         * deregisteres CtxChangeListener
         */
        public void handleConfigEvent(ConfigRemoveEvent event) {
            SipBindingCtx sipBindingCtx = contexts.remove(event.getNode());

            CtxChangeListener listener = ctxListeners.remove(event.getNode());

            if (listener != null) {
                ConfigFactory.instance().deregisterConfigChangeListener(listener);
            }
            
            processSipBindingCtxRemove(event.getNode());
        }
    }

    class CtxChangeListener implements ConfigChangeListener {
        public void handleConfigEvent(ConfigAddEvent event) {
            if (event.getNode().endsWith("/Ssl")) {
                String ctx = getCtxFromSslNode(event.getNode());
                snapshot(ctx);
                processSipBindingCtxUpdated(ctx);
            } else {
                //ignore
            }
        }

        public void handleConfigEvent(ConfigUpdateEvent event) {
            String ctx = event.getNode();
            if (event.getNode().endsWith("/Ssl")) {
                ctx = getCtxFromSslNode(event.getNode());
            }
            
            snapshot(ctx);
            processSipBindingCtxUpdated(ctx);
        }

        public void handleConfigEvent(ConfigRemoveEvent event) {
            if (event.getNode().endsWith("/Ssl")) {
                String ctx = getCtxFromSslNode(event.getNode());
                snapshot(ctx);
                processSipBindingCtxUpdated(ctx);
            } else {
                //ignore
            }
        }
        
        private String getCtxFromSslNode(String node) {
            return node.substring(0,node.length()-"/Ssl".length());            
        }
        
        private void snapshot(String ctx) {
            contexts.put(ctx,
                new SipBindingCtx(SipBindingResolver.INSTANCE, ctx, 
                resolveLocalTargetTuples(ctx)));                            
        }
    }
    
    class PublicCtxChangeListener implements ConfigChangeListener {
        public void handleConfigEvent(ConfigAddEvent event) {
            //ignore
        }

        /**
         * Prepares the new TargetTuples for the ctx.
         * Cancels all associated leases.
         */
        public void handleConfigEvent(ConfigUpdateEvent event) {
            currentPublicHost = null;
            contexts.put(PUBLIC_BINDING_CTX, 
                    new SipBindingCtx(SipBindingResolver.INSTANCE, 
                    PUBLIC_BINDING_CTX, resolvePublicTargetTuples(), false));
            getCurrentPublicHost();
        }

        public void handleConfigEvent(ConfigRemoveEvent event) {
            //ignore
        }
    }    
}
