/*
* The contents of this file are subject to the terms
* of the Common Development and Distribution License
* (the License).  You may not use this file except in
* compliance with the License.
*
* You can obtain a copy of the license at
* https://glassfish.dev.java.net/public/CDDLv1.0.html or
* glassfish/bootstrap/legal/CDDLv1.0.txt.
* See the License for the specific language governing
* permissions and limitations under the License.
*
* When distributing Covered Code, include this CDDL
* Header Notice in each file and include the License file
* at glassfish/bootstrap/legal/CDDLv1.0.txt.
* If applicable, add the following below the CDDL Header,
* with the fields enclosed by brackets [] replaced by
* you own identifying information:
* "Portions Copyrighted [year] [name of copyright owner]"
*
* Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
* Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
*/
package com.ericsson.ssa.sip;

import java.io.Serializable;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.servlet.sip.Parameterable;
import javax.servlet.sip.ServletParseException;


public class ParameterableImpl implements Parameterable, Serializable {
    private ParameterByteMap parameters;
    private String value;

    // System headers cannot be modified.
    protected boolean _readOnly = false;

    public ParameterableImpl() {
    }

    public void setParameterByteMap(byte[] array, int offset, int length,
        char separator) throws ServletParseException {
        parameters = new ParameterByteMap(array, offset, length, separator);
    }

    /**
     * Returns a clone of this ParameterableImpl instance. The clone will be equal to its original.
     * This is the implementation of the javax.servlet.sip.ParameterableImpl.clone() method.
     * The clone will not have the readOnly property set, since an application may wish to
     * modify the clone even if it was created from a System Header Parameterable object.
     */
    public Object clone() {
        return clone(false);
    }

    public Object clone(boolean copyReadOnly) {
        // ParameterableImpl result = new ParameterableImpl();
        ParameterableImpl result = null;

        try {
            result = (ParameterableImpl) super.clone();
        } catch (CloneNotSupportedException cex) {
            cex.printStackTrace();
        }

        if (value != null) {
            result.value = value;
        }

        if (parameters != null) {
            result.parameters = (ParameterByteMap) parameters.clone();
        }

        if (!copyReadOnly) {
            // Set ReadOnly to false (default behaviour)
            result._readOnly = false;
        }

        return result;
    }

    public boolean equals(Object o) {
        if (o == null) {
            return false;
        }

        if (this == o) {
            return true;
        }

        if (!(o instanceof ParameterableImpl)) {
            return false;
        }

        ParameterableImpl p = (ParameterableImpl) o;

        if (value == null) {
            if (p.value != null) {
                return false;
            }
        } else {
            if (!value.equals(p.value)) {
                return false;
            }
        }

        return parameters.equals(p.parameters);
    }

    public String getParameter(String key) {
        if (parameters == null) {
            return null;
        }

        return parameters.get(key);
    }

    public Iterator<String> getParameterNames() {
        if (parameters == null) {
            return Collections.EMPTY_LIST.iterator();
        }

        return parameters.getKeys();
    }

    public Set<Map.Entry<String, String>> getParameters() {
        if (parameters != null) {
            return parameters.entrySet();
        } else {
            return Collections.EMPTY_SET;
        }
    }

    public String getValue() {
        return value;
    }

    public void removeParameter(String name) {
        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        if (parameters != null) {
            parameters.remove(name);
        }
    }

    public void setParameter(String name, String value) {
        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        if (parameters == null) {
            parameters = new ParameterByteMap(';');
        }

        parameters.put(name, value);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();

        //  sb.append(" Filed Value : ");
        if (value != null) {
            sb.append(value);
        }

        // sb.append(" Parameters : ");
        if (parameters != null) {
            sb.append(parameters.toString());
        }

        return sb.toString();
    }

    public void setValue(String value) {
        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        this.value = value;
    }

    public boolean isReadOnly() {
        return _readOnly;
    }

    public void setReadOnly(boolean writeable) {
        _readOnly = writeable;
    }

    /* str format :  filed-value *(;param-name=param-value)  */
    static ParameterableImpl parse(String str) {
        ParameterableImpl result = new ParameterableImpl();
        StringTokenizer stokener = new StringTokenizer(str, ";");

        if (stokener.countTokens() > 0) {
            result.setValue(stokener.nextToken());
        }

        while (stokener.hasMoreTokens()) {
            StringTokenizer paramTokens = new StringTokenizer(stokener.nextToken(),
                    "=");
            String paramName = null;
            String paramValue = null;

            if (paramTokens.hasMoreTokens()) {
                paramName = paramTokens.nextToken();
            }

            if (paramTokens.hasMoreTokens()) {
                paramValue = paramTokens.nextToken();
            }

            result.setParameter(paramName, paramValue);
        }

        return result;
    }
}
