/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */

package com.ericsson.ssa.sip;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;

/**
 * @author jluehe
 */
public abstract class PersistentDialogFragmentManager extends DialogFragmentManager {
    private static final long REAPER_THREAD_SLEEP_TIME = 60 * 1000L; // 1 minute

    protected boolean isRepairDuringFailure = true;

    // The persistent store for dialog fragments
    private DialogFragmentStore store = null;

    /**
     * The background reaper thread.
     */
    private Thread reaperThread = null;
    private boolean reaperThreadDone = false;

    protected PersistentDialogFragmentManager() {
        super();
        reaperThread = new Thread(new ReaperThread(), "DialogFragmentReaper");
        reaperThread.setDaemon(true);
    }

    /**
     * Gets the DialogFragment with the given id.
     *
     * This method first checks the active cache for a DialogFragment with the
     * given id. If not found, it queries the persistent store.
     *
     * @return The DialogFragment with the given id, or null if not found
     */
    public DialogFragment findDialogFragment(String id) {
        
        DialogFragment df = super.findDialogFragment(id);
        if (df != null || store == null) {
            return df;
        }

        try {
            df = swapInDialogFragment(id);
        } catch (IOException ioe) {
            LogUtil.SIP_LOGGER.severe(
                ioe, "sipstack.swap_in_dialog_fragment_error", id);
        }

        return df;
    }

    /**
     * Removes the given DialogFragment.
     *
     * @param d The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */
    public void removeDialogFragment(DialogFragment df, boolean hasTimedOut) {

        super.removeDialogFragment(df, hasTimedOut);

        if (store != null && df != null && df.isReplicable()) {
            try {
                store.remove(df.getFragmentId());
            } catch (IOException ioe) {
                LogUtil.SIP_LOGGER.severe(
                    ioe,
                    "sipstack.remove_dialog_fragment_from_store_error",
                    df.getFragmentId());
            }
        }
    }    

    /**
     * Persists the given DialogFragment.
     *
     * @param df The DialogFragment to be persisted
     */
    public void saveDialogFragment(DialogFragment df)
            throws IOException {
        if (store != null && df != null && df.isReplicable()) {
            store.save(df);
        }
    }

    /**
     * Swaps in the DialogFragment with the given id from the persistent
     * store of this dialog manager.
     *
     * @return The DialogFragment with the given id, or null if not found
     */
    protected DialogFragment swapInDialogFragment(String id)
            throws IOException {

        if (store == null || id == null) {
            return null;
        }
        
        DialogFragment df = store.load(id);
        if (df == null) {
            return null;
        }

        registerDialogFragment(df);

        return df;
    }

    /**
     * Gets the DialogFragment with the given id 
     * from the active cache only
     *
     * @return The DialogFragment with the given id, or null if not
     * found
     */
    public DialogFragment findDialogFragmentFromCacheOnly(String id) {
        return super.findDialogFragment(id);
    }

    /**
     * Removes the given DialogFragment from only the active cache
     *
     * @param df The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */    
    protected void removeDialogFragmentFromCache(String id,
                                                 boolean hasTimedOut) {
        DialogFragment removeIt = findDialogFragmentFromCacheOnly(id);
        if (removeIt != null) {
            super.removeDialogFragment(removeIt, hasTimedOut);
        }
    }     


    /**
     * Removes the given DialogFragment from only the active cache
     *
     * @param df The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */    
    protected void removeDialogFragmentFromCache(DialogFragment df,
                                                 boolean hasTimedOut) {
        super.removeDialogFragment(df, hasTimedOut);
    }     

    /**
     * Sets the repairDuringFailure property of this session manager.
     *
     * @param isRepairDuringFailure The value of the repairDuringFailure
     * property
     */
    public void setRepairDuringFailure(boolean isRepairDuringFailure) {
        this.isRepairDuringFailure = isRepairDuringFailure;
    }

    /**
     * @return True if this session manager should participate in a repair
     * during failure, and false otherwise
     */
    public boolean isRepairDuringFailure() {
        return isRepairDuringFailure;
    }

    public DialogFragmentStore getDialogFragmentStore() {
        return store;
    }
    
    public void setDialogFragmentStore(DialogFragmentStore store) {
        this.store = store;
    }
    
    @Override 
    protected void init() {
       super.init();
       // perform init for this class here
    }
    
    @Override
    protected void doStart() {
        super.doStart(); 
        startReaperThread();
        // perform start for this class here
    }
    
    @Override
    protected void doStop() {
        super.doStop();
        stopReaperThread();
        // perform stop for this class here
    }

    /**
     * Starts the background reaper thread.
     */
    private void startReaperThread() {
        reaperThreadDone = false;
        reaperThread.start();
    }

    /**
     * Stops the background reaper thread.
     */
    private void stopReaperThread() {
        reaperThreadDone = true;
        reaperThread.interrupt();
        try {
            reaperThread.join();
        } catch (InterruptedException e) {
            ;
        }
    }

    /**
     * Reaper thread responsible for purging expired dialog fragments
     */
    private class ReaperThread implements Runnable {

        public void run() {
            while (!reaperThreadDone) {
                try {
                    Thread.sleep(REAPER_THREAD_SLEEP_TIME);
                } catch (InterruptedException e) {
                    reaperThreadDone = true;
                }
                if (!reaperThreadDone) {
                    if (store != null) {
                        store.processExpires();
                    }
                }
            }
        }
    }

}
