/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.config.LayerHandler;
import com.ericsson.ssa.container.SipBindingResolver;
import com.ericsson.ssa.container.sim.ApplicationDispatcher;
import com.ericsson.ssa.dd.SessionCase;

import org.jvnet.glassfish.comms.security.auth.impl.AuthInfoImpl;

import java.io.UnsupportedEncodingException;

import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.sip.Address;
import javax.servlet.sip.AuthInfo;
import javax.servlet.sip.Parameterable;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.URI;


/**
 * @author ekrigro
 * @reviewed ejoelbi 2006-oct-19
 */
public class SipFactoryImpl implements SipFactory {
    public static final String SIP_CHARSET = "UTF-8";
    public static final String PROTOCOL_LINE = "SIP/2.0";
    public static final String NEW_LINE = "\r\n";
    public static final String SIP_URI_PROTOCOL = "sip";
    public static final String HTTP_URI_PROTOCOL = "http";
    public static final String MAILTO_URI_PROTOCOL = "mailto";
    public static final String SIPS_URI_PROTOCOL = "sips";
    public static final String TEL_URI_PROTOCOL = "tel";
    public static final String SUPPORTED_100REL = "100rel";
    public static final String REMOTE_TARGET = "com.ericsson.ssa.RemoteTarget";
    public static final int START_CSEQ = 1;
    private static AtomicInteger m_GlobalCallID = new AtomicInteger(1);
    private static AtomicLong m_GlobalTag = new AtomicLong(1);
    private static ServiceHandler m_ServiceHandler = null;
    private static final SipFactoryImpl _instance = new SipFactoryImpl();
    private static final Logger _log = Logger.getLogger("SipContainer");
    public static final int SIP_RFC_PORT = 5060;
    public static final int SIPS_RFC_PORT = 5061;

    // initial
    // value
    // needed
    // by
    // JUnit
    // tests
    public static SipFactoryImpl getInstance() {
        return _instance;
    }

    public URI createURI(String uri) throws ServletParseException {
        // search for the protocol
        String protocol = null;
        byte[] buri = null;

        try {
            buri = uri.getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            _log.log(Level.SEVERE, "All JVM must have UTF-8 enc", e);
            throw new ServletParseException("Unsupported URI protocol = " +
                protocol);
        }

        int position = 0;
        int cnt = 0;

        while ((cnt < buri.length) && (buri[cnt] != ':')) {
            cnt++;
        }

        try {
            protocol = new String(buri, position, (cnt - position), SIP_CHARSET);
        } catch (UnsupportedEncodingException e) {
            _log.log(Level.SEVERE, "All JVM must have UTF-8 enc", e);
            throw new ServletParseException("Unsupported URI protocol = " +
                protocol);
        }

        if (protocol.equals(SIP_URI_PROTOCOL)) {
            return new SipURIImpl(SIP_URI_PROTOCOL, buri, cnt + 1);
        }

        if (protocol.equals(TEL_URI_PROTOCOL)) {
            return new TelURLImpl(buri, cnt + 1);
        }

        if (protocol.equals(SIPS_URI_PROTOCOL)) {
            return new SipURIImpl(SIPS_URI_PROTOCOL, buri, cnt + 1);
        }

        if (protocol.equals(MAILTO_URI_PROTOCOL)) {
            return new GeneralURIImpl(MAILTO_URI_PROTOCOL, buri, cnt + 1);
        }

        if (protocol.equals(HTTP_URI_PROTOCOL)) {
            return new GeneralURIImpl(HTTP_URI_PROTOCOL, buri, cnt + 1);
        }

        throw new ServletParseException("Unsupported URI protocol = " +
            protocol);
    }

    public SipURI createSipURI(String user, String host) {
        SipURI uri = new SipURIImpl();
        uri.setHost(host);
        uri.setUser(user);

        return uri;
    }

    public Address createAddress(String sipAddress)
        throws ServletParseException {
        return new AddressImpl(sipAddress);
    }

    public Address createAddress(URI uri) {
        return new AddressImpl(uri);
    }

    public Address createAddress(URI uri, String displayName) {
        Address result = new AddressImpl(uri);
        result.setDisplayName(displayName);

        return result;
    }

    public SipServletRequestImpl createRequest(
        SipApplicationSession appSession, String method, Address from,
        Address to) {
        return createRequestImpl((SipApplicationSessionImpl) appSession,
            method, from, to, false, null);
    }

    private void createSession(SipServletRequestImpl req,
        SipApplicationSessionImpl appSess, String handlerName) {
        // creates a new sip session and dialog structure
        DialogSet ds = new DialogSet(req.getCallId(), req.getFrom());
        req.setDialog(ds.getInitialDialogFragment());

        DialogFragment d = req.getDialog();
        SipSessionBase s = appSess.getSipSessionManager()
                                  .createSipSession(d.getDialogSet(),
                req.getTo(), appSess, handlerName);
        req.setSession(s);
    }

    public SipServletRequestImpl createRequestImpl(
        SipApplicationSessionImpl appSession, String method, Address from,
        Address to, boolean sameCallID, SipServletRequestImpl origRequest) {
        if (!method.equals("ACK") && !method.equals("CANCEL")) {
            Address fromCopy = (Address) from.clone();
            Address toCopy = (Address) to.clone();

            // Remove protection, lock up
            ((AddressImpl) fromCopy).setReadOnly(false);
            ((AddressImpl) toCopy).setReadOnly(false);
            fromCopy.setParameter(AddressImpl.TAG_PARAM, createTag());
            toCopy.removeParameter(AddressImpl.TAG_PARAM);

            // Put protection, lock again
            ((AddressImpl) fromCopy).setReadOnly(true);
            ((AddressImpl) toCopy).setReadOnly(true);

            SipServletRequestImpl req = null;
            String defaultHandler = null;

            if (origRequest == null) {
                URI requestURI = (URI) toCopy.getURI().clone();
                if ("REGISTER".equals(method) && requestURI.isSipURI()) {
                    ((SipURI)requestURI).setUser(null);
                }
                req = new SipServletRequestImpl(method, requestURI,
                        SipFactoryImpl.PROTOCOL_LINE);
            } else {
                req = (SipServletRequestImpl) origRequest.clone();
                req.clearApplicationDispatchers();
                // The clone will copy the session-case. For the new request
                // the session-case should be internal.
                req.setSessionCase(SessionCase.INTERNAL);
                req.removeSystemHeaders();
                defaultHandler = origRequest.getSessionImpl().getHandler();
            }

            // set CallID
            Header callIDHeader = new SingleLineHeader(Header.CALL_ID, true);

            if ((origRequest == null) || !sameCallID) {
                callIDHeader.setValue(createCallID(), false);
            } else {
                callIDHeader.setValue(origRequest.getCallId(), false);
            }

            req.setHeader(callIDHeader);

            Header toHeader = new SingleLineHeader(Header.TO, true);
            toHeader.setAddressValue(toCopy, false);
            req.setHeader(toHeader);

            Header fromHeader = new SingleLineHeader(Header.FROM, true);
            fromHeader.setAddressValue(fromCopy, false);
            req.setHeader(fromHeader);
            req.setMaxForwards(70);

            if (defaultHandler == null) {
                defaultHandler = appSession.getCurrentServlet();
            }

            createSession(req, appSession, defaultHandler);

            // MUST be placed after createSession is called
            // Add the contact header if the request method is one of
            // INVITE, REFER, SUBSCRIBE eller NOTIFY.
            if (isDialogCreational(req.getMethod())) {
                SessionManager.getInstance().addContact(req);
            }

            Header cSeqHeader = new SingleLineHeader(Header.CSEQ, true);
            cSeqHeader.setValue(Integer.toString(START_CSEQ) + " " +
                req.getMethod(), false);
            req.setHeader(cSeqHeader);
            req.setInitial(true);

            List<Layer> layers = LayerHandler.getInstance().getLayers();
            req._applicationStack.addAll(layers);

            // The ApplicationDispatcher is the last layer.
            if (!layers.isEmpty()) {
                Layer layer = layers.get(layers.size() - 1);

                if (layer instanceof ApplicationDispatcher) {
                    ((ApplicationDispatcher) layer).pushServiceDispatcher(req);
                } else {
                    Iterator<Layer> i = layers.iterator();

                    while (i.hasNext()) {
                        layer = i.next();

                        if (layer instanceof ApplicationDispatcher) {
                            ((ApplicationDispatcher) layer).pushServiceDispatcher(req);
                        }
                    }
                }
            }

            return req;
        }

        throw new IllegalArgumentException(
            "ACK and CANCEL is not allowed to create here.");
    }

    public SipServletRequestImpl createRequest(
        SipApplicationSession appSession, String method, URI from, URI to) {
        return createRequest(appSession, method, createAddress(from),
            createAddress(to));
    }

    public SipServletRequestImpl createRequest(
        SipApplicationSession appSession, String method, String from, String to)
        throws ServletParseException {
        return createRequest(appSession, method, createAddress(from),
            createAddress(to));
    }

    public SipServletRequestImpl createRequest(SipServletRequest origRequest,
        boolean sameCallId) {
        SipServletRequestImpl origRequestImpl = (SipServletRequestImpl) origRequest;
        SipServletRequestImpl req = createRequestImpl((SipApplicationSessionImpl) origRequest.getApplicationSession(),
                origRequest.getMethod(), origRequest.getFrom(),
                origRequest.getTo(), sameCallId, origRequestImpl);

        // HG54598, copy the Route header
        Header route = origRequestImpl.getRawHeader(Header.ROUTE);

        if (route != null) {
            req.setHeader((Header) route.clone());
        }

        // HH20252, copy the Contact header
        Header contact = origRequestImpl.getRawHeader(Header.CONTACT);

        if ((contact != null) && req.getMethod().equals("REGISTER")) {
            req.setHeader((Header) contact.clone());
        }

        return req;
    }

    public SipApplicationSessionImpl createApplicationSession() {
        /*
         * This method should never be called and therefore always returns null.
         * We only call createApplicationSession() on the context specific
         * facade of this singleton, which delegates the creation of the
         * SipApplicationSession to the context's associated session manager
         */
        return null;
    }

    /**
     * Returns a unique to tag.
     */
    public String createTag() {
        StringBuilder sb = new StringBuilder(
                Long.toString(System.currentTimeMillis(),
                36)).append('-').append(Long.toString(
                m_GlobalTag.getAndIncrement(), 36));

        return sb.toString();
    }

    /**
     * Returns a unique Call-ID.
     */
    public String createCallID() {
        int id = m_GlobalCallID.getAndIncrement();
        long r = Math.abs(new Random().nextLong());
        StringBuilder sb = new StringBuilder();
        sb.append(SipBindingResolver.instance().getCurrentPublicHost());
        sb.append('_').append(id).append('_').append(r);

        return sb.toString();
    }

    public Parameterable createParameterable(String s) {
        return ParameterableImpl.parse(s);
    }

    public ServiceHandler getServiceHandler() {
        return m_ServiceHandler;
    }

    public void setServiceHandler(ServiceHandler s) {
        m_ServiceHandler = s;
    }

    /**
     * Check if the Contact header can be added (mandatory or optional in RFC) to
     * the response, according to the followning RFC's
     * <ul>
     * <li>RFC 3311</li>
     * <li>RFC 3262</li>
     * <li>RFC 3428</li>
     * <li>RFC 3261</li>
     * <li>RFC 3265</li>
     * <li>RFC 3515 </li>
     * </ul>
     *
     * @param resp
     *           The status code and SIP method will be used when performing the
     *           check.
     * @return true if the Contact header can be added to the response.
     */
    public static HeaderRequirement getContactRequirement(
        SipServletResponse resp) {
        int status = resp.getStatus();
        String method = resp.getMethod();

        if ("BYE".equals(method)) // RFC 3261
         {
            if (((status >= 300) && (status < 400)) || (status == 485)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("INVITE".equals(method)) // RFC 3261
         {
            if ((status > 100) && (status < 200)) {
                return HeaderRequirement.OPTIONAL;
            }

            if ((status >= 200) && (status < 300)) {
                return HeaderRequirement.MANDATORY;
            }

            if (((status >= 300) && (status < 400)) || (status == 485)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("OPTION".equals(method) || "REGISTER".equals(method)) // RFC 3261
         {
            if (((status >= 200) && (status < 400)) || (status == 485)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("SUBSCRIBE".equals(method)) // RFC-3265
         {
            if ((status >= 100) && (status < 200)) {
                return HeaderRequirement.OPTIONAL;
            }

            if ((status >= 200) && (status < 400)) {
                return HeaderRequirement.MANDATORY;
            }

            if (status == 485) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("NOTIFY".equals(method)) // RFC-3265
         {
            if ((status >= 100) && (status < 300)) {
                return HeaderRequirement.OPTIONAL;
            }

            if ((status >= 300) && (status < 400)) {
                return HeaderRequirement.MANDATORY;
            }

            if (status == 485) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("REFER".equals(method)) // RFC 3515
         {
            if ((status >= 200) && (status < 300)) {
                return HeaderRequirement.MANDATORY;
            }

            if ((status >= 300) && (status < 700)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("UPDATE".equals(method)) // RFC 3311
         {
            if ((status >= 100) && (status < 200)) {
                return HeaderRequirement.OPTIONAL;
            }

            if ((status >= 200) && (status < 300)) {
                return HeaderRequirement.MANDATORY;
            }

            if (((status >= 300) && (status < 400)) || (status == 485)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("MESSAGE".equals(method)) // RFC 3428
         {
            if (((status >= 300) && (status < 400)) || (status == 485)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        if ("PRACK".equals(method)) // RFC 3262
         {
            if (((status >= 300) && (status < 400)) || (status == 485)) {
                return HeaderRequirement.OPTIONAL;
            }
        }

        return HeaderRequirement.NOT_APPLICAPLE;
    }

    public static boolean isDialogCreational(String method) {
        return method.equals("SUBSCRIBE") || method.equals("INVITE") ||
        method.equals("REFER") || method.equals("NOTIFY");
    }

    public static boolean isContactMandatory(SipServletRequest req) {
        return isDialogCreational(req.getMethod()) ||
        req.getMethod().equals("UPDATE");
    }

    public AuthInfo createAuthInfo() {
        return new AuthInfoImpl();
    }
}
