/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.config.LayerHandler;
import com.ericsson.ssa.config.SipRequestDispatcher;
import com.ericsson.ssa.container.NetworkManager;
import com.ericsson.ssa.dd.SessionCase;
import com.ericsson.ssa.sip.PathNode.Type;

import org.jvnet.glassfish.comms.security.auth.impl.AuthInfoImpl;
import org.jvnet.glassfish.comms.security.auth.impl.ClientDigestCreator;

import java.io.BufferedReader;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.UnsupportedEncodingException;

import java.nio.ByteBuffer;

import java.security.cert.X509Certificate;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletInputStream;
import javax.servlet.sip.Address;
import javax.servlet.sip.AuthInfo;
import javax.servlet.sip.B2buaHelper;
import javax.servlet.sip.Proxy;
import javax.servlet.sip.SipApplicationRoutingDirective;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.TooManyHopsException;
import javax.servlet.sip.URI;


/**
 * @author ekrigro
 * @reviewed ejoelbi 2006-oct-19
 */
public class SipServletRequestImpl extends SipServletMessageImpl
    implements SipServletRequest, Externalizable {
    /**
     * Comment for <code>serialVersionUID</code>
     */
    public static final String CLIENT_CERT = "javax.servlet.request.X509Certificate";
    private static final long serialVersionUID = 3257849874501548343L;
    private static final Set<String> _disallowedRecordRouteURIParams = new HashSet<String>();

    static {
        _disallowedRecordRouteURIParams.add("transport");
        _disallowedRecordRouteURIParams.add("lr");
        _disallowedRecordRouteURIParams.add("ttl");
        _disallowedRecordRouteURIParams.add("user");
        _disallowedRecordRouteURIParams.add("method");
        _disallowedRecordRouteURIParams.add("maddr");
    }

    private URI _requestURI;
    private boolean _isInitial = false;
    private ProxyContext _proxyContext = null;
    private boolean m_IsInPath = false;
    private SessionCase m_SessionCase = SessionCase.INTERNAL;
    private SipServletRequestImpl _transactionRequest = null;
    protected boolean _IsRecordRouteIndicated = false;
    protected boolean _IsPathIndicated = false;
    private int _sentFinalResponseCode = -1;
    private int _rseq = 0;
    protected List<Dispatcher> _retransmission_applicationStack = null;
    private Address _poppedRoute = null;
    private Object _stateInfo = null;
    private SipServletRequestImpl _linkedRequest = null;
    private B2buaHelper _b2buahelper = null;
    private Logger _logger = (Logger) Logger.getLogger("SipContainer");
    private boolean _supervised = true;
    private Map<String, String> _recordRouteURIParams = new HashMap<String, String>();
    private Map<String, String> _pathURIParams = new HashMap<String, String>();
    private Map<String, String> _requestParams = new HashMap<String, String>();

    private boolean requestSent = false;

    public SipServletRequestImpl() {
        super();
        _messageType = SipMessageType.SipRequest;
    }

    public SipServletRequestImpl(String method, URI requestURI, String protocol) {
        super(method, protocol);
        _requestURI = requestURI;
        _messageType = SipMessageType.SipRequest;
    }

    public URI getRequestURI() {
        return _requestURI;
    }

    /**
     * Returns that a Record Route header should be added to this message
     *
     * @return whether a Record Route header should be added
     */
    public boolean isRecordRouteIndicated() {
        return _IsRecordRouteIndicated;
    }

    /**
     * States that a Record Route header should be added to this message
     */
    public void indicateRecordRoute() {
        _IsRecordRouteIndicated = true;
    }

    /**
     * Returns that a Path header should be added to this message
     *
     * @return whether a Path header should be added
     */
    public boolean isPathIndicated() {
        return _IsPathIndicated;
    }

    /**
     * States that a Path header should be added to this message
     */
    public void indicatePath() {
        _IsPathIndicated = true;
    }

    public void setRequestURI(URI requestURI) {
        _requestURI = requestURI;
    }

    // TEST for user centric
    public void writeExternal(ObjectOutput output) throws IOException {
        try {
            // Write the req uri
            output.writeObject(_requestURI); // TODO check write External on URI
                                             // classes
                                             // Let the Message class handle it's properties

            super.writeExternal(output);
        } catch (Exception ignore) {
        }
    }

    public void readExternal(ObjectInput input) throws IOException {
        try {
            _requestURI = (URI) input.readObject();
            super.readExternal(input);
        } catch (Exception ignore) {
        }
    }

    public Type isDirection() {
        if (isInitial()) {
            _Type = Type.Caller;

            return _Type;
        }

        return super.isDirection();
    }

    public void pushRoute(SipURI route) {
        Address routeAddress = _sf.createAddress(route);
        pushRoute(routeAddress);
    }

    public void pushRoute(Address routeAddress) {
        if (!isInitial()) {
            throw new IllegalStateException(
                "It's not allowed to call pushRoute for subsequent requests.");
        }

        if (headerMap.containsKey(Header.ROUTE)) {
            Header header = headerMap.get(Header.ROUTE);
            header.setAddressValue(routeAddress, true);
        } else {
            Header header = Header.createFormated(Header.ROUTE, this);
            header.setAddressValue(routeAddress, true);
            headerMap.put(Header.ROUTE, header);
        }
    }

    public void pushPath(Address pathAddress) {
        if (!getMethod().equals("REGISTER")) {
            throw new IllegalStateException(
                "It's not allowed to call pushPath for non-REGISTER requests.");
        }

        //Verify that the REGISTER request has a Supported header containing path
        boolean pathSupported = false;
        ListIterator<String> supportedList = getHeaders(Header.SUPPORTED);
        String supported = null;

        while (supportedList.hasNext()) {
            supported = (String) supportedList.next();

            if (supported.equals("path")) {
                pathSupported = true;
            }
        }

        if (!pathSupported) {
            throw new IllegalStateException(
                "The REGISTER request is missing required option path in Supported header.");
        }

        if (headerMap.containsKey(Header.PATH)) {
            Header header = headerMap.get(Header.PATH);
            header.setAddressValue(pathAddress, true);
        } else {
            Header header = Header.createFormated(Header.PATH, this);
            header.setAddressValue(pathAddress, true);
            headerMap.put(Header.PATH, header);
        }
    }

    public int getMaxForwards() {
        Header header = headerMap.get(Header.MAX_FORWARDS);

        if (header == null) {
            return -1;
        }

        String mf = header.getValue();

        if (mf == null) {
            return -1;
        }

        return Integer.parseInt(mf);
    }

    public void setMaxForwards(int nr) {
        // TODO Could be optimized since no pretty print is required in this
        // case
        setHeader(Header.MAX_FORWARDS, String.valueOf(nr));
    }

    public void setInitial(boolean isInitial) {
        _isInitial = isInitial;
    }

    public boolean isInitial() {
        return _isInitial;
    }

    public ServletInputStream getInputStream() throws IOException {
        // TODO Auto-generated method stub
        return null;
    }

    public BufferedReader getReader() throws IOException {
        // TODO Auto-generated method stub
        return null;
    }

    public void setInApplicationPath(boolean inPath) {
        m_IsInPath = inPath;
    }

    public boolean isInApplicationPath() {
        return m_IsInPath;
    }

    public ProxyContext getProxyContext() {
        return _proxyContext;
    }

    public void setProxyContext(ProxyContext proxyContext) {
        _proxyContext = proxyContext;
    }

    public Proxy getProxy() throws TooManyHopsException {
        return getProxy(true);
    }

    public Proxy getProxy(boolean create) throws TooManyHopsException {
        if (!isInitial()) {
            throw new IllegalStateException("Applications should not attempt " +
                "to explicitly proxy subsequent requests");
        }

        if (_b2buahelper != null) {
            throw new IllegalStateException("Only allowed to be B2bua");
        }

        if (headerMap.get(Header.MAX_FORWARDS).getValue().equals("0")) {
            throw new TooManyHopsException();
        }

        Proxy proxy = null;

        if (getProxyContext() == null) {
            if (isInApplicationPath()) {
                // support of SSA1.1, 10.2.3 Sending Responses (as a Proxy),
                // mandates to first fetch proxy before sending response
                throw new IllegalStateException("Only allowed to be UA");
            }

            if (create) {
                ProxyImpl proxyImpl = new ProxyImpl(getApplicationSessionImpl(),
                        this);
                setProxyContext(new ProxyContext(proxyImpl, getSessionImpl()));
                proxy = proxyImpl.getFacade(this);
            }
        } else {
            proxy = getProxyContext().getProxy().getFacade(this);
        }

        return proxy;
    }

    public SipServletResponse createResponse(int code)
        throws IllegalArgumentException, IllegalStateException {
        checkResponseCode(code);

        return createResponseImpl(code);
    }

    public SipServletResponseImpl createTerminatingResponse(int code) {
        // Don't send a terminating response, if the request was an 'ACK',
        // see TR HH52078
        if (getMethod().equals("ACK")) {
            return null;
        }

        SipServletResponseImpl resp = createResponseImpl(code);

        if (!resp.hasToTag()) {
            resp.createTag(Header.TO);
        }

        // Temp fix which sets the remote target to avoid nullpointer in
        // NetworkManager
        // TODO Investigate if we should add the ResolverManager to the
        // transactionStack if
        // a response needs to be sent on the way up in the stack from a layer
        // below ResolverManager
        // (when ResolverManager is not yet pushed on the transactionStack)
        // If resolverManager is added it can handle reconnections through
        // firewalls (received/rport param)
        //
        resp.setRemote(getRemote());
        resp.setRequest(this); //To satisfy Proxy component if faled to send...
                               //TODO check if final proposal and why not in the createTerminatingResponse(int,String) method?
                               //Same for set remote!!! - Stoffe

        return resp;
    }

    public SipServletResponseImpl create100TryingResponse() {
        return createResponseImpl(100);
    }

    public SipServletResponseImpl createResponseImpl(int code) {
        return populateResponse(new SipServletResponseImpl(this,
                this.getProtocol(), code));
    }

    public SipServletResponse createResponse(int code, String phrase)
        throws IllegalArgumentException, IllegalStateException {
        checkResponseCode(code);

        return createResponseImpl(code, phrase);
    }

    public SipServletResponseImpl createTerminatingResponse(int code,
        String phrase) {
        // Don't send a terminating response, if the request was an 'ACK',
        // see TR HH52078
        if (getMethod().equals("ACK")) {
            return null;
        }

        SipServletResponseImpl resp = createResponseImpl(code, phrase);

        if (!resp.hasToTag()) {
            resp.createTag(Header.TO);
        }

        return resp;
    }

    public SipServletResponseImpl createResponseImpl(int code, String phrase) {
        return populateResponse(new SipServletResponseImpl(this,
                this.getProtocol(), code, phrase));
    }

    public SipServletRequest createCancel() throws IllegalStateException {
        return createCancelImpl();
    }

    public SipServletRequestImpl createCancelImpl()
        throws IllegalStateException {
        if (isInitial() && getMethod().equals("INVITE")) {
            Address fromCopy = (Address) ((AddressImpl) getFrom()).clone(true,
                    true);
            Address toCopy = (Address) ((AddressImpl) getTo()).clone(true, true);
            SipServletRequestImpl req = new SipServletRequestImpl("CANCEL",
                    getRequestURI(), SipFactoryImpl.PROTOCOL_LINE);

            // set To
            Header toHeader = new SingleLineHeader(Header.TO, true);
            toHeader.setAddressValue(toCopy, false);
            req.setHeader(toHeader);

            // set From
            Header fromHeader = new SingleLineHeader(Header.FROM, true);
            fromHeader.setAddressValue(fromCopy, false);
            req.setHeader(fromHeader);

            // set Max-Forwards
            Header maxForwardsHeader = new SingleLineHeader(Header.MAX_FORWARDS,
                    false);
            maxForwardsHeader.setValue("70", false);
            req.setHeader(maxForwardsHeader);

            // set CallID
            Header callIDHeader = new SingleLineHeader(Header.CALL_ID, true);
            callIDHeader.setValue(getCallId(), false);
            req.setHeader(callIDHeader);

            // set CSeq
            Header cSeqHeader = new SingleLineHeader(Header.CSEQ, true);
            cSeqHeader.setValue(Integer.toString(getCSeqNumber()) + " CANCEL",
                false);
            req.setHeader(cSeqHeader);

            // copy ROUTE of INVITE...
            Header route = getRawHeader(Header.ROUTE);

            if (route != null) {
                req.setHeader((Header) route.clone());
            }

            // update new request with session...
            req.setSession(getSessionImpl());
            // ...and dialog
            req.setDialog(getDialog());
            // find out direction
            req.setDirection(isDirection());
            // subsequent request
            req.setInitial(false);

            // Fetch the Application stack
            List<Layer> layers = LayerHandler.getInstance().getLayers();
            req._applicationStack.addAll(layers);
            // The destination address, port, and transport for the CANCEL MUST
            // be identical to those used to send the original request.
            req.setRemote(getRemote());

            return req;
        } else {
            throw new IllegalStateException(
                "Cancel is only allowed for initial INVITE.");
        }
    }

    public String getParameter(String name) {
        return _requestParams.get(name);
    }

    public Enumeration getParameterNames() {
        final Iterator<String> it = _requestParams.keySet().iterator();

        return new Enumeration<String>() {
                public boolean hasMoreElements() {
                    return it.hasNext();
                }

                public String nextElement() {
                    return it.next();
                }
            };
    }

    public String[] getParameterValues(String name) {
        return new String[] { _requestParams.get(name) };
    }

    public Map getParameterMap() {
        return Collections.unmodifiableMap(_requestParams);
    }

    public String getScheme() {
        return _requestURI.getScheme();
    }

    public String getServerName() {
        // Use information from getLocalAddr()....
        return getLocalAddr();
    }

    public int getServerPort() {
        // Use information from getLocalPort()....
        return getLocalPort();
    }

    public String getRemoteHost() {
        // Use information from getRemoteAddr()....
        return getRemoteAddr();
    }

    /**
     * Removes an attribute from the message.
     * This method should be moved to SipServletMessageImpl when SipServletMessage
     * is modified to contain the removeAttribute method.
     */
    public void removeAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }

        if (attrib != null) {
            attrib.remove(name);
        }
    }

    public Locale getLocale() {
        // Use information from getAcceptLanguage()....
        return getAcceptLanguage();
    }

    public Enumeration<Locale> getLocales() {
        // Use information from getAcceptLanguages()....
        final Iterator<Locale> it = getAcceptLanguages();

        return new Enumeration<Locale>() {
                public boolean hasMoreElements() {
                    return it.hasNext();
                }

                public Locale nextElement() {
                    return it.next();
                }
            };
    }

    public RequestDispatcher getRequestDispatcher(String name) {
        return new SipRequestDispatcher(name,
            SipFactoryImpl.getInstance().getServiceHandler());
    }

    @Deprecated
    public String getRealPath(String arg0) {
        // Shall return null in a SIP context
        return null;
    }

    public String getLocalName() {
        // Use information from getLocalAddr()....
        return getLocalAddr();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(_method);
        sb.append(' ');
        sb.append(_requestURI);
        sb.append(' ');
        sb.append(SipFactoryImpl.PROTOCOL_LINE);
        sb.append(SipFactoryImpl.NEW_LINE);

        int len = 0;
        byte[] content = null;

        try {
            content = getRawContent();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        if (content != null) {
            len = content.length;
        }

        Header cl = headerMap.get(Header.CONTENT_LENGTH);

        if (cl == null) {
            cl = new SingleLineHeader(Header.CONTENT_LENGTH, false);
            cl.setValue(new StringBuilder().append(len).toString(), false);
            headerMap.put(Header.CONTENT_LENGTH, cl);
        } else {
            cl.setValue(new StringBuilder().append(len).toString(), false);
        }

        Iterator<Header> i = headerMap.values().iterator();

        while (i.hasNext()) {
            sb.append(i.next().toString());
        }

        sb.append(SipFactoryImpl.NEW_LINE);

        if ((content != null) && (content.length > 0)) {
            sb.append(new String(content));
        }

        return sb.toString();
    }

    ByteBuffer toBufferFirstline(ByteBuffer bb)
        throws UnsupportedEncodingException {
        bb.put(_method.getBytes());
        bb.put((byte) 0x20);
        bb.put(_requestURI.toString().getBytes(SipFactoryImpl.SIP_CHARSET));
        bb.put((byte) 0x20);
        bb.put(SipFactoryImpl.PROTOCOL_LINE.getBytes());
        bb.put(SipFactoryImpl.NEW_LINE.getBytes());

        return bb;
    }

    public String toDebugString() {
        StringBuilder sb = new StringBuilder(_method);
        sb.append(" ");
        sb.append(_requestURI);
        sb.append(" ");
        sb.append(super.toDebugString());

        return sb.toString();
    }

    private SipServletResponseImpl populateResponse(
        SipServletResponseImpl response) {
        if (getMethod().equals("ACK")) {
            throw new IllegalStateException("Not allowed to create response " +
                response.getStatus() + " for ACK with callId = " + getCallId() +
                ", from = " + getFrom() + ", to = " + getTo());
        }

        Iterator<Header> i = headerMap.values().iterator();

        while (i.hasNext()) {
            Header next = i.next();

            if (next.equals(Header.VIA)) {
                response.headerMap.put(Header.VIA, (Header) next.clone());
            } else if (next.equals(Header.FROM)) {
                response.headerMap.put(Header.FROM, (Header) next.clone());
            } else if (next.equals(Header.TO)) {
                response.headerMap.put(Header.TO, (Header) next.clone());
            } // else if( next.equals( Header.P_ASSERTED_ID ) )
              // response.headerMap.put( Header.P_ASSERTED_ID,
              // (Header)next.clone()
              // );

            else if (next.equals(Header.CSEQ)) {
                response.headerMap.put(Header.CSEQ, (Header) next.clone());
            } else if (next.equals(Header.CALL_ID)) {
                response.headerMap.put(Header.CALL_ID, (Header) next.clone());
            } else if (next.equals(Header.RECORD_ROUTE)) {
                response.headerMap.put(Header.RECORD_ROUTE,
                    (Header) next.clone());
            }
        }

        // Dialog must be set before setting Contact
        response._dialog = _dialog;

        if (SipFactoryImpl.getContactRequirement(response) != HeaderRequirement.NOT_APPLICAPLE) {
            SessionManager.getInstance().addContact(response);
        }

        response._method = _method;
        response._transactionStack.addAll(_transactionStack);
        response._sf = _sf;
        response._local = _local;
        // response._remote = _remote; ........ no longer do this, HF85359
        response._session = _session;
        response.setDirection(isDirection());
        response.setBeKey(getBeKey());

        // Set Version information in Server header
        String version = System.getProperty("sip.module.version");

        if (version != null) {
            response.addHeader(Header.SERVER, version);
        }

        // is this a feature of JSR 289 10.2.3 Sending Responses?
        if (isInitial() && (getProxyContext() != null)) {
            getProxyContext().getProxy().setVirtualProxyBranchRequest(response);
        }

        return response;
    }

    public void send() throws IOException {
        if (NetworkManager.getInstance().isRunning() == false) {
            throw new IllegalStateException(
                "Stack is not yet available for sending requests:");
        }

        DialogFragment p = getDialog();
        PathNode uac = null;

        if (isInitial()) {
            uac = new UA(getApplicationSessionImpl(), true);
            // add to application path
            p.addToPath(uac);
            // need to add to transaction path
            pushTransactionDispatcher(uac);
            // when this thread returns it should not continue...
            setSent(true);
        } else {
            // lets build the dispatcher list from the application path
            Iterator<PathNode> iter = null;

            if (isDirection() == Type.Caller) {
                iter = p.getCallee2CallerPath();
            } else if (isDirection() == Type.Callee) {
                iter = p.getCaller2CalleePath();
            } else {
                throw new IOException("Don't know the direction of the flow.");
            }

            // add the dispatchers
            while (iter.hasNext()) {
                uac = iter.next();
                pushApplicationDispatcher(uac);
            }

            // last should not be stored because its the uac
            popDispatcher();
            // need to add to transaction path
            pushTransactionDispatcher(uac);
        }

        uac.send(this);
        requestSent = true;
    }

    public SessionCase getSessionCase() {
        return m_SessionCase;
    }

    public void setSessionCase(SessionCase sessionCase) {
        this.m_SessionCase = sessionCase;
    }

    public void removeSystemHeaders() {
        // remove system headers
        Iterator<String> i = Header.SYSTEM_HEADER_MAP.values().iterator();

        while (i.hasNext()) {
            headerMap.remove(i.next());
        }
    }

    public Object clone() {
        SipServletRequestImpl clone = new SipServletRequestImpl(getMethod(),
                getRequestURI(), getProtocol());
        Iterator<Header> i = headerMap.values().iterator();

        // deep copy
        while (i.hasNext()) {
            Header next = i.next();
            clone.headerMap.put(next.getName(), (Header) next.clone());
        }

        if (attrib != null) {
            if (systemAttrib != null) {
                clone.attrib = new HashMap<String, Object>(systemAttrib);
            } else {
                clone.attrib = new HashMap<String, Object>(attrib.size());
            }

            clone.attrib.putAll(attrib);
        } else if (systemAttrib != null) {
            clone.systemAttrib = new HashMap<String, Object>(systemAttrib.size());
            clone.systemAttrib.putAll(systemAttrib);
        }

        // shallow copy
        clone.m_SessionCase = m_SessionCase;
        clone._content_enc = _content_enc;
        clone._content_byte = _content_byte;
        clone._content_obj = _content_obj;
        clone._roles = _roles;
        clone._remote = _remote;
        clone._local = _local;
        clone._isInitial = _isInitial;
        clone._session = _session;
        clone._dialog = _dialog;
        clone._sf = _sf;
        clone._IsContactIndicated = _IsContactIndicated;
        clone._IsPathIndicated = _IsPathIndicated;
        clone._IsRecordRouteIndicated = _IsRecordRouteIndicated;
        clone._headersComplete = _headersComplete;
        clone._messageComplete = _messageComplete;
        clone.m_IsInPath = m_IsInPath;
        clone._rseq = _rseq;
        clone._fragmentId = _fragmentId;
        clone._supervised = _supervised;
        clone._recordRouteURIParams = _recordRouteURIParams;
        clone._pathURIParams = _pathURIParams;
        clone._requestParams = _requestParams;
        // JSR 289
        clone._poppedRoute = _poppedRoute;
        clone._stateInfo = _stateInfo;
        // find out direction
        clone.setDirection(isDirection());

        // copy transaction and application stacks
        if (_transactionStack != null) {
            clone._transactionStack.addAll(_transactionStack);
        }

        if (_applicationStack != null) {
            clone._applicationStack.addAll(_applicationStack);
        }
        
        clone.setBeKey(getBeKey());

        return clone;
    }

    public void saveRetransmissionApplicationStack() {
        _retransmission_applicationStack = new ArrayList<Dispatcher>(_applicationStack);
    }

    public void restoreRetransmissionApplicationStack() {
        // if null or empty...
        if ((_applicationStack == null) ||
                ((_applicationStack != null) && _applicationStack.isEmpty())) {
            _applicationStack = new ArrayList<Dispatcher>(_retransmission_applicationStack);
        }
    }

    public Dispatcher popDispatcher() {
        int size = _applicationStack.size();

        if (size > 0) {
            return _applicationStack.remove(size - 1);
        }

        return null;
    }

    public Dispatcher peekDispatcher() {
        int size = _applicationStack.size();

        if (size > 0) {
            return _applicationStack.get(size - 1);
        }

        return null;
    }

    /**
     * Returns the request that should be used by the response when internally
     * traversing the path of the transaction or null if transaction path is
     * empty.
     *
     * @return the request used by the response when traversing the transaction
     *         path
     */
    public SipServletRequestImpl getTransactionRequest() {
        return _transactionRequest;
    }

    /**
     * Set the transaction request used by the response when travsersing the path
     * of the transaction.
     *
     * @param request
     *        the request mapped to the response
     */
    public void setTransactionRequest(SipServletRequestImpl request) {
        _transactionRequest = request;
    }

    public void setFragmentId(String id) {
        _fragmentId = id;
    }

    /**
     * This method will be called by the reponse object at send operation. Sets
     * the indicator status code.
     *
     * @param status
     *        the indicator
     */
    public void setSentFinalResponse(int statusCode) {
        _sentFinalResponseCode = statusCode;
    }

    public boolean isSentFinalResponse2xx() {
        return (_sentFinalResponseCode != -1)
        ? ((_sentFinalResponseCode / 100) == 2) : false;
    }

    private boolean isValidStatusCode(int statusCode) {
        if (_sentFinalResponseCode != -1) {
            if ((_sentFinalResponseCode / 100) == 2) {
                if ((statusCode / 100) != 2) {
                    // already sent 2xx final response,
                    // can't send non-2xx after...
                    return false;
                }

                // it's ok to send multiple 2xx with different to-tag.
            } else {
                // already sent non-2xx final response
                return false;
            }
        }

        return true;
    }

    /*
     * This method will check for: 1. If we have already sent a final response
     * throw an IllegalStateException 2. If the status code out of range SIP
     * codes throw an IllegalArgumentException
     */
    private void checkResponseCode(int statusCode)
        throws IllegalArgumentException, IllegalStateException {
        if (!isValidStatusCode(statusCode)) {
            throw new IllegalStateException(
                "The request has already responded with a final response.");
        }

        if ((statusCode < 100) || (statusCode > 699)) {
            throw new IllegalArgumentException("Invalid SIP status code.");
        }
    }

    /**
     * Will return true for an INVITE request which support 100rel extension in a
     * Supported or Require header otherwise false.
     *
     * @return true for an INVITE request which support 100rel extension in a
     *         Supported or Require header otherwise false.
     */
    public boolean is100RelSupportedOrRequire() {
        if (getMethod().equals("INVITE")) {
            ListIterator supportedList = getHeaders(Header.SUPPORTED);
            String supported = null;

            while (supportedList.hasNext()) {
                supported = (String) supportedList.next();

                if (supported.equals(SipFactoryImpl.SUPPORTED_100REL)) {
                    return true;
                }
            }

            ListIterator requireList = getHeaders(Header.REQUIRE);
            String require = null;

            while (requireList.hasNext()) {
                require = (String) requireList.next();

                if (require.equals(SipFactoryImpl.SUPPORTED_100REL)) {
                    return true;
                }
            }
        }

        return false;
    }

    public int getAndIncrementRSeq() {
        return ++_rseq;
    }

    public void setPoppedRoute(Address route) {
        _poppedRoute = route;
    }

    public Address getPoppedRoute() {
        if (_poppedRoute == null) {
            return null;
        }

        return (Address) _poppedRoute.clone();
    }

    public void setStateInfo(Object stateInfo) {
        _stateInfo = stateInfo;
    }

    public Object getStateInfo() {
        return _stateInfo;
    }

    public void setRoutingDirective(SipApplicationRoutingDirective directive,
        SipServletRequest origRequest) {
        // TODO Auto-generated method stub
    }

    public SipServletRequestImpl getLinkedRequest() {
        return _linkedRequest;
    }

    public void setLinkedRequest(SipServletRequestImpl linkedRequest) {
        _linkedRequest = linkedRequest;
    }

    public B2buaHelper getB2buaHelper() {
        if (_proxyContext != null) {
            throw new IllegalStateException("Only allowed to be proxy");
        }

        if (_b2buahelper == null) {
            _b2buahelper = new B2buaHelperImpl(this);
        }

        return _b2buahelper;
    }

    public void setB2buaHelper(B2buaHelper bua) {
        _b2buahelper = bua;
    }

    /**
     * Gets the supervise state of this request.
     * @return
     */
    public boolean getSupervised() {
        return _supervised;
    }

    /**
     * Sets the supervised state of this request.
     * @param supervised
     */
    public void setSupervised(boolean supervised) {
        _supervised = supervised;
    }

    public String getPathURIParam(String name) {
        return _pathURIParams.get(name);
    }

    public Iterator getPathURIParamNames() {
        return _pathURIParams.keySet().iterator();
    }

    public void removePathURIParam(String name) {
        _pathURIParams.remove(name);
    }

    public void setPathURIParam(String name, String value) {
        // Assume the same parameters are disallowed as for Record-Route
        if (!_disallowedRecordRouteURIParams.contains(name)) {
            _pathURIParams.put(name, value);
        } else {
            throw new IllegalArgumentException("Not allowed to set parameter '" +
                name + "' on Path URI.");
        }
    }

    public String getRecordRouteURIParam(String name) {
        return _recordRouteURIParams.get(name);
    }

    public Iterator getRecordRouteURIParamNames() {
        return _recordRouteURIParams.keySet().iterator();
    }

    public void removeRecordRouteURIParam(String name) {
        _recordRouteURIParams.remove(name);
    }

    public void setRecordRouteURIParam(String name, String value) {
        if (!_disallowedRecordRouteURIParams.contains(name)) {
            _recordRouteURIParams.put(name, value);
        } else {
            throw new IllegalArgumentException("Not allowed to set parameter '" +
                name + "' on Record-Route URI.");
        }
    }

    /**
     * Used on incoming requests to populate the parameters from the popped Route header
     * @param uri the URI to extract parameters from
     */
    public void setRecordRouteURI(URIImpl uri) {
        Iterator<String> iter;

        if (uri instanceof SipURIImpl) {
            iter = ((SipURIImpl) uri).getParameterNames();
        } else if (uri instanceof TelURLImpl) {
            iter = ((TelURLImpl) uri).getParameterNames();
        } else {
            return;
        }

        while (iter.hasNext()) {
            String name = iter.next();
            String value = uri.getParameter(name);

            if (!_disallowedRecordRouteURIParams.contains(name)) {
                // Put parameters in _requestParams map
                _requestParams.put(name, value);
            }
        }
    }

    /**
     * Used on incoming requests to populate the parameters from the request URI
     * @param uri the URI to extract parameters from
     */
    public void setRequestParams(URIImpl uri) {
        Iterator<String> iter;

        if (uri instanceof SipURIImpl) {
            iter = ((SipURIImpl) uri).getParameterNames();
        } else if (uri instanceof TelURLImpl) {
            iter = ((TelURLImpl) uri).getParameterNames();
        } else {
            return;
        }

        while (iter.hasNext()) {
            String name = iter.next();
            String value = uri.getParameter(name);
            // Put parameters in _requestParams map
            _requestParams.put(name, value);
        }
    }

    @Override
    public void setCertificate(X509Certificate[] cert) {
        super.setCertificate(cert);

        if (systemAttrib == null) {
            systemAttrib = new HashMap<String, Object>();
        }

        this.systemAttrib.put(CLIENT_CERT, cert);
    }

    public void addAuthHeader(SipServletResponse response, AuthInfo info) {
        if (response != null) {
            ClientDigestCreator cdc = new ClientDigestCreator();
            cdc.createDigest(info, this, response);
        }
    }

    public void addAuthHeader(SipServletResponse response, String username,
        String password) {
        AuthInfo ai = new AuthInfoImpl();
        //  ai.addAuthInfo(response.getStatus(), username, password);
        //todo - to pass realm name as a parameter.
        addAuthHeader(response, ai);
    }

    @Override
    public boolean isCommitted() {
        if (isDirection() == Type.Callee) { // incoming request
            if (_sentFinalResponseCode != -1) {
                return true;
            }
        } else { // outgoing request
            if (requestSent) {
                return true;
            }
            if (_sentFinalResponseCode == 408)
                return true;
        }
        return false;
    }

}
