/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.sip.Address;
import javax.servlet.sip.SipSession;
import javax.servlet.sip.SipSessionActivationListener;
import javax.servlet.sip.SipSessionEvent;
import javax.servlet.sip.SipSessionListener;
import javax.servlet.sip.URI;

import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.dd.SipApplicationListeners;
import com.ericsson.ssa.sip.PathNode.Type;


public class SipSessionDialogImpl extends SipSessionImplBase
    implements Serializable {
    private static LogUtil logger = LogUtil.SIP_LOGGER;
    private static final long serialVersionUID = 3762535598732096310L;
    private transient ArrayList<SipSessionListener> sessionListeners = null;
    private Map<String, Object> sessionAttributeMap;
    private SipApplicationSessionImpl sipApplicationSession = null;
    private SipSessionManager sipSessionManager = null;
    private String appId = null;
    private String sasId = null;
    private String handler = null;
    private Type type = null;
    private String id; // final
    protected String description;
    private Address to; // Final
    private long creationDate = new Date().getTime(); // Final
    private long lastAccessedDate;
    private String dialogFragmentId; // Final
    private boolean isDerived = false;
    private AtomicInteger cSeq = null;
    private URI remoteTarget = null;
    private boolean is1xxReliableOngoing = false;
    private boolean is1xxReliableSDP = false;
    private boolean updateOngoing = false;
    private SipSession.State sessionState = State.INITIAL;
    private transient SipSessionEvent sipSessionEvent;

    // JSR289
    private String routingRegion = null;
    private URI subscriberURI = null;
    private String linkedSipSessionId = null;

    /**
     * This boolean controls the values that getLocalParty() and
     * getRemoteParty() should return. It is false by default, which provides
     * the correct behaviour for UAC and proxy SipSessions. When an application
     * has decided to take the UAS role this attribute is set to true for the
     * SipSession, causing getLocalParty() and getRemoteParty() to respond
     * differently.
     */
    private boolean swapLocalRemote = false;

    public SipSessionDialogImpl(SipSessionManagerBase manager, DialogSet set,
        Address to, SipApplicationSessionImpl appSession, String handler,
        Type type) {
        super(set);
        sipSessionManager = manager;
        this.appId = manager.getApplicationId();
        // remove to-tag
        this.to = (Address) ((AddressImpl) to).clone(false, false);
        ((AddressImpl) this.to).setReadOnly(true);
        sipApplicationSession = appSession;
        sasId = appSession.getId();
        this.handler = handler;
        this.type = type;
        sipSessionEvent = new SipSessionEvent(this);

        // if user agent then a cseq counter is needed, otherwise save memory...
        if (type.equals(Type.Caller) || type.equals(Type.Callee)) {
            createPFieldCSeq();
        }

        // important that id exist before adding to app session...
        id = createID().toString();
        sipApplicationSession.addSession(this);
        initListeners(sipApplicationSession.getSipApplicationListeners());

        description = "SipSession with id " + manager.getApplicationId() + ":" +
            id;
    }

    /**
     * For serialization only.
     */
    public SipSessionDialogImpl() {
        super(null);
    }

    /**
     * Notifies any interested session attributes that this session has been
     * activated.
     */
    public void notifySessionDidActivate() {
        Iterator attrValues = getSessionAttributeMap(false).values().iterator();

        while (attrValues.hasNext()) {
            Object attrValue = attrValues.next();

            if (attrValue instanceof SipSessionActivationListener) {
                ((SipSessionActivationListener) attrValue).sessionDidActivate(sipSessionEvent);
            }
        }
    }

    /**
     * Notifies any interested session attributes that this session is about
     * to be passivated.
     */
    public void notifySessionWillPassivate() {
        Iterator attrValues = getSessionAttributeMap(false).values().iterator();

        while (attrValues.hasNext()) {
            Object attrValue = attrValues.next();

            if (attrValue instanceof SipSessionActivationListener) {
                ((SipSessionActivationListener) attrValue).sessionWillPassivate(sipSessionEvent);
            }
        }
    }

    protected final void notifySessionCreated() {
        ArrayList<SipSessionListener> someSessionListeners = getSipSessionListeners();

        if (someSessionListeners == null) {
            return;
        }

        for (Iterator<SipSessionListener> lIter = someSessionListeners.iterator();
                lIter.hasNext();) {
            SipSessionListener listener = lIter.next();
            listener.sessionCreated(sipSessionEvent);
        }
    }

    protected final void notifySessionDestroyed() {
        ArrayList<SipSessionListener> someSessionListeners = getSipSessionListeners();

        if (someSessionListeners == null) {
            return;
        }

        for (Iterator<SipSessionListener> lIter = someSessionListeners.iterator();
                lIter.hasNext();) {
            SipSessionListener listener = lIter.next();
            listener.sessionDestroyed(sipSessionEvent);
        }
    }

    protected final void initListeners(SipApplicationListeners appListeners) {
        if (appListeners != null) {
            setSipSessionListeners(appListeners.getSipSessionListeners());
        }

        notifySessionCreated();
    }

    protected void setSipSessionListeners(
        ArrayList<SipSessionListener> sipSessionListeners) {
        this.sessionListeners = sipSessionListeners;
    }

    private synchronized ArrayList<SipSessionListener> getSipSessionListeners() {
        if (sessionListeners == null) {
            sessionListeners = getPFieldSipSessionManagerField().getContext()
                                   .getSipApplicationListeners()
                                   .getSipSessionListeners();

            if (sessionListeners == null) {
                logger.warning("SipSessionListener list not found");
            }
        }

        return sessionListeners;
    }

    @Override
    protected void removeMeFromManager() {
        sipSessionManager.removeSipSession(this);
    }

    @Override
    protected Object getFromPFieldSessionAttribute(String name) {
        return getSessionAttributeMap(false).get(name);
    }

    @Override
    protected void addToPFieldSessionAttribute(String name, Object value) {
        getSessionAttributeMap(true).put(name, value);
    }

    @Override
    protected Collection<String> getFromPFieldSessionAttributeNames() {
        return getSessionAttributeMap(false).keySet();
    }

    @Override
    protected void removeFromPFieldSessionAttribute(String name) {
        getSessionAttributeMap(false).remove(name);
    }

    private Map<String, Object> getSessionAttributeMap(boolean create) {
        if (sessionAttributeMap == null) {
            if (create) {
                sessionAttributeMap = new HashMap<String, Object>();
            } else {
                return Collections.emptyMap();
            }
        }

        return sessionAttributeMap;
    }

    @Override
    protected boolean getPField1xxReliableOngoing() {
        return is1xxReliableOngoing;
    }

    @Override
    protected boolean getPField1xxReliableSDP() {
        return is1xxReliableSDP;
    }

    @Override
    protected int incrementAndGetPFieldCSeq() {
        return cSeq.incrementAndGet();
    }

    @Override
    protected boolean hasPFieldCSeq() {
        return cSeq != null;
    }

    @Override
    protected void createPFieldCSeq() {
        cSeq = new AtomicInteger(SipFactoryImpl.START_CSEQ);
    }

    // also give subclasses access cseq
    protected void setPFieldCSeq(int aCseq) {
        cSeq = new AtomicInteger(aCseq);
    }

    public int getPFieldCSeq() {
        if (cSeq == null) return -1;
        return cSeq.get();
    }

    @Override
    protected long getPFieldCreationDate() {
        return creationDate;
    }

    @Override
    protected boolean getPFieldDerived() {
        return isDerived;
    }

    @Override
    protected String getPFieldDialogFragmentId() {
        return dialogFragmentId;
    }

    @Override
    protected String getPFieldHandler() {
        return handler;
    }

    @Override
    protected String getPFieldId() {
        return id;
    }

    @Override
    protected long getPFieldLastAccessedDate() {
        return lastAccessedDate;
    }

    @Override
    protected String getPFieldLinkedSipSessionId() {
        return linkedSipSessionId;
    }

    @Override
    protected String getPFieldRoutingRegion() {
        return routingRegion;
    }

    @Override
    protected URI getPFieldRemoteTarget() {
        return remoteTarget;
    }

    @Override
    protected synchronized SipApplicationSessionImpl getPFieldSipApplicationSession() {
        if (sipApplicationSession == null) {
            sipApplicationSession = (SipApplicationSessionImpl) getPFieldSipSessionManagerField()
                                                                    .findSipApplicationSession(sasId);
        }

        return sipApplicationSession;
    }

    @Override
    protected SipSessionManager getPFieldSipSessionManagerField() {
        if (sipSessionManager == null) {
            sipSessionManager = SipSessionManagerBase.get(appId);

            if (sipSessionManager == null) {
                logger.warning("SipSessionManager " + appId + " not found");
            }
        }

        return sipSessionManager;
    }

    @Override
    protected State getPFieldSessionState() {
        return sessionState;
    }

    @Override
    protected URI getPFieldSubscriberURI() {
        return subscriberURI;
    }

    @Override
    protected boolean getPFieldSwapLocalRemote() {
        return swapLocalRemote;
    }

    @Override
    protected Address getPFieldTo() {
        return to;
    }

    @Override
    protected Type getPFieldType() {
        return type;
    }

    @Override
    protected boolean getPFieldUpdateOngoing() {
        return updateOngoing;
    }

    @Override
    protected void setPField1xxReliableOngoing(boolean is1xxReliableOngoing) {
        this.is1xxReliableOngoing = is1xxReliableOngoing;
    }

    @Override
    protected void setPField1xxReliableSDP(boolean is1xxReliableSDP) {
        this.is1xxReliableSDP = is1xxReliableSDP;
    }

    @Override
    protected void setPFieldCreationDate(long creationDate) {
        this.creationDate = creationDate;
    }

    @Override
    protected void setPFieldDerived(boolean isDerived) {
        this.isDerived = isDerived;
    }

    @Override
    protected void setPFieldDialogFragmentId(String dialogFragmentId) {
        this.dialogFragmentId = dialogFragmentId;
    }

    @Override
    protected void setPFieldHandler(String handler) {
        this.handler = handler;
    }

    @Override
    protected void setPFieldId(String id) {
        this.id = id;
    }

    @Override
    protected void setPFieldLinkedSipSessionId(String linkedSipSessionId) {
        this.linkedSipSessionId = linkedSipSessionId;
    }

    @Override
    protected void setPFieldRemoteTarget(URI remoteContact) {
        this.remoteTarget = remoteContact;
    }

    @Override
    protected void setPFieldRoutingRegion(String routingRegion) {
        this.routingRegion = routingRegion;
    }

    @Override
    protected synchronized void setPFieldSipApplicationSession(
        SipApplicationSessionImpl sipApplicationSession) {
        this.sipApplicationSession = (SipApplicationSessionImpl) sipApplicationSession;
    }

    @Override
    protected void setPFieldSipSessionManager(
        SipSessionManager sipSessionManager) {
        this.sipSessionManager = sipSessionManager;
    }

    @Override
    protected void setPFieldSessionState(State state) {
        this.sessionState = state;
    }

    @Override
    protected void setPFieldSubscriberURI(URI subscriberURI) {
        this.subscriberURI = subscriberURI;
    }

    @Override
    protected void setPFieldSwapLocalRemote(boolean swapLocalRemote) {
        this.swapLocalRemote = swapLocalRemote;
    }

    @Override
    protected void setPFieldTo(Address to) {
        this.to = to;
    }

    @Override
    protected void setPFieldType(Type type) {
        this.type = type;
    }

    @Override
    protected void setPFieldUpdateOngoing(boolean updateOngoing) {
        this.updateOngoing = updateOngoing;
    }

    @Override
    protected void setPFieldLastAccessedDate(long lastAccessedDate) {
        this.lastAccessedDate = lastAccessedDate;
    }

    public void setShouldBePersisted() {
        sipApplicationSession.setSipSessionShouldBePersisted(this);
    }

    public boolean shouldBePersisted() {
        return sipApplicationSession.getPersistentSessionIds()
                                    .contains(this.getId());
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        // FIXME check that we are in confirmed state, otherwise throw IllegalStateException
        out.writeObject(appId);
        out.writeObject(sasId);

        Map<String, Object> sessionAttributes = getSessionAttributeMap(false);
        out.writeInt(sessionAttributes.size());

        for (Map.Entry<String, Object> entry : sessionAttributes.entrySet()) {
            out.writeObject(entry.getKey());
            out.writeObject(entry.getValue());
        }

        out.writeObject(handler);
        out.writeObject(type);
        out.writeObject(id);
        out.writeObject(to);
        out.writeLong(creationDate);
        out.writeLong(lastAccessedDate);
        out.writeObject(dialogFragmentId);
        out.writeBoolean(isDerived);
        out.writeBoolean(swapLocalRemote);

        if (cSeq != null) {
            out.writeInt(cSeq.get());
        } else {
            out.writeInt(-1); // Indicate null cSeq
        }

        out.writeObject(remoteTarget);

        // JSR289
        out.writeObject(subscriberURI);
        out.writeObject(routingRegion);
        out.writeObject(linkedSipSessionId);

        out.writeObject(sessionState);
    }

    private void readObject(ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        // Read the appid, and use it to get the corresponding
        // SipSessionManager and associated context.
        appId = (String) in.readObject();
        sasId = (String) in.readObject();

        int size = in.readInt();

        if (size > 0) {
            sessionAttributeMap = new HashMap<String, Object>();

            for (int i = 0; i < size; i++) {
                sessionAttributeMap.put((String) in.readObject(),
                    in.readObject());
            }
        }

        handler = (String) in.readObject();
        type = (Type) in.readObject();
        id = (String) in.readObject();
        description = "SipSession with id " + appId + ":" + id;
        to = (Address) in.readObject();
        creationDate = in.readLong();
        lastAccessedDate = in.readLong();
        dialogFragmentId = (String) in.readObject();
        isDerived = in.readBoolean();
        swapLocalRemote = in.readBoolean();

        int cseqInt = in.readInt();

        if (cseqInt > -1) {
            // OK cSeq was not null at writeObject()
            cSeq = new AtomicInteger(cseqInt);
        }

        remoteTarget = (URI) in.readObject();

        // JSR289
        subscriberURI = (URI) in.readObject();
        routingRegion = (String) in.readObject();
        linkedSipSessionId = (String) in.readObject();

        sessionState = (State) in.readObject();

        sipSessionEvent = new SipSessionEvent(this);
    }

    public String toString() {
        return description;
    }
}
