/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.persistence;

import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.DialogFragmentManager;
import com.ericsson.ssa.sip.PathNode;
import com.ericsson.ssa.sip.PersistentDialogFragmentManager;
import com.ericsson.ssa.sip.PersistentSipSessionManagerBase;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipSessionBase;
import com.ericsson.ssa.sip.SipSessionDialogImpl;
import com.ericsson.ssa.sip.timer.ServletTimerImpl;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Level;


/**
 * This class contains support methods for persisting SipApplicationSession,
 * SipSession, ServletTimer and DialogFragment objects.
 */
public class PersistenceUtil {
    private static final LogUtil logger = LogUtil.SIP_LOGGER;
    private static PersistenceUtil instance = new PersistenceUtil();

    protected PersistenceUtil() {
    }

    public static PersistenceUtil getInstance() {
        return instance;
    }

    /**
     * Save a SipApplicationSession and all its children (SipSession &
     * ServletTimer).
     *
     * @param sas the SipApplicationSession which shall be saved
     */
    public void saveSipApplicationSessionDeeply(SipApplicationSessionImpl sas) {
        if ((sas != null) && sas.isReplicable()) {
            PersistentSipSessionManagerBase sessionManager = (PersistentSipSessionManagerBase) sas.getSipSessionManager();

            if (logger.isLoggable(Level.FINE)) {
                logger.logMsg(Level.FINE, "Saving SAS: " + sas);
            }

            if (sas.shouldBePersisted()) {
                if (sas.getAndResetApplicationLockUsed()) {
                    synchronized (sas.getSasApplicationLock()) {
                        doSave(sas, sessionManager);
                    }
                } else {
                    doSave(sas, sessionManager);
                }
            }
        }
    }

    protected void doSave(DialogFragment df) {
        PersistentDialogFragmentManager dfm = (PersistentDialogFragmentManager) DialogFragmentManager.getInstance();

        try {
            dfm.saveDialogFragment(df);
        } catch (IOException e) {
            if (logger.isLoggable(Level.WARNING)) {
                logger.warning(PersistenceUtil.class.getCanonicalName() +
                    ".df_save_error", e);
            }
        }
    }

    protected void doSave(SipApplicationSessionImpl sas,
        PersistentSipSessionManagerBase sessionManager) {
        Iterable<String> sessionIds;
        Iterable<String> timerIds;

        synchronized (sas.getSasObjectLock()) {
            sessionIds = sas.getPersistentSessionIds();
            timerIds = sas.getPersistentTimerIds();

            try {
                sessionManager.saveSipApplicationSession(sas);

                if (logger.isLoggable(Level.FINE)) {
                    logger.logMsg(Level.FINE, "Saved SAS: " + sas);
                }
            } catch (IOException e) {
                logger.warning(PersistenceUtil.class.getCanonicalName() +
                    ".sas_save_error", e);

                return;
            }
        }

        for (String ssId : sessionIds) {
            SipSessionDialogImpl sipSession = sessionManager.findSipSessionFromCacheOnly(ssId);

            if (sipSession != null) {
                doSaveSession(sessionManager, sipSession);
            }
        }

        for (String timerId : timerIds) {
            ServletTimerImpl timer = sessionManager.findServletTimerFromCacheOnly(timerId);

            if ((timer != null)) {
                doSaveTimer(sessionManager, timer);
            }
        }
    }

    protected void doSaveTimer(PersistentSipSessionManagerBase sessionManager,
        ServletTimerImpl timer) {
        try {
            sessionManager.saveServletTimer(timer);

            if (logger.isLoggable(Level.FINE)) {
                logger.logMsg(Level.FINE, "Saved timer: " + timer);
            }
        } catch (IOException e) {
            logger.warning(PersistenceUtil.class.getCanonicalName() +
                ".timer_save_error", e);
        }
    }

    protected void doSaveSession(PersistentSipSessionManagerBase sessionManager,
        SipSessionDialogImpl sipSession) {
        synchronized (sipSession.getSipSessionObjectLock()) {
            try {
                sessionManager.saveSipSession(sipSession);

                if (logger.isLoggable(Level.FINE)) {
                    logger.logMsg(Level.FINE, "Saved SS: " + sipSession);
                }
            } catch (IOException e) {
                logger.warning(PersistenceUtil.class.getCanonicalName() +
                    ".ss_save_error", e);
            }
        }
    }

    /**
     * Saves the specified SipSession and its sibling SipSessions and Timers.
     * @param ss the SipSesssion
     */
    public void saveSipSessionWidely(SipSessionBase ss) {
        if (ss != null) {
            saveSipApplicationSessionDeeply(ss.getApplicationSessionImpl());
        }
    }

    /**
     * Saves the specified SipSession.
     * @param ss the SipSesssion
     */
    public void saveSipSessionNarrowly(SipSessionBase ss) {
        if ((ss != null) && ss.isReplicable()) {
            PersistentSipSessionManagerBase sessionManager = (PersistentSipSessionManagerBase) ((SipSessionDialogImpl) ss).getSipSessionManager();

            if (logger.isLoggable(Level.FINE)) {
                logger.logMsg(Level.FINE, "Saving SS: " + ss);
            }

            if (((SipSessionDialogImpl) ss).shouldBePersisted()) {
                doSaveSession(sessionManager, (SipSessionDialogImpl) ss);
            }
        }
    }

    /**
     * Saves the specified timer and its sibling SipSessions and Timers.
     * @param ss the timer
     * @throws IOException
     */
    public void saveTimerWidely(ServletTimerImpl st) {
        if (st != null) {
            saveSipApplicationSessionDeeply((SipApplicationSessionImpl) st.getApplicationSession());
        }
    }

    /**
     * Saves the specified timer.
     * @param ss the timer
     * @throws IOException
     */
    public void saveTimerNarrowly(ServletTimerImpl st) {
        if ((st != null) &&
                ((SipApplicationSessionImpl) st.getApplicationSession()).isReplicable()) {
            PersistentSipSessionManagerBase sessionManager = (PersistentSipSessionManagerBase) ((SipApplicationSessionImpl) st.getApplicationSession()).getSipSessionManager();

            if (logger.isLoggable(Level.FINE)) {
                logger.logMsg(Level.FINE, "Saving Timer: " + st);
            }

            doSaveTimer(sessionManager, st);
        }
    }

    /**
     * saves the specified dialogFragment and all the associated items (SAS/SS/Timers)
     *
     * @param aDF the dialogFragment
     * @throws IOException
     */
    public void saveDialogueFragmentDeeply(DialogFragment aDF) {
        if (aDF.isReplicable()) {
            for (SipApplicationSessionImpl sas : getSipApplicationSessions(aDF)) {
                saveSipApplicationSessionDeeply(sas);
            }

            // XXX locking?
            doSave(aDF);
        }
    }

    private Iterable<SipApplicationSessionImpl> getSipApplicationSessions(
        DialogFragment aDf) {
        Set<SipApplicationSessionImpl> sases = new HashSet<SipApplicationSessionImpl>();
        Iterator<PathNode> pathnodes = aDf.getCallee2CallerPath();

        while (pathnodes.hasNext()) {
            sases.add(pathnodes.next().getApplicationSession());
        }

        return sases;
    }
}
