/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.config.ConvergedContext;
import com.ericsson.ssa.dd.SipApplicationListeners;
import com.ericsson.ssa.sip.LifeCycle;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipSessionManager;
import com.ericsson.ssa.sip.SipSessionManagerBase;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import java.util.UUID;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;

import javax.servlet.sip.TimerListener;


public class ServletTimerImpl extends ServletTimerImplBase
    implements Serializable, LifeCycle {
    private static Logger logger = Logger.getLogger("SipContainer");

    /**
     * Class identifier for serialization
     */
    private static final long serialVersionUID = 8123619219185134043L;

    /**
     * A future delayed scheduled action to be run
     */
    private ScheduledFuture<?> future;

    /**
     * Information about what this timer is.
     */
    private Serializable info;

    /**
     * The associated SipApplicationSession
     */
    private SipApplicationSessionImpl sas = null;

    /**
     * The id of the associated SipApplicationSession
     */
    private String sasId = null;

    /**
     * The application id of the associated context
     */
    private String appId = null;

    /**
     * The associated session manager
     */
    private SipSessionManager sipSessionManager;

    /**
     * Absolute time in milliseconds for next execution.
     */
    private long scheduledExecutionTime = 0;

    /**
     * Delay from creation of timer to execution
     */
    private long delay = 0;

    /**
     * Period between executions. Only applicable for repeating timers.
     */
    private long period = 0;

    /**
     * Number of times execution has happened.
     */
    private long numInvocations = 0;

    /**
     * Absolute time for first execution.
     */
    private long firstExecution = 0;

    /**
     * Whether executions should be scheduled with fixed delay.
     *
     * @see java.util.Timer for semantics.
     */
    private boolean fixedDelay = false;

    /**
     * Whether this timer is persistent.
     */
    private boolean persistent = true;

    /**
     * Registered listener that will get a timeout event when executed.
     */
    private TimerListener listener;
    private transient AtomicBoolean listenerLookedUp = null;

    /**
     * Whether execution should be repeated.
     */
    private boolean isRepeatingTimer = true;

    /**
     * The unique id of this timer.
     */
    private String id = null;
    protected String fullyQualifiedId;

    /**
     * Constructor for non-repeating timer.
     *
     * @param manager The associated session manager
     * @param sas The SipApplicationSession with which this ServletTimer has
     * been associated
     * @param info Information about the timer
     * @param delay Delay until execution
     * @param listener Listener that will get timeout events.
     */
    public ServletTimerImpl(SipSessionManager manager,
        SipApplicationSessionImpl sas, Serializable info, long delay,
        TimerListener listener) {
        this(manager, sas, info, delay, false, 0, listener);
        isRepeatingTimer = false;
    }

    /**
     * Constructor for repeating times
     *
     * @param manager The associated session manager
     * @param sas The SipApplicationSession with which this ServletTimer has
     * been associated
     * @param info Information about the timer
     * @param delay Delay until first execution
     * @param fixedDelay Whether fixed delay mode should be used
     * @param period Period between execution
     * @param listener Listener that will get timeout events.
     */
    public ServletTimerImpl(SipSessionManager manager,
        SipApplicationSessionImpl sas, Serializable info, long delay,
        boolean fixedDelay, long period, TimerListener listener) {
        this.sipSessionManager = manager;
        this.appId = manager.getApplicationId();
        this.sas = sas;
        this.sasId = sas.getId();
        this.info = info;
        this.delay = delay;
        scheduledExecutionTime = delay + System.currentTimeMillis();
        this.fixedDelay = fixedDelay;
        this.period = period;
        this.listener = listener;
        this.listenerLookedUp = new AtomicBoolean(true);

        id = UUID.randomUUID().toString();
        sas.addServletTimer(this);
        fullyQualifiedId = "ServletTimer with id " + appId + ":" + id;
    }

    /**
     * @see java.io.Externalizable#writeExternal(java.io.ObjectOutput)
     */
    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeUTF(id);
        out.writeObject(info);

        out.writeUTF(appId);
        out.writeUTF(sasId);

        out.writeLong(scheduledExecutionTime);
        out.writeLong(delay);
        out.writeLong(period);
        out.writeLong(numInvocations);
        out.writeLong(firstExecution);
        out.writeBoolean(fixedDelay);
        out.writeBoolean(isRepeatingTimer);
    }

    /**
     * @see java.io.Externalizable#readExternal(java.io.ObjectInput)
     */
    private void readObject(ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        id = in.readUTF();
        info = (Serializable) in.readObject();

        // Read the appid, and use it to get the corresponding
        // SipSessionManager and associated context.
        appId = in.readUTF();
        sasId = in.readUTF();

        fullyQualifiedId = "ServletTimer with id " + appId + ":" + id;

        scheduledExecutionTime = in.readLong();
        delay = in.readLong();
        period = in.readLong();
        numInvocations = in.readLong();
        firstExecution = in.readLong();
        fixedDelay = in.readBoolean();
        isRepeatingTimer = in.readBoolean();
        persistent = true;
        listenerLookedUp = new AtomicBoolean(false); // Need to explicitely instantiate.  
    }

    // --- PField accessors ---
    @Override
    protected String getPFieldId() {
        return id;
    }

    @Override
    protected ScheduledFuture<?> getPFieldFuture() {
        return future;
    }

    @Override
    protected void setPFieldFuture(ScheduledFuture<?> future) {
        this.future = future;
    }

    @Override
    protected Serializable getPFieldInfo() {
        return info;
    }

    @Override
    protected synchronized SipApplicationSessionImpl getPFieldAppSession() {
        if (sas == null) {
            sas = getSipSessionManager().findSipApplicationSession(sasId);

            if (sas == null) {
                logger.warning("SipApplication Session " + sasId +
                    " not found");
            }
        }

        return sas;
    }

    @Override
    protected long getPFieldFirstExecution() {
        return firstExecution;
    }

    @Override
    protected long getPFieldPeriod() {
        return period;
    }

    @Override
    protected long getPFieldScheduledExecutionTime() {
        return scheduledExecutionTime;
    }

    @Override
    protected long incrementAndGetPFieldNumInvocations() {
        return ++numInvocations;
    }

    @Override
    protected void setPFieldFirstExecution(long firstExecution) {
        this.firstExecution = firstExecution;
    }

    @Override
    protected void setPFieldScheduledExecutionTime(long scheduledExecutionTime) {
        this.scheduledExecutionTime = scheduledExecutionTime;
    }

    @Override
    protected long getPFieldDelay() {
        return delay;
    }

    @Override
    protected boolean getPFieldFixedDelay() {
        return fixedDelay;
    }

    @Override
    protected boolean getPFieldPersistent() {
        return persistent;
    }

    @Override
    protected void setPFieldPersistent(boolean p) {
        this.persistent = p;
    }

    @Override
    protected boolean getPFieldIsRepeatingTimer() {
        return isRepeatingTimer;
    }

    @Override
    protected synchronized TimerListener getPFieldListener() {
        if (!listenerLookedUp.getAndSet(true)) {
            SipSessionManager sipSessionManager = getSipSessionManager();

            if (sipSessionManager != null) {
                ConvergedContext ctx = sipSessionManager.getContext();

                if (ctx != null) {
                    SipApplicationListeners appListeners = ctx.getSipApplicationListeners();

                    if (appListeners != null) {
                        listener = appListeners.getTimerListener();
                    }
                }
            }
        }

        return listener;
    }

    private synchronized SipSessionManager getSipSessionManager() {
        if (sipSessionManager == null) {
            sipSessionManager = SipSessionManagerBase.get(appId);

            if (sipSessionManager == null) {
                logger.warning("SipSessionManager " + appId + " not found");
            }
        }

        return sipSessionManager;
    }

    protected String getFullyQualifiedId() {
        return fullyQualifiedId;
    }

    private void restartTimer() {
        // Rescedule timer
        TimerServiceImpl.getInstance().rescheduleTimer(this);
    }

    @Override
    public void cancel(boolean mayInterruptIfRunning) {
        super.cancel(mayInterruptIfRunning);
        getSipSessionManager().removeServletTimer(this);
    }

    /*
     * (non-Javadoc)
     * @see com.ericsson.ssa.sip.LifeCycle#activate()
     */
    public boolean activate() {

        if (!validate()) {
            return false;
        }

        // Add to active cache
        SipSessionManager mgr = getSipSessionManager();
        if (mgr != null) {
            mgr.addServletTimer(this);
        }

        // Restart timer if not already started
        if (future == null) {
            restartTimer();
        }

        return true;
    }

    protected boolean validate() {

        if (getPFieldAppSession() == null) {
            // ServletTimer is orphaned
            getSipSessionManager().removeServletTimer(this);
            return false;
        }

        return true;
    }

    /*
     * (non-Javadoc)
     * @see com.ericsson.ssa.sip.LifeCycle#passivate()
     */
    public void passivate() {
        cancel();
    }
}
