/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.sip.SipApplicationSessionImpl;

import org.apache.catalina.session.SessionLock;

import java.io.Serializable;

import java.util.concurrent.ScheduledFuture;

import javax.servlet.sip.ServletTimer;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.TimerListener;


/**
 * Implementation of SSA interface <code>javax.servlet.sip.ServletTimer</code>.
 */
public abstract class ServletTimerImplBase extends GeneralTimerBase
    implements ServletTimer {
    private final SessionLock timerLock = new SessionLock();

    // Empty constructor for serialization
    public ServletTimerImplBase() {
    }

    public void timeout() {
        getPFieldListener().timeout(this);
    }

    /**
    *
    * Cancel this timer, possibly by also interrupting the thread (from the
    * thread pool) running the task. Note that interupting the thread may have
    * undesired consequences.
    *
    * @param mayInterruptIfRunning
    */
    @Override
    public void cancel(boolean mayInterruptIfRunning) {
        SipApplicationSessionImpl appSessionToCancelThisTimersFrom = null;

        synchronized (SIPP_APPLICATION_LOCK) {
            ScheduledFuture<?> future = getPFieldFuture();

            if (future != null) {
                // need to force cancel to get rid of
                // the task which is currently scheduled
                boolean res = future.cancel(mayInterruptIfRunning);
                // used for debugging/optimizeIt purpose
                // kan be kept in production code since object should
                // be due for gc anyway....
                iscanceled = new Boolean(res);
                appSessionToCancelThisTimersFrom = getPFieldAppSession();
                setPFieldFuture(null);
            }
        }

        if (appSessionToCancelThisTimersFrom != null) {
            appSessionToCancelThisTimersFrom.cancelServletTimer(this);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#getApplicationSession()
     */
    public SipApplicationSession getApplicationSession() {
        return getPFieldAppSession();
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#getInfo()
     */
    public Serializable getInfo() {
        return getPFieldInfo();
    }

    /**
     * Gets the unique id of this timer.
     *
     * @return The unique id of this timer
     */
    public String getId() {
        return getPFieldId();
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String fullyQualifiedId = getFullyQualifiedId();

        if (fullyQualifiedId != null) {
            sb.append(fullyQualifiedId).append('\n');
        }

        sb.append("Info = ").append(getPFieldInfo()).append('\n');
        sb.append("Scheduled execution time = ")
          .append(getPFieldScheduledExecutionTime()).append('\n');
        sb.append("Time now = ").append(System.currentTimeMillis()).append('\n');
        sb.append("SipApplicationSession = ").append(getPFieldAppSession())
          .append('\n');
        sb.append("ScheduledFuture = ").append(getPFieldFuture()).append('\n');
        sb.append("Delay = ").append(getPFieldDelay()).append('\n');

        return sb.toString();
    }

    /**
     * Predicate for persistence property
     *
     * @return
     */
    public boolean isPersistent() {
        return getPFieldPersistent();
    }

    /**
     * Setter for persistence property
     *
     * @param p
     */
    public void setPersistent(boolean p) {
        setPFieldPersistent(p);

        SipApplicationSessionImpl appSession = getPFieldAppSession();

        if (p && (appSession != null)) {
            appSession.setServletTimerShouldBePersisted(this);
        }
    }

    /* To be implemented */
    public String getID() {
        throw new RuntimeException("Not yet implemented");
    }

    /**
     * Getter for listener object
     *
     * @return
     */
    TimerListener getListener() {
        return getPFieldListener();
    }

    //begin lock related code

    /**
     * get this timer locked for foreground if the timer is found to be
     * presently background locked; retry logic in a time-decay polling loop
     * waits for background lock to clear after 6 attempts (12.6 seconds) it
     * unlocks the timer and acquires the foreground lock
     */
    protected boolean getTimerLockForForeground() {
        boolean result = false;

        // now lock the timer
        long pollTime = 200L;
        int tryNumber = 0;
        int numTries = 7;
        boolean keepTrying = true;
        boolean lockResult = false;

        // try to lock up to numTries (i.e. 7) times
        // poll and wait starting with 200 ms
        while (keepTrying) {
            lockResult = lockForeground();

            if (lockResult) {
                keepTrying = false;
                result = true;

                break;
            }

            tryNumber++;

            if (tryNumber < (numTries - 1)) {
                pollTime = pollTime * 2L;
            } else {
                // unlock the background so we can take over
                // FIXME: need to log warning for this situation
                unlockBackground();
            }
        }

        return result;
    }

    /**
     * return whether this timer is currently foreground locked
     */
    public synchronized boolean isForegroundLocked() {
        return timerLock.isForegroundLocked();
    }

    /**
     * lock the timer for foreground returns true if successful; false if
     * unsuccessful
     */
    public synchronized boolean lockBackground() {
        return timerLock.lockBackground();
    }

    /**
     * lock the timer for background returns true if successful; false if
     * unsuccessful
     */
    public synchronized boolean lockForeground() {
        return timerLock.lockForeground();
    }

    /**
     * unlock the timer completely irregardless of whether it was foreground
     * or background locked
     */
    public synchronized void unlockForegroundCompletely() {
        timerLock.unlockForegroundCompletely();
    }

    /**
     * unlock the timer from foreground
     */
    public synchronized void unlockForeground() {
        timerLock.unlockForeground();
    }

    /**
     * unlock the timer from background
     */
    public synchronized void unlockBackground() {
        timerLock.unlockBackground();
    }

    /**
     * return the timer lock
     */
    public SessionLock getTimerLock() {
        return timerLock;
    }

    //end lock related code

    // --- Persistable fields
    protected abstract String getPFieldId();

    protected abstract Serializable getPFieldInfo();

    protected abstract SipApplicationSessionImpl getPFieldAppSession();

    protected abstract boolean getPFieldPersistent();

    protected abstract void setPFieldPersistent(boolean p);

    protected abstract String getFullyQualifiedId();

    protected abstract TimerListener getPFieldListener();
}
