/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.admin.mbeans.extensions;

import java.util.Enumeration;
import java.util.ArrayList;
import java.util.Properties;

import javax.management.*;

import com.sun.enterprise.admin.config.MBeanConfigInstanceNotFoundException;
import com.sun.enterprise.admin.config.MBeanConfigException;
import com.sun.enterprise.admin.config.ConfigMBeanHelper;
import com.sun.enterprise.admin.mbeans.ConfigsMBean;
import com.sun.enterprise.admin.mbeans.MBeanExceptionFormatter;
import com.sun.enterprise.admin.target.Target;
import com.sun.enterprise.admin.target.TargetType;
import com.sun.enterprise.admin.target.TargetBuilder;
import com.sun.enterprise.admin.target.ConfigTarget;

import com.sun.enterprise.config.serverbeans.ConfigAPIHelper;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.SecurityService;
import com.sun.enterprise.config.serverbeans.IdentityAssertionTrust;
import com.sun.enterprise.config.serverbeans.TrustHandler;
import com.sun.enterprise.config.serverbeans.TrustedEntity;
import com.sun.enterprise.config.serverbeans.ElementProperty;

import com.sun.enterprise.util.i18n.StringManager;

public class SipConfigMBean extends ConfigsMBean {
    private ObjectName on;
    private static final StringManager _strMgr = StringManager.getManager(SipConfigMBean.class);
    private static final String DEFAULT_TRUST_HANDLER = "org.jvnet.glassfish.comms.security.auth.impl.TrustHandlerImpl";

    private static final TargetType[] VALID_LIST_TYPES = new TargetType[] {
        TargetType.CLUSTER, TargetType.CONFIG, TargetType.SERVER, TargetType.DAS};


    public SipConfigMBean() {
        super();
    }

    public ObjectName createSipListener(AttributeList attrList,
        Properties props, String targetName) throws MBeanException {
        final Target target = getTarget(targetName);

        //        check1ToN(target);
        ObjectName sipService = getSipServiceMBean(target);
        ObjectName mbean = (ObjectName) invoke1(sipService,
                "createSipListener", attrList, AttributeList.class.getName());
        setProperties(mbean, props);

        return mbean;
    }

    public boolean deleteSipListener(String listenerId, String targetName)
        throws MBeanException {
        final Target target = getTarget(targetName);

        //        check1ToN(target);
        final ObjectName sipService = getSipServiceMBean(target);
        invoke1(sipService, "removeSipListenerById", listenerId,
            String.class.getName());

        return true;
    }

    public ObjectName[] listSipListeners(String targetName)
        throws MBeanException {
        final Target target = getListTarget(targetName);
        ObjectName sipService = getSipServiceMBean(target);
        ObjectName[] ret = (ObjectName[]) invoke0(sipService, "getSipListener");

        return ret;
    }

    public ObjectName getSipService(String targetName)
        throws MBeanException {
        return getChild("sip-service", null, targetName);
    }

    public ObjectName getSipListener(String listenerName, String targetName)
        throws MBeanException {
        return getChild("sip-listener", new String[] { listenerName },
            targetName);
    }

    /** The function is used to get the ssl element for the specified listenerName. If the ssl element
     *  does not exist then an MBeanConfigInstanceNotFoundException is thrown.
     */
    public ObjectName getSSL(String listenerName, String targetName)
        throws MBeanException {
        ObjectName sipListener = getSipListener(listenerName, targetName);

        try {
            return (ObjectName) invoke0(sipListener, "getSsl");
        } catch (MBeanException e) {
            if ((e.getTargetException() != null) &&
                    (e.getTargetException().getCause() instanceof MBeanConfigInstanceNotFoundException)) {
                return null;
            }

            throw e;
        }
    }

    protected ObjectName getSipServiceMBean(Target target)
        throws MBeanException {
        final ObjectName configMBean = getConfigMBean(target);
        ObjectName ret = (ObjectName) invoke0(configMBean, "getSipService");

        return ret;
    }

    /**
     * Creates the identity-assertion-trust element under security-service
     * Trust config can contain either trust-handler or trusted-entity, not both
     * If user doesn't specify any options, then a trust config with a default
     * trust handler implementation class will be created.
     */
    public ObjectName createIdentityAssertionTrust( AttributeList   attrList,
                                                    AttributeList entityList,
                                                    Properties      props,
                                                    String          targetName)
    throws Exception
    {
        final Target target = getTarget(targetName);
        final ConfigTarget configTarget = target.getConfigTarget();
        SecurityService securityService = getSecurityServiceConfigBean(configTarget.getName());
        return createIdentityAssertionTrust(attrList, entityList, props, configTarget.getName(), securityService);

    }

    public ObjectName createIdentityAssertionTrust( AttributeList   attrList,
                                                    AttributeList   entityList,
                                                    Properties      props,
                                                    String    configName,
                                                    SecurityService securityService)
    throws Exception
    {
        String cName = null;
        String ipAdd = null;
        String hostName = null;
        String principal = null;
        String trustedAs = null;
        String trustId = null;
        String entityId = null;
        String isDefault = "false";

        // get all the values
        for (int i = 0; i <attrList.size(); i++) {
            Attribute attr = (Attribute)attrList.get(i);
            if (isAttrNameMatch(attr, "class-name"))
                cName = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "trusted-as"))
                trustedAs = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "id"))
                trustId = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "is-default"))
                isDefault = (String)attr.getValue();
        }

        // get all the trusted entity info
        for (int i = 0; i < entityList.size(); i++) {
            Attribute attr1 = (Attribute)entityList.get(i);
            if (isAttrNameMatch(attr1, "id"))
                entityId = (String)attr1.getValue();
            else if (isAttrNameMatch(attr1, "trusted-as"))
                trustedAs = (String)attr1.getValue();
            else if (isAttrNameMatch(attr1, "ip-address"))
                ipAdd = (String)attr1.getValue();
            else if (isAttrNameMatch(attr1, "host-name"))
                hostName = (String)attr1.getValue();
            else if (isAttrNameMatch(attr1, "principal"))
                principal = (String)attr1.getValue();
        }

        // can have either trust handler or trusted entity not both
        if (cName != null && (ipAdd != null || hostName != null || principal != null)) {
            String msg = _strMgr.getString("EitherTrustHandlerOrTrustedEntity");
            throw new MBeanConfigException(msg);
        }

        if (cName == null && (ipAdd == null && (hostName != null || principal != null || trustedAs != null))) {
            String msg = _strMgr.getString("IpAddMustforTrustedEntity");
            throw new MBeanConfigException(msg);
        }

        // identity assertion trust
        IdentityAssertionTrust trustConfig = securityService.getIdentityAssertionTrustById(trustId);
        if (trustConfig != null)
        {
            String msg = _strMgr.getString("IdentityTrustAlreadyExists", trustId);
            throw new MBeanConfigException(msg);
        }

        // we have to create it
        trustConfig = new IdentityAssertionTrust();
        if (trustId == null)
            trustId = generateTrustId(securityService);
        trustConfig.setId(trustId);
        trustConfig.setIsDefault(Boolean.parseBoolean(isDefault));

        // create the trust handler
        if ( cName != null) {
            TrustHandler tHandler = new TrustHandler();
            tHandler.setClassName(cName);
            securityService.addIdentityAssertionTrust(trustConfig);
            if (props != null) {
                tHandler.setElementProperty(convertPropertiesToElementProperties(props));
            }
            trustConfig.setTrustHandler(tHandler);
//            return trustId;
            return getMBeanRegistry().getMbeanObjectName("trust-handler",
                   new String[]{getDomainName(), configName, trustId});
        }

        // ip address is must for trusted entity config
        if ( ipAdd != null) {
            TrustedEntity tEntity = new TrustedEntity();
            if (entityId == null)
                entityId = "entityid-0";
            tEntity.setId(entityId);
            if (trustedAs != null)
                tEntity.setTrustedAs(trustedAs);
            tEntity.setIpAddress(ipAdd);
            tEntity.setPrincipal(principal);
            tEntity.setHostName(hostName);
            trustConfig.addTrustedEntity(tEntity);
            securityService.addIdentityAssertionTrust(trustConfig);
//            return trustId;
            return getMBeanRegistry().getMbeanObjectName("trusted-entity",
                   new String[]{getDomainName(), configName, trustId, entityId});
        }

        TrustHandler tHandler = new TrustHandler();
        // use the default trust handler implementation
        tHandler.setClassName(DEFAULT_TRUST_HANDLER);

        trustConfig.setTrustHandler(tHandler);
        securityService.addIdentityAssertionTrust(trustConfig);

//        return trustId;
        return getMBeanRegistry().getMbeanObjectName("trust-handler",
               new String[]{getDomainName(), configName, trustId});
    }

    private static boolean isAttrNameMatch(Attribute attr, String name)
    {
        // for now we supporting both "dashed" and "underscored" names
        return attr.getName().replace('_','-').equals(name.replace('_','-'));
    }

    private ElementProperty[] convertPropertiesToElementProperties(Properties props)
    {
        ArrayList list = new ArrayList();
        Enumeration keys = props.keys();
        while (keys.hasMoreElements())
        {
            final String key = (String)keys.nextElement();
            ElementProperty property = new ElementProperty();
            property.setName(key);
            property.setValue((String)props.get(key));
            list.add(property);
        }
        return (ElementProperty[])list.toArray(new ElementProperty[list.size()]);
    }

    /**
     * Method to generate trust ids automatically for identity-assertion-trust
     */
    public String generateTrustId(SecurityService ss) {
        String id = null;
        IdentityAssertionTrust trustConfig = null;
        for (int i=0; ; i++) { 
            id = "trustid-" + i;
            trustConfig = ss.getIdentityAssertionTrustById(id);
            if (trustConfig == null)
                break;
        }
        return id;
    }

    /**
     * Method to generate entity ids automatically for trusted-entity
     */
    public String generateEntityId(IdentityAssertionTrust iat) {
        String id = null;
        TrustedEntity tEntity = null;
        for (int i=0; ; i++) { 
            id = "entityid-" + i;
            tEntity = iat.getTrustedEntityById(id);
            if (tEntity == null)
                break;
        }
        return id;
    }

    /**
     * Deletes the identity-assertion-trust from security-service
     */
    public boolean deleteIdentityAssertionTrust( String id, String targetName )
    throws Exception
    {
        final Target target = getTarget(targetName);
//        check1ToN(target);
        SecurityService securityService = getSecurityServiceConfigBean(target.getConfigTarget().getName());
        IdentityAssertionTrust trustConfig = securityService.getIdentityAssertionTrustById(id);

        if (trustConfig == null)
        {
            String msg = _strMgr.getString("IdentityTrustDoesntExist", id);
            throw new MBeanConfigException(msg);
        }

        securityService.removeIdentityAssertionTrust(trustConfig);

        return true;
    }

    /**
     * Method to return the list of trust configs.
     * If trustId is specified, then we return all the trusted-entities for it.
     */
    public Object[] listIdentityAssertionTrusts( String trustId, String targetName )
    throws Exception
    {
        final Target target = getListTarget(targetName);
        SecurityService securityService = getSecurityServiceConfigBean(target.getConfigTarget().getName());

        // if trustId is specified, list all trusted entities for that config
        if (trustId != null)
        {
            IdentityAssertionTrust trustConfig =
                   securityService.getIdentityAssertionTrustById(trustId);
            if (trustConfig == null)
            {
                String msg = _strMgr.getString("IdentityTrustDoesntExist", trustId);
                throw new MBeanConfigException(msg);
            }
            return getTrustedEntities(trustConfig);
        }

        // trustId is not specified, list all trust configs
        IdentityAssertionTrust[] trustConfigs = securityService.getIdentityAssertionTrust();
        return ConfigMBeanHelper.getConfigBeansObjectNames(
            this.getMBeanRegistry(), this.getDomainName(), trustConfigs);
    }

    /**
     * This will return all the trusted-entities for a given trust config.
     */
    public Object[] getTrustedEntities( IdentityAssertionTrust trust )
    throws Exception
    {
        TrustedEntity[] entities = trust.getTrustedEntity();
        return ConfigMBeanHelper.getConfigBeansObjectNames(
            this.getMBeanRegistry(), this.getDomainName(), entities);

    }

    /**
     * Creates a trusted entity for a trust config
     */
    public boolean createTrustedEntity( AttributeList   attrList,
                                        String trustId,
                                        String targetName )
    throws Exception
    {
        String id = null;
        String trustedAs = null;
        String ipAdd = null;
        String host = null;
        String principal = null;

        final Target target = getTarget(targetName);
//        check1ToN(target);
        SecurityService securityService = getSecurityServiceConfigBean(target.getConfigTarget().getName());

        for (int i = 0; i <attrList.size(); i++) {
            Attribute attr = (Attribute)attrList.get(i);
            if (isAttrNameMatch(attr, "id"))
                id = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "trusted-as"))
                trustedAs = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "ip-address"))
                ipAdd = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "host-name"))
                host = (String)attr.getValue();
            else if (isAttrNameMatch(attr, "principal"))
                principal = (String)attr.getValue();
        }

        IdentityAssertionTrust trustConfig =
            securityService.getIdentityAssertionTrustById(trustId);
        // invalid trustId was specified
        if (trustConfig == null)
        {
            String msg = _strMgr.getString("IdentityTrustDoesntExist", trustId);
            throw new MBeanConfigException(msg);
        }

        // check if a trust handler is already configured for this trust config
        TrustHandler tHandler = trustConfig.getTrustHandler();
        if (tHandler != null) {
            String msg = _strMgr.getString("TrustHandlerAlreadyExistsForTrustConfig", trustId);
            throw new MBeanConfigException(msg);
        }
            
        TrustedEntity tEntity = new TrustedEntity();
        if (id == null)
            id = generateEntityId(trustConfig);

        tEntity.setId(id);
        tEntity.setTrustedAs(trustedAs);
        tEntity.setIpAddress(ipAdd);
        if (host != null)
            tEntity.setHostName(host);
        if (principal != null)
            tEntity.setPrincipal(principal);

        trustConfig.addTrustedEntity(tEntity);

        return true;
    }

    /**
     * Deletes the specified trusted-entity from the trust config
     */
    public boolean deleteTrustedEntity( String entityId, String trustId, String targetName )
    throws Exception
    {
        final Target target = getTarget(targetName);
//        check1ToN(target);
        SecurityService securityService = getSecurityServiceConfigBean(target.getConfigTarget().getName());
        IdentityAssertionTrust trustConfig =
            securityService.getIdentityAssertionTrustById(trustId);
        // invalid trustId was specified
        if (trustConfig == null)
        {
            String msg = _strMgr.getString("IdentityTrustDoesntExist", trustId);
            throw new MBeanConfigException(msg);
        }
        TrustedEntity[] entities = trustConfig.getTrustedEntity();
        // if its THE only trusted entity element, we might as well
        // remove the trust config itself
        if (entities.length==1 && entityId.equals(trustConfig.getTrustedEntity(0).getId()))
        {
            securityService.removeIdentityAssertionTrust(trustConfig);
            return true;
        }
        // get the trusted entity from the trust config and remove
        TrustedEntity tEntity = trustConfig.getTrustedEntityById(entityId);
        if (tEntity == null)
        {
            String msg = _strMgr.getString("TrustedEntityDoesntExist", entityId);
            throw new MBeanConfigException(msg);
        }
        trustConfig.removeTrustedEntity(tEntity);

        return true;
    }

    private SecurityService getSecurityServiceConfigBean(String configName)
    throws Exception
    {
        Config config = ConfigAPIHelper.getConfigByName(getConfigContext(), configName);
        return config.getSecurityService();
    }
}
