package org.jvnet.glassfish.comms.clb.core;

import com.sun.enterprise.admin.server.core.AdminService;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.AvailabilityService;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.ConvergedLbClusterRef;
import com.sun.enterprise.config.serverbeans.ConvergedLbConfig;
import com.sun.enterprise.config.serverbeans.ConvergedLbConfigs;
import com.sun.enterprise.config.serverbeans.ConvergedLoadBalancer;
import com.sun.enterprise.config.serverbeans.Domain;
import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.ServerRef;
import com.sun.enterprise.server.ServerContext;
import java.util.logging.Logger;

import org.jvnet.glassfish.comms.clb.admin.CLBConfigurator;
import org.jvnet.glassfish.comms.clb.core.sip.SipLoadBalancerManager;
import org.jvnet.glassfish.comms.httplayers.HttpLayer;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.File;

import java.util.logging.Level;


public class ConvergedLoadBalancerFactory {
    private static SipLoadBalancerManager sipLBM;
    private static HttpLoadBalancingManager httpLBM;
    private static final Logger _logger = LogUtil.CLB_LOGGER.getLogger();
    private static String dcrFileName;
    private static boolean initialized = false;
    private static String clbConfigFileName;

    /**
     * Gets the SIP load balancing manager layer and also initializes. Used by the
     * {@link com.ericsson.ssa.config.LayerHandler} (via reflection) when
     * setting up the SIP and HTTP processing chains at start-up.
     *
     */
    public static SipLoadBalancerManager getInstance() {
        createLoadBalancingManagers();
        //Initiliazing the converged load balancer as both HttpLoadBalancingManager
        //and SipLoadBalancingManager must be initialized together
        //This is done here as SIP container will be available now
        //for SIP component to initialize
        initialize();

        return sipLBM;
    }

    private static void initialize() {
        if ((httpLBM != null) && (sipLBM != null)) {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE, "clb.initializing_the_converged_load_balancer");

            File configFile = new File(clbConfigFileName);
            CLBConfigurator clbConfigurator = CLBConfigurator.getInstance();
            clbConfigurator.init(configFile, sipLBM, httpLBM);
            dcrFileName = clbConfigurator.getDCRFileName();
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.completed_initialization_of_the_converged_load_balancer");
        } else {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE, "clb.not_initializing_the_converged_load_balancer");
        }
    }

    private static synchronized void createLoadBalancingManagers() {
        if (initialized) {
            return;
        }

        initialized = true;

        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE, "clb.configuring_the_converged_load_balancer");

        ServerContext sc = com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext();
        ConfigContext instanceConfigContext = sc.getConfigContext();

        //Added check whether this instance is DAS, as DAS cannot act as a converged load-balancer
        //This check may not be needed in case config of DAS will never have converged-load-balancer
        //under availability-service
        try {
            if (ServerHelper.isDAS(instanceConfigContext, sc.getInstanceName())) {
                initialized = true;
                _logger.log(Level.INFO, 
                    "clb.clb_not_enabled_on_DAS");

                return;
            }
        } catch (ConfigException ex) {
            _logger.log(Level.SEVERE, 
                    "clb.unable_to_determine_whether_instance_is_DAS_or_not",
                    new Object[]{ex.getMessage()});
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.caught_an_exception",
                        ex);
        }

        Config instanceConfig = null;

        try {
            instanceConfig = ServerBeansFactory.getConfigBean(instanceConfigContext);
        } catch (ConfigException ex) {
            _logger.log(Level.SEVERE, 
                    "clb.configexception_will_getting_instance_config",
                    new Object[]{ex.getMessage()});
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.caught_an_exception",
                        ex);
            return;
        }

        AvailabilityService haService = instanceConfig.getAvailabilityService();

        if (haService == null) {
            _logger.log(Level.SEVERE,
                        "clb.no_availability_service_found");
            return;
        }

        ConvergedLoadBalancer clb = haService.getConvergedLoadBalancer();

        if (clb == null) {
            _logger.log(Level.INFO,
                    "clb.no_clb_entry_found");
            return;
        }

        _logger.log(Level.INFO,
                    "clb.clb_entry_found");
        
        //Setting static variable with DCR file name
        //converged load-balancer xml file name : 
        //as of now it is <config-file>.<converged-load-balanacer-name>
        clbConfigFileName = clb.getConfigFile() + "." + clb.getName();

        //Creating instance of HttpLoadBalancingManager
        //and SipLoadBalancerManager
        //They will be initialized later
        if (httpLBM == null) {
            httpLBM = HttpLoadBalancingManager.getInstance();
        }

        if (sipLBM == null) {
            sipLBM = SipLoadBalancerManager.getInstance();
        }

        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE, "clb.configured_the_converged_load_balancer");
    }

    /**
     * This is the method used by the {@link org.jvnet.glassfish.comms.httplayers.HttpLayerHandler} when initializing from dispatcher.xml.
     * @return always null;
     */
    public static HttpLayer getHttpLayerInstance() {
        createLoadBalancingManagers();

        return httpLBM;
    }

    /**
     * Returns the DCR file name
     *
     * @return the name of dcr file
     */
    public static String getDCRFileName() {
        return dcrFileName;
    }
    
    /**
     * This method is used to find out whether this instance is front-ended 
     * by CLB or not.
     * 
     * @return true, if instance is front-ended by CLB
     */
    public static boolean isInstanceFrontEndedByCLB() {

        boolean matchFound = false;

        try {
            ServerContext sc =
                com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext();
            ConfigContext instanceConfigContext = sc.getConfigContext();
            ConfigContext adminConfigContext =
                AdminService.getAdminService().getAdminContext().getAdminConfigContext();
            String instanceName = sc.getInstanceName();
            String clusterName = null;
            boolean isCluster = false;
            //Determine whether this instance is a standalone instance
            //or a clustered instance
            if (ServerHelper.isServerClustered(adminConfigContext,
                instanceName)) {
                Cluster cluster =
                    ClusterHelper.getClusterForInstance(adminConfigContext,
                    instanceName);
                clusterName = cluster.getName();
                isCluster = true;
            }

            //get all converged lb configs
            Domain domain = (Domain) adminConfigContext.getRootConfigBean();
            ConvergedLbConfigs clbConfigs = domain.getConvergedLbConfigs();
            // Null in case of a DAS or developer profile instance.Hence no CLB 
            // frontending  this instance.
            if( clbConfigs == null ) return false;
            ConvergedLbConfig[] clbConfigArray = clbConfigs.getConvergedLbConfig();
            //get all converged lb ref and determine if one points to this instance
            if (isCluster) {
                for (int i = 0; i < clbConfigArray.length && !matchFound; i++) {
                    ConvergedLbClusterRef[] clusterRefs =
                        clbConfigArray[i].getConvergedLbClusterRef();
                    for (int j = 0; j < clusterRefs.length; j++) {
                        if (clusterRefs[j].getRef().equals(clusterName)) {
                            matchFound = true;
                            break;
                        }
                    }
                }
            } else {
                for (int i = 0; i < clbConfigArray.length && !matchFound; i++) {
                    ServerRef[] serverRefs =
                        clbConfigArray[i].getServerRef();
                    for (int j = 0; j < serverRefs.length; j++) {
                        if (serverRefs[j].getRef().equals(instanceName)) {
                            matchFound = true;
                            break;
                        }
                    }
                }
            }
        //May be a further check whether any converged-load-balancer refers to this 
        //converged lb config can be done
        } catch (ConfigException ex) {
            _logger.log(Level.SEVERE,
                "clb.exception_when_detecting_whether_instance_is_front_ended_by_CLB", ex);
        }
        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE, "clb.is_instance_front_ended_by_CLB",
                new Object[]{matchFound});
        }
        return matchFound;
    }
}
