/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core.sip;

import com.ericsson.ssa.sip.dns.SipTransports;

import java.io.IOException;

import java.util.StringTokenizer;


/**
 * Information that specifies a connection between two endpoints.
 */
public class Connection {
    private static final String DELIMITER = ":";
    private static final String UNDEFINED = "UNDEFINED";
    private SipTransports transport;
    private Socket remoteEndpoint;
    private Socket localEndpoint;

    /**
     * Create a connection.
     *
     * @param transport the transport protocol
     * @param localSocket the local end point; set to when creating a new
     *                outgoing connection
     * @param remoteSocket the remote end point
     */
    public Connection(SipTransports transport, Socket localSocket,
        Socket remoteSocket) {
        this.transport = transport;
        this.localEndpoint = localSocket;
        this.remoteEndpoint = remoteSocket;
    }

    /**
     * Create connection from a string encoded using {@link #getEncodedValue()}.
     *
     * @param encodedValue
     * @throws ConnectionParseException
     */
    private Connection(String encodedValue) throws ConnectionParseException {
        try {
            StringTokenizer st = new StringTokenizer(encodedValue, DELIMITER);

            while (st.hasMoreTokens()) {
                transport = SipTransports.getTransport(st.nextToken());

                String hostName = st.nextToken();
                String portStr = st.nextToken();

                if (!portStr.equals(hostName)) {
                    int port = Integer.parseInt(portStr);
                    localEndpoint = new Socket(hostName, port);
                }

                hostName = st.nextToken();
                portStr = st.nextToken();

                if (!portStr.equals(hostName)) {
                    int port = Integer.parseInt(portStr);
                    remoteEndpoint = new Socket(hostName, port);
                }
            }
        } catch (Throwable t) {
            throw new ConnectionParseException(
                "The encoded connection data was incorrectly formatted: " +
                encodedValue, t);
        }
    }

    /**
     * Gets the connection encoded in one string
     *
     * @return the connection encoded in one string
     */
    public String getEncodedValue() {
        String value = '"' + toString() + '"';

        return value.trim();
    }

    /**
     * Creates an instance from an encoded string.
     *
     * @param s the encoded string
     * @return an instance from a Base64 encoded string
     * @throws ConnectionParseException in case connection was malformed
     * @throws IOException in case Base64 decoding failed
     */
    public static Connection getFromEncoded(String s)
        throws ConnectionParseException, IOException {
        String str = s.substring(1, s.length()-1);

        return new Connection(str);
    }

    /**
     * Gets the remote end point.
     *
     * @return the remote end point
     */
    public Socket getRemoteEndpoint() {
        return remoteEndpoint;
    }

    /**
     * Gets the local end point.
     *
     * @return the local end point
     */
    public Socket getLocalEndpoint() {
        return localEndpoint;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return transport.name() + DELIMITER +
        ((localEndpoint != null) ? localEndpoint.getHostName() : UNDEFINED) +
        DELIMITER +
        ((localEndpoint != null) ? localEndpoint.getPort() : UNDEFINED) +
        DELIMITER +
        ((remoteEndpoint != null) ? remoteEndpoint.getHostName() : UNDEFINED) +
        DELIMITER +
        ((remoteEndpoint != null) ? remoteEndpoint.getPort() : UNDEFINED);
    }

    /**
     * Gets the transport
     *
     * @return the transport
     */
    public SipTransports getTransport() {
        return transport;
    }
}
