/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 *  Copyright 1999-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.jvnet.glassfish.comms.clb.proxy.outbound;

import com.sun.grizzly.http.Constants;
import com.sun.grizzly.http.algorithms.ContentLengthAlgorithm;

import java.nio.ByteBuffer;
import java.nio.BufferUnderflowException;
import java.util.logging.Level;

import java.util.logging.Logger;
import org.jvnet.glassfish.comms.clb.proxy.config.LoadBalancerProxyConstants;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;

/**
 * Code duplicated from ContentLengthAlgorith from grizzly. Could
 * not be re-used because the parse and parseheader methods needed
 * some modifications.
 * 
 * @author Ramesh
 */
public class ResponseHandler extends ContentLengthAlgorithm {

    protected int payloadLength = 0;
    private boolean transferEncoding;
    protected String encodingType = null;
    protected long bytesWritten = 0;
    private boolean hasRemaining;
    private Logger _logger;

    /** Creates a new instance of ResponseHandler */
    public ResponseHandler() {
        super();
        _logger = ProxyConfig.getInstance().getLogger();
    }

    /**
     * Parse the <code>ByteBuffer</code> and try to determine if the bytes
     * stream has been fully read from the <code>SocketChannel</code>.
     *
     * Drain the <code>SocketChannel</code> and determine if the request bytes
     * has been fully read. For POST method, parse the bytes and seek for the
     * content-type header to determine the length of the request bytes.
     * @return true if we need to call back the <code>SelectorThread</code>
     *              This occurs when the stream doesn't contains all the
     *              request bytes.
     *         false if the stream contains all request bytes.
     *
     * @paran byteBuffer the bytes read.
     * @return true if the algorithm determines the end of the stream.
     */
    @Override
    public boolean parse(ByteBuffer byteBuffer) {
        isFound = false;

        curLimit = byteBuffer.limit();
        curPosition = byteBuffer.position();

        try {
            if (byteBuffer.position() == 0) {
                return false;
            }

            byteBuffer.flip();
            lastValid = byteBuffer.limit();

            if (!requestLineParsed) {
                requestLineParsed = parseRequestLine(byteBuffer);
                if (!requestLineParsed) {
                    return false;
                }
            }
            while (parseHeader(byteBuffer)) {
                ;
            }
            if (isFound) {
                payloadLength = ((contentLength < 0) ? 0 : contentLength) + headerLength;
            }
            boolean parsed = false;
            if ((contentLength > 0) || transferEncoding) {
                parsed = true;
            }
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "Content Length " + contentLength);
                _logger.log(Level.FINE, "Header Length " + headerLength);
                _logger.log(Level.FINE, "Payloadlength " + payloadLength);
                _logger.log(Level.FINE, "Transfer encoding " + transferEncoding);
            }
            return parsed;
        } catch (BufferUnderflowException bue) {
            _logger.log(Level.SEVERE,
                    "Buffer underflow", bue);
            return false;
        } finally {
            byteBuffer.limit(curLimit);
            byteBuffer.position(curPosition);
            if (isFound) {
                byteBuffer.flip();
            }
        }
    }

    public boolean isTransferEncoding() {
        return transferEncoding;
    }

    public void updateBytesWritten(long bytes) {
        bytesWritten += bytes;
    }

    public boolean hasRemaining() {
        return transferEncoding ? hasRemaining : (payloadLength > bytesWritten);
    }

    /**
     * Parse the headers, looking for content-length header and value.
     */
    @Override
    protected boolean parseHeader(ByteBuffer byteBuffer) {

        boolean headerFound = false;
        byte chr = 0;

        if (state == 4) {
            while (true) {
                if (pos >= lastValid) {
                    return false;
                }

                chr = byteBuffer.get(pos);

                if ((chr == Constants.CR) || (chr == Constants.LF)) {
                    if (chr == Constants.LF) {
                        pos++;
                        headerLength = pos;
                        isFound = true;
                        state = 4;
                        return false;
                    }
                } else {
                    break;
                }

                pos++;
            }
            state = 5;
        }

        // Mark the current buffer position
        int start = pos;

        if (state == 5) {
            boolean colon = false;
            while (!colon) {
                // Read new bytes if needed
                if (pos >= lastValid) {
                    return false;
                }

                chr = byteBuffer.get(pos);
                if (chr == Constants.COLON) {
                    colon = true;

                    if (contentLength == -1 && findBytes(ascbuf, start, pos,
                            CL_HEADER) != -1) {
                        headerFound = true;
                    } else if (transferEncoding == false && findBytes(ascbuf, start, pos,
                            LoadBalancerProxyConstants.TE_HEADER) != -1) {
                        transferEncoding = true;
                    }
                }

                if ((chr >= Constants.A) && (chr <= Constants.Z)) {
                    chr = (byte) (chr - Constants.LC_OFFSET);
                }

                ascbuf[pos] = chr;

                pos++;
            }
            state = 6;
        }

        // Mark the current buffer position
        start = pos;
        int realPos = pos;

        boolean eol = false;
        boolean validLine = true;

        while (validLine) {

            boolean space = true;

            if (state == 6) {
                while (space) {
                    if (pos >= lastValid) {
                        return false;
                    }

                    chr = byteBuffer.get(pos);
                    if ((chr == Constants.SP) || (chr == Constants.HT)) {
                        pos++;
                    } else {
                        space = false;
                    }
                }
                state = 7;
            }

            int lastSignificantChar = realPos;
            if (state == 7) {
                while (!eol) {
                    // Read new bytes if needed
                    if (pos >= lastValid) {
                        return false;
                    }

                    chr = byteBuffer.get(pos);
                    if (chr == Constants.CR) {
                    } else if (chr == Constants.LF) {
                        eol = true;
                    } else if (chr == Constants.SP) {
                        realPos++;
                    } else {
                        realPos++;
                        lastSignificantChar = realPos;
                    }
                    pos++;
                }
                state = 8;
                realPos = lastSignificantChar;
            }

            if (state == 8) {
                if (pos >= lastValid) {
                    return false;
                }

                chr = byteBuffer.get(pos);
                if ((chr != Constants.SP) && (chr != Constants.HT)) {
                    validLine = false;
                } else {
                    eol = false;
                    realPos++;
                }
                state = 4;
            }
        }

        if (headerFound) {
            byteBuffer.position(start + 1);
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < realPos - start; i++) {
                sb.append((char) byteBuffer.get());
            }
            contentLength = Integer.parseInt(sb.toString());
        } else if (transferEncoding) {
            byteBuffer.position(start + 1);
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < realPos - start; i++) {
                sb.append((char) byteBuffer.get());
            }
            encodingType = sb.toString();
        }
        return true;

    }

    public void updateEnd(ByteBuffer buff) {
        int ret = findBytes(buff.array(),
                buff.position(), buff.limit(), LoadBalancerProxyConstants.END_BYTES);
        hasRemaining = (ret < 0);
    }

    /***
     * Recylce this object.
     */
    @Override
    public void recycle() {
        super.recycle();
        payloadLength = 0;
        transferEncoding = false;
        hasRemaining = true;
        bytesWritten = 0;
        encodingType = null;
    }
}
