/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import com.ericsson.ssa.sip.DialogSet;
import com.ericsson.ssa.sip.PathNode.Type;
import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipSessionDialogImpl;
import com.ericsson.ssa.sip.SipSessionManagerBase;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;

import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;

import javax.servlet.sip.Address;

import org.jvnet.glassfish.comms.replication.dialogmgmt.HADialogFragment;

import java.io.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import org.jvnet.glassfish.comms.util.LogUtil;

public class HASipSession extends SipSessionDialogImpl {
    
    private static final Logger _logger = Logger.getLogger(LogUtil.SIP_LOG_DOMAIN);
    // XXX make configurable?
    private static final long DELTA = (10 * 1000); // 10 secs
    private transient boolean isDirty = false;

    // The version of this SipSession
    private transient AtomicLong version = new AtomicLong(-1);

    private transient SipSessionExtraParams extraParams;

    private transient String previousOwnerInstanceName;
    private transient String currentOwnerInstanceName;
    
    protected transient boolean isReplicated = false;
    private transient long expirationTime;

    /**
     * Constructor.
     */
    public HASipSession(SipSessionManagerBase manager, DialogSet set,
        Address to, SipApplicationSessionImpl appSession, String handler,
        Type type) {
        super(manager, set, to, appSession, handler, type);
        version = new AtomicLong(-1);
        extraParams = new SipSessionExtraParams(this);
        currentOwnerInstanceName = ReplicationUtil.getInstanceName();
    }

    /**
     * Gets the name of the instance.that previously owned this SipSession.
     *
     * @return The name of the instance that previously owned this SipSession
     */
    public String getPreviousOwnerInstanceName() {
        return previousOwnerInstanceName;
    }

    /**
     * Gets the name of the instance.that currently owns this SipSession.
     *
     * @return The name of the instance that currently owns this SipSession
     */
    public String getCurrentOwnerInstanceName() {
        return currentOwnerInstanceName;
    }

    /**
     * Checks whether this session is dirty or not.
     *
     * @return true if this session has been marked as dirty, false otherwise
     */
    public boolean isDirty() {
        return isDirty;
    }

    /**
     * Marks this session as dirty or non-dirty.
     *
     * @param isDirty The dirty flag
     */
    public void setDirty(boolean isDirty) {
        this.isDirty = isDirty;
    }

    /**
     * Increments the version of this SipSession.
     */
    public void incrementVersion() {
        version.incrementAndGet();
    }

    /**
     * Gets the version of this SipSession.
     *
     * @return The version of this SipSession
     */
    public long getVersion() {
        return version.get();
    }

    /**
     * Sets the version of this SipSession.
     *
     * @value The new version of this SipSession
     */
    public void setVersion(long value) {
        version.getAndSet(value);
    }
    
    /** 
     * is the SipSession persistent
     */    
    public boolean isReplicated() {
        return isReplicated;
    }
    
    /** 
     * this sets the persistent flag
     */    
    public void setReplicated(boolean value) {
        isReplicated = value;
    }     

    public SipSessionExtraParams getExtraParameters() {
        return extraParams;
    }

    /**
     * Updates this SipSession with the given extra 
     * parameters.
     *
     * @param extraParams The serialized SipSessionExtraParams 
     * used to update this
     * SipSession
     */
    public void update(byte[] extraParamsState) {        
        
        SipSessionExtraParams extraParams = null;
        if(extraParamsState != null) {
            try {
                extraParams 
                    = (SipSessionExtraParams)ReplicationState.getObjectValue(extraParamsState);
                this.setPFieldCSeq(extraParams.getCseq());
                this.updateExpirationTimeOnSSOnly(extraParams.getExpirationTime());
            } catch (Exception ex) {
                _logger.warning("unable to deserialize SipSession extraParams for id " + this.getId() + ":" + ex);
            }
        } 
    }       

    private void updateExpirationTimeOnSSOnly(long anExpirationTime) {
        expirationTime = anExpirationTime;
    }
    
    public void setExpirationTime(long anValue) {
        updateExpirationTimeOnSSOnly(anValue);
        // load the DF child and update the expiration time if shorter
        // TODO there is a problem. If the application invokes
        // setExpires(100) and then setExpires(10), the SAS timer will be reset to
        // 10 minutes, but the DF expiration time will remain at 100 minutes.
        // since we just take the max of the the DF and the SS, not of al the SSes
        // Taking the max of all the SSes would be possible, but would require
        // (part) of all the other trees to be loaded as well.
        // We assume that this is a rare case and the worst that can happen is
        // that the DF only times out long after all the SSes have timedout.
        // Which is no big deal.
        HADialogFragment df = (HADialogFragment) getDF();
        if (df.getExpirationTime() < (anValue + DELTA)) {
            df.setExpirationTime(anValue + DELTA);
        }
    }
    
    public long getExpirationTime() {
        return expirationTime;
    }

    private void writeObject(ObjectOutputStream oos)
            throws IOException {
        oos.writeObject(version);
        oos.writeObject(currentOwnerInstanceName);
        oos.writeLong(expirationTime); // XXX should we write or not?
    }

    private void readObject(ObjectInputStream in)
            throws IOException, ClassNotFoundException {
        version = (AtomicLong) in.readObject();
        previousOwnerInstanceName = (String) in.readObject();
        currentOwnerInstanceName = ReplicationUtil.getInstanceName();
        expirationTime = in.readLong();
        extraParams = new SipSessionExtraParams(this);
    }

    public String toString() {
        return description + ":version:" + getVersion(); 
    }
}
