/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.config;

import com.ericsson.ssa.sip.ServiceHandler;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;

import java.io.IOException;

import javax.servlet.RequestDispatcher;
import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;


/**
 * This dispatcher handles forwarding of SIP requests/responses according to SSA
 * 1.0 section 10.2.5. According to that specification the
 * {@link #include(ServletRequest, ServletResponse)} have no meaning in the SIP
 * context.
 */
public class SipRequestDispatcher implements RequestDispatcher {
    private String servletName;
    private ServiceHandler serviceHandler;

    /**
     * Create a dispatcher for the SIP servlet with the specified name.
     *
     * @param servletName
     *           name of servlet
     * @param serviceHandler
     *           the service handler via which servlets can be invoked
     */
    public SipRequestDispatcher(String servletName,
        ServiceHandler serviceHandler) {
        this.servletName = servletName;
        this.serviceHandler = serviceHandler;
    }

    /**
     * {@inheritDoc}
     */
    public void forward(ServletRequest request, ServletResponse response)
        throws ServletException, IOException {
        SipApplicationSessionImpl session = null;

        if (request != null) {
            if (request instanceof SipServletRequestImpl) {
                session = (SipApplicationSessionImpl) ((SipServletRequestImpl) request).getApplicationSession();
            } else {
                throw new IllegalArgumentException(
                    "Only SIP requests are allowed.");
            }
        }

        if (response != null) {
            if (response instanceof SipServletResponseImpl) {
                if (session == null) {
                    session = (SipApplicationSessionImpl) ((SipServletResponseImpl) response).getApplicationSession();
                }
            } else {
                throw new IllegalArgumentException(
                    "Only SIP responses are allowed.");
            }
        }

        if (session == null) {
            throw new IllegalArgumentException(
                "A SIP request or a SIP response must be supplied.");
        }

        Servlet servlet = serviceHandler.getHandler(session.getName(),
                servletName);

        if (servlet == null) {
            throw new IllegalStateException("SIP servlet " + servletName +
                " not found.");
        }

        servlet.service(request, response);
    }

    /**
     * This method has no meaning in the SIP context according to SSA 1.0 section
     * 10.2.5
     */
    public void include(ServletRequest request, ServletResponse response)
        throws ServletException, IOException {
        throw new UnsupportedOperationException(
            "This method have no meaning for SIP servlets");
    }
}
