/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import java.net.URI;


/**
 * HttpHost holds all information necessary to describe
 * an HTTP connection to a specific host.
 *
 * @author epiesan
 * @since Apr 16, 2006
 */
public final class HttpHost implements Cloneable {
    private final static int HTTP_DEFAULT_PORT = 80;
    private String hostname;
    private int port;

    /**
     * Constructor for HttpHost.
     *
     * @param hostname The hostname (may not be null).
     * @param port The port. Specify -1 to use the default port for HTTP.
     */
    public HttpHost(final String hostname, int port) {
        if (hostname == null) {
            throw new IllegalArgumentException("Host name must be specified.");
        }

        this.hostname = hostname;

        if (port < 0) {
            this.port = HTTP_DEFAULT_PORT; // Set default port for HTTP
        } else {
            this.port = port;
        }
    }

    /**
     * Constructor for HttpHost.
     *
     * @param hostname The hostname (may not be null).
     */
    public HttpHost(final String hostname) {
        this(hostname, -1);
    }

    /**
     * Constructor for HttpHost.
     *
     * @param uri The URI.
     */
    public HttpHost(final URI uri) {
        this(uri.getHost(), uri.getPort());
    }

    /**
     * Constructor needed for implementation of clone().
     *
     * @param source The source HttpHost to copy values from.
     */
    private HttpHost(final HttpHost source) {
        this.hostname = source.hostname;
        this.port = source.port;
    }

    /**
     * Returns the host name.
     *
     * @return The host name.
     */
    public String getHostname() {
        return hostname;
    }

    /**
     * Returns the port of this host.
     *
     * @return The port.
     */
    public int getPort() {
        return port;
    }

    /**
     * Clones this HttpHost instance.
     */
    public Object clone() {
        return new HttpHost(this);
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(final Object obj) {
        if (obj instanceof HttpHost) {
            if (obj == this) {
                return true;
            }

            HttpHost that = (HttpHost) obj;

            if (!this.hostname.equalsIgnoreCase(that.hostname)) {
                return false;
            }

            if (this.port != that.port) {
                return false;
            }

            return true;
        } else {
            return false;
        }
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        int hash = 7;

        hash = (31 * hash) + hostname.hashCode();
        hash = (31 * hash) + port;

        return hash;
    }

    /**
     * Returns a string representation of the HttpHost instance.
     * The return string is in the form 'http://<host>:<port>'.
     *
     * @return The string representation of this HttpHost instance.
     */
    public String toString() {
        StringBuffer s = new StringBuffer();
        s.append("http://");
        s.append(this.hostname);

        if (this.port != HTTP_DEFAULT_PORT) {
            s.append(':');
            s.append(this.port);
        }

        return s.toString();
    }
}
