/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container.sim;

import com.ericsson.ssa.container.annotation.AnnotationHandler;
import com.ericsson.ssa.sip.Dispatcher;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.ServiceHandler;
import com.ericsson.ssa.sip.SessionManager;
import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.SipServletReadOnlyRequestImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.SipURIImpl;
import com.ericsson.ssa.sip.SipSessionBase;

import org.apache.catalina.ContainerEvent;
import org.apache.catalina.ContainerListener;
import org.apache.catalina.Context;
import org.apache.catalina.LifecycleEvent;
import org.apache.catalina.LifecycleListener;
import org.apache.catalina.util.StringManager;
import org.apache.catalina.valves.ValveBase;

import org.apache.coyote.tomcat5.CoyoteResponse;

import java.io.IOException;
import java.io.Serializable;

import java.util.ArrayList;
import java.util.List;
import java.util.TreeMap;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.sip.Address;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipApplicationRouter;
import javax.servlet.sip.SipApplicationRouterInfo;
import javax.servlet.sip.SipApplicationRoutingDirective;
import javax.servlet.sip.SipApplicationRoutingRegion;
import javax.servlet.sip.SipApplicationRoutingRegionType;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipRouteModifier;
import javax.servlet.sip.SipSession;
import javax.servlet.sip.TooManyHopsException;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.container.callflow.CallflowResolver;


public class ApplicationDispatcher implements Layer,
    ContainerListener, LifecycleListener, ServiceHandler {
    public static final String APPLICATION_DISPATCHER = "com.ericsson.ssa.container.sim.ApplicationDispatcher/1.0";

    /* Singelton instance */
    private static ApplicationDispatcher m_singeltonInstance = new ApplicationDispatcher();
    private static TreeMap<String, ServletDispatcher> m_contextMapping = new TreeMap<String, ServletDispatcher>();
    private Logger m_logger = Logger.getLogger("SipContainer");
    private Layer m_nextLayer = null;
    private SipApplicationRouter m_applicationRouter = null;
    private ClassLoader saved = null;

    // this is an inner helper class to be able to come back
    // to the ApplicationDispatcher after invoking servlet
    private Dispatcher m_serviceDispatcher = new Dispatcher() {
            public void dispatch(SipServletRequestImpl req) {
                dispatchViaStatelessProxy(req);
            }

            public void dispatch(SipServletResponseImpl resp) {
                dispatch(resp); // NOTE: calls AD.dispatch(resp)
            }
        };

    private Reporter _reporter;
    
    public void setReporters(String reporters){
        _reporter = CallflowResolver.getInstance().getReporter(reporters);
    }
    
    public Reporter getReporter(){
        return _reporter;
    }
        
    protected ApplicationDispatcher() {
        super();
    }

    public synchronized void start() {
        if (m_applicationRouter == null) {
            throw new RuntimeException(
                "Can not start without application router.");
        }

        setRouterContext();
        m_applicationRouter.init(new ArrayList<String>(
                m_contextMapping.keySet()));
        resetRouterContext();
    }

    public synchronized void stop() {
        setRouterContext();
        m_applicationRouter.destroy();
        resetRouterContext();
    }

    public void lifecycleEvent(LifecycleEvent event) {
    }

    public void containerEvent(ContainerEvent event) {
        m_logger.fine("Got Container event!");
    }

    /**
     * The method is a part of the dispatch chain. For the initial request the
     * method will try to call the ServletDispatcher to invoke some servlets,
     * otherwise it will pop up the next dispatcher to invoke.
     *
     * @param request
     *           A <code>SipServletRequestImpl</code>.
     */
    public void dispatch(SipServletRequestImpl request) {
        // Check if request is initial
        if (request.isInitial()) {
            // request.pushApplicationDispatcher(m_serviceDispatcher);
            request.pushApplicationDispatcher(this);
            // JSR 289
            invokeServletDispatchers(request);
        } else { // Subsequent request, should go through the default proxy.
                 // request.popDispatcher().dispatch(request);
            dispatchViaStatelessProxy(request);
        }
    }

    /*
     *
     */
    private void invokeServletDispatchers(SipServletRequestImpl request) {
        request.setSent(false);

        Serializable stateInfo = (Serializable)request.getStateInfo();
        String regionStr = null;
        SipApplicationRoutingRegion region = null;
        SipSessionBase session = request.getSessionImpl();
        if (session != null){
                session.access(); // modify the access times
        	regionStr = session.getRegion();
        	if (regionStr != null){
        		region = new SipApplicationRoutingRegion(regionStr, 
        				SipApplicationRoutingRegionType.valueOf(regionStr));	
        	}
        }

        setRouterContext();
        SipApplicationRouterInfo info = 
    		m_applicationRouter.getNextApplication(
				new SipServletReadOnlyRequestImpl(request),
				region, 
				SipApplicationRoutingDirective.NEW,
				stateInfo);
        resetRouterContext();

        Serializable newStateInfo = null;

        if (info != null) {
            newStateInfo = info.getStateInfo();
        }

        request.setStateInfo(newStateInfo);

        while (info != null) {
            
        	SipRouteModifier mod = info.getRouteModifier();
        	String route = info.getRoute();
            if (mod.equals(SipRouteModifier.ROUTE)) {
        		try {
        			Address a = 
        				SipFactoryImpl.getInstance().createAddress(route);
        			request.pushRoute(a);
        		} catch (ServletParseException e) {
        			m_logger.severe(
    					"The following address in ApplicationRouterInfo " + 
    					"does not have a correct format and will not be " +
    					"added: " + route);
        		}
            	dispatchViaStatelessProxy(request);
                return;
            } else if (mod.equals(SipRouteModifier.ROUTE_BACK)){
        		// First push a route that will eventually send the request 
            	// back to this server 
            	try{
            		// TODO what is the correct way to create a route to this host?
            		Address a = SipFactoryImpl.getInstance().createAddress("localhost");
            		request.pushRoute(a);
            	} catch (ServletParseException e) {
            		m_logger.severe("Could not push a route back to this host");
            	}
            	// now push the remaining route
            	try {
        			Address a = 
        				SipFactoryImpl.getInstance().createAddress(route);
        			request.pushRoute(a);
        		} catch (ServletParseException e) {
        			m_logger.severe(
    					"The following address in ApplicationRouterInfo " + 
    					"does not have a correct format and will not be " +
    					"added: " + route);
        		}
            	dispatchViaStatelessProxy(request);
                return;
            } else if (mod.equals(SipRouteModifier.CLEAR_ROUTE)) {
            	// clear_route -> remove popped route
                request.setPoppedRoute(null);
                dispatchViaStatelessProxy(request);
                return;
            }

            // else treat as NO_ROUTE and look for application name
            String application = info.getNextApplicationName();

            if (application == null) {
                dispatchViaStatelessProxy(request);

                return;
            }

            ServletDispatcher sd = m_contextMapping.get(application);

            if (sd==null){
	            m_logger.log(
            		Level.SEVERE, "Unable to invoke application " + application 
            		+ ", check your AR configuration!");

	            SipServletResponseImpl response = request.createTerminatingResponse(500);
                response.popDispatcher().dispatch(response);
                return;
            }
            boolean sent = false;
            try {
                // TODO should be only activate if filter option on
                // not for standard JSR289 protected by a 64T1 timer
                request.activateSent(); //Activate the function
                sd.invoke(request, info);
            } catch (TooManyHopsException tmhe) {
                if (m_logger.isLoggable(Level.FINE)) {
                    m_logger.log(Level.FINE, "Too Many Hops Exception ", tmhe);
                }

                SipServletResponseImpl response = request.createTerminatingResponse(483);
                response.popDispatcher().dispatch(response);

                return;
            } catch (Exception e) {
                if (m_logger.isLoggable(Level.FINE)) {
                    m_logger.log(Level.FINE, "Error invoking service ", e);
                }

                SipServletResponseImpl response = request.createTerminatingResponse(500);
                response.popDispatcher().dispatch(response);

                return;
            } finally {
                sent = request.hasSent();
                request.clearSent(); //Important not to leak mem
            }

            if (sent) {
                return;
            } else { // Filter serlet app, maybe make sure that it has not
                     // been changed!?

                setRouterContext();
                info = m_applicationRouter.getNextApplication(
                		new SipServletReadOnlyRequestImpl(request),
                		region,
                        SipApplicationRoutingDirective.CONTINUE, 
                        newStateInfo);
                resetRouterContext();

                newStateInfo = null;

                if (info != null) {
                    newStateInfo = info.getStateInfo();
                }

                request.setStateInfo(newStateInfo);
            }
        }

        dispatchViaStatelessProxy(request); // TODO should it be 404?
    }

    /**
     * The method will pop up and invoke the next <code>Dispatcher</code>.
     *
     * @param response
     *           A <code>SipServletResponseImpl</code>.
     */
    public void dispatch(SipServletResponseImpl response) {
        // The transaction layer populates the dispatchers
        // for the request, so here we pop the dispatchers
        try {
            response.popDispatcher().dispatch(response);
        } catch (Exception ex) {
            m_logger.log(Level.SEVERE, "Unhandled exception ", ex);
        }
    }

    private void dispatchViaStatelessProxy(SipServletRequestImpl request) {
        // Internal stateless proxy, decrease Max-Forwards by one.
        int maxForwards = request.getMaxForwards();
        --maxForwards;

        if (maxForwards <= 0) {
            SipServletResponseImpl response = request.createTerminatingResponse(483);

            if (response != null) {
                dispatch(response);
            }

            return;
        }

        request.setMaxForwards(maxForwards);
        log("The default proxy will decrease the Max-Forwards by one.");

        // When a request already landed up here it should goes down directly to
        // the lower layer e.g. SessionManager, whatever is next layer. Avoid
        // making dependency to the SessionManager.
        // The UAC and proxy application path are different.
        // UAC application path (ApplicationDispatcher$, SessionManager,
        // ResolverManager, ...)
        // Proxy application path (SessionManager, ResolverManager, ...)
        Dispatcher dispatcher = request.popDispatcher();

        if (dispatcher instanceof ApplicationDispatcher ||
                (dispatcher == m_serviceDispatcher)) {
            // SessionManager should be popped as the next called layer.
            request.popDispatcher().dispatch(request);
        } else {
            // already SessionManager.
            dispatcher.dispatch(request);
        }
    }

    /**
     * Registers a Servlet Dispatcher associated to a application Name
     *
     * @param applicationName
     *           The application name.
     * @param servletDispatcher
     *           A <code>ServletDispatcher</code>.
     */
    public void addServletDispatcher(String applicationName,
        ServletDispatcher servletDispatcher) {
        m_contextMapping.put(applicationName, servletDispatcher);

        // Notify the application router
        List<String> apps = new ArrayList<String>(1);
        apps.add(applicationName);

        setRouterContext();
        m_applicationRouter.applicationDeployed(apps);
        resetRouterContext();
    }

    /**
     * Deregisters a Servlet Dispatcher for a specified application
     *
     * @param applicationName
     */
    public void removeServletDispatcher(String applicationName) {
        m_contextMapping.remove(applicationName);

        // Notify the application router
        List<String> apps = new ArrayList<String>(1);
        apps.add(applicationName);

        setRouterContext();
        m_applicationRouter.applicationUndeployed(apps);
        resetRouterContext();
    }

    // Helper method for serialization
    public ServletDispatcher getServletDispatcher(String applicationName) {
        return m_contextMapping.get(applicationName);
    }

    /**
     * Returns the SipServlet of the application name and SipSession handler
     *
     * @param applicationName
     *           the application name
     * @param handler
     *           the SipSession handler
     * @return SipServlet of the application name and SipSession handler
     */
    public Servlet getHandler(String applicationName, String handler) {
        ServletDispatcher disp = m_contextMapping.get(applicationName);

        if (disp != null) {
            return disp.getHandler(handler);
        }

        return null;
    }

    /**
     * @param request
     */
    public void next(SipServletRequestImpl request) {
        LayerHelper.next(request, this, m_nextLayer);
    }

    /**
     * @param response
     */
    public void next(SipServletResponseImpl response) {
        LayerHelper.next(response, this, m_nextLayer);
    }

    /**
     * @param layer
     */
    public void registerNext(Layer layer) {
        m_nextLayer = layer;
    }

    /**
     * Return an instance of <code>ApplicationDispatcher</code>.
     *
     * @return A <code>ApplicationDispatcher</code>.
     */
    public static ApplicationDispatcher getInstance() {
        return m_singeltonInstance;
    }

    /**
     * Set the Application Router from an external location. This way it would
     * be possible to instantiate a different Application Router e.g. in a
     * servlet and use servlet deployment to get the AR on the container.
     *
     * @param aRouter
     */
    public void setApplicationRouter(SipApplicationRouter aRouter) {
        m_logger.log(Level.FINE, "Setting Application Router: " + aRouter);

        // TODO think about upgrade
        this.m_applicationRouter = aRouter;

        setRouterContext();
        aRouter.init();
        resetRouterContext();

        AnnotationHandler.processSipAnnotation(m_applicationRouter);
    }

    public void setApplicationRouterClass(String className) {
        m_logger.log(Level.FINE,
            "Instantiating Application Route : " + className);

        try {
            Class ar = Class.forName(className);
            m_applicationRouter = (SipApplicationRouter) ar.newInstance();
            AnnotationHandler.processSipAnnotation(m_applicationRouter);
        } catch (ClassNotFoundException e) {
            m_logger.log(Level.WARNING,
                "Faild to instantiate ApplicationRouter", e);
        } catch (InstantiationException e) {
            m_logger.log(Level.WARNING,
                "Faild to instantiate ApplicationRouter", e);
        } catch (IllegalAccessException e) {
            m_logger.log(Level.WARNING,
                "Faild to instantiate ApplicationRouter", e);
        }
    }

    /**
     * Push a service dispatcher on top of the application stack.
     *
     * @param request
     *           On which the service dispatcher will be pushed.
     */
    public void pushServiceDispatcher(SipServletRequestImpl request) {
        request.pushApplicationDispatcher(m_serviceDispatcher);
    }

    /**
     * Return descriptive information about this Valve implementation.
     */
    public String getInfo() {
        return (APPLICATION_DISPATCHER);
    }

    private void log(String log) {
        if (m_logger.isLoggable(Level.FINE)) {
            m_logger.log(Level.FINE, log);
        }
    }


    /**
     * Utility method for setting the context classloader. 
     * Move this to common module?
     */
    private ClassLoader _setContextClassLoader(ClassLoader newClassLoader) {

        // Can reference only final local variables from dopriveleged block
        final ClassLoader classLoaderToSet = newClassLoader;

        final Thread currentThread = Thread.currentThread();
        ClassLoader originalClassLoader = currentThread.getContextClassLoader();

        if (classLoaderToSet != originalClassLoader) {
            if (System.getSecurityManager() == null) {
                currentThread.setContextClassLoader(classLoaderToSet);
            } else {
                java.security.AccessController.doPrivileged(
                        new java.security.PrivilegedAction() {
                            public java.lang.Object run() {
                                currentThread.setContextClassLoader
                                (classLoaderToSet);
                                return null;
                            }
                        }
                );
            }
        }
        return originalClassLoader;
    }

    /**
     * Sets the context for invoking the application router
     */
    private void setRouterContext() {
        try {
            ClassLoader routerCL = 
            m_applicationRouter.getClass().getClassLoader();
            saved = _setContextClassLoader(routerCL);
        } catch (Exception e) {
            if (m_logger.isLoggable(Level.WARNING)) {
                m_logger.log(Level.WARNING, e.getMessage(), e);
            }
        }
    }

    /**
     * Resets the context after invoking the application router
     */
    private void resetRouterContext() {
        try {
            _setContextClassLoader(saved);
        } catch (Exception e) {
            if (m_logger.isLoggable(Level.WARNING)) {
                m_logger.log(Level.WARNING, e.getMessage(), e);
            }
        }
    }
}
