/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.sip.PathNode.Type;

import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.sip.Address;
import javax.servlet.sip.B2buaHelper;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipServletMessage;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;
import javax.servlet.sip.SipSession;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.TooManyHopsException;
import javax.servlet.sip.UAMode;
import javax.servlet.sip.URI;


/**
 * Implementation of jsr289 B2buaHelper interface.
 *
 * @author ehsroha
 */
public class B2buaHelperImpl implements B2buaHelper {
    private Logger _log = Logger.getLogger("SipContainer");
    private SipServletRequestImpl _originalRequest = null;
    private final SipFactory _sf;

    public B2buaHelperImpl(SipServletRequestImpl req) {
        _sf = req.getApplicationSessionImpl().getServletDispatcher()
                 .getSipFactory();

        if (req.isInitial()) {
            _originalRequest = req;
        }
    }

    public synchronized SipServletRequest createRequest(
        SipServletRequest origRequest, boolean linkedSessions,
        Map<String, Set<String>> headerMap)
        throws IllegalArgumentException, TooManyHopsException {
        if (origRequest == null) {
            throw new NullPointerException();
        }

        if (!origRequest.isInitial()) {
            throw new IllegalStateException();
        }

        if (origRequest.getMaxForwards() == 0) {
            throw new TooManyHopsException();
        }

        if ((headerMap != null) && !headerMap.isEmpty()) {
            // are there any illegal system headers to add?
            if (isSystemHeader(headerMap.keySet(),
                        (SipServletRequestImpl) origRequest, true)) {
                // not allowed to add system headers...
                throw new IllegalArgumentException();
            }
        }

        // time to give birth to a new request...
        SipServletRequest copiedReq = null;

        // need special treatment for from and to headers
        Set<String> from = headerMap.remove("From");
        Set<String> to = headerMap.remove("To");

        try {
            if ((from != null) && (to != null)) {
                copiedReq = _sf.createRequest(origRequest.getApplicationSession(),
                        origRequest.getMethod(), from.iterator().next(),
                        to.iterator().next());
            } else if (from != null) {
                copiedReq = _sf.createRequest(origRequest.getApplicationSession(),
                        origRequest.getMethod(), from.iterator().next(),
                        origRequest.getTo().toString());
            } else if (to != null) {
                copiedReq = _sf.createRequest(origRequest.getApplicationSession(),
                        origRequest.getMethod(),
                        origRequest.getFrom().toString(), to.iterator().next());
            } else {
                copiedReq = _sf.createRequest(origRequest, false);
            }
        } catch (ServletParseException e) {
            throw new IllegalArgumentException(e);
        }

        // lets add non system headers
        if ((headerMap != null) && !headerMap.isEmpty()) {
            Iterator<String> names = headerMap.keySet().iterator();
            Iterator<Set<String>> values = headerMap.values().iterator();

            while (names.hasNext()) {
                setHeader(names.next(), values.next(),
                    (SipServletRequestImpl) copiedReq);
            }
        }

        // always link requests...
        if (isAlreadyLinked((SipServletRequestImpl) origRequest)) {
            throw new IllegalArgumentException();
        } else {
            // requests are not linked, lets link them...
            linkRequests((SipServletRequestImpl) origRequest,
                (SipServletRequestImpl) copiedReq);
        }

        if (linkedSessions) {
            SipSession session1 = origRequest.getSession();
            SipSession session2 = copiedReq.getSession();

            if ((session1 == null) || (session2 == null)) {
                throw new NullPointerException();
            }

            // are sessions consistent?
            if ((isValidSessions((SipSessionBase) session1,
                        (SipSessionBase) session2) &&
                    (origRequest.getApplicationSession(false) != null) &&
                    session1.getApplicationSession()
                                .equals(origRequest.getApplicationSession())) == false) {
                throw new IllegalArgumentException();
            }

            if (!isAlreadyLinkedToEachOther((SipSessionBase) session1,
                        (SipSessionBase) session2)) {
                // sessions are not linked to each other...
                if (!isNotLinkedToAnyone((SipSessionBase) session1,
                            (SipSessionBase) session2)) {
                    // ...but one session is linked somewhere else.
                    throw new IllegalArgumentException();
                }

                // lets link sessions.
                linkSipSessions((SipSessionBase) session1,
                    (SipSessionBase) session2);
            }
        }

        return copiedReq;
    }

    public synchronized SipServletRequest createRequest(SipSession session2,
        SipServletRequest origRequest, Map<String, Set<String>> headerMap)
        throws IllegalArgumentException {
        if ((origRequest == null)) {
            throw new NullPointerException();
        }

        if (isAlreadyLinked((SipServletRequestImpl) origRequest)) {
            throw new IllegalArgumentException();
        }

        SipSession session1 = origRequest.getSession();

        if ((session1 == null) || (session2 == null)) {
            throw new NullPointerException();
        }

        // are sessions consistent?
        if (isValidSessions((SipSessionBase) session1, (SipSessionBase) session2) &&
                (origRequest.getApplicationSession(false) != null) &&
                (session1.getApplicationSession()
                             .equals(origRequest.getApplicationSession()) == false)) {
            throw new IllegalArgumentException();
        }

        if ((headerMap != null) && !headerMap.isEmpty()) {
            // are there any illegal system headers to add?
            if (isSystemHeader(headerMap.keySet(),
                        (SipServletRequestImpl) origRequest, false)) {
                // not allowed to add system headers...
                throw new IllegalArgumentException();
            }
        }

        boolean shouldLinkSessions = false;

        if (!isAlreadyLinkedToEachOther((SipSessionBase) session1,
                    (SipSessionBase) session2)) {
            // sessions are not linked to each other...
            if (!isNotLinkedToAnyone((SipSessionBase) session1,
                        (SipSessionBase) session2)) {
                // ...but one session is linked somewhere else.
                throw new IllegalArgumentException();
            }

            // ...lets link them later.
            shouldLinkSessions = true;
        }

        SipServletRequest subRequest = session2.createRequest(origRequest.getMethod());

        // lets add non system headers
        if ((headerMap != null) && !headerMap.isEmpty()) {
            Iterator<String> names = headerMap.keySet().iterator();
            Iterator<Set<String>> values = headerMap.values().iterator();

            while (names.hasNext()) {
                setHeader(names.next(), values.next(),
                    (SipServletRequestImpl) subRequest);
            }
        }

        if (shouldLinkSessions) {
            // lets link sessions.
            linkSipSessions((SipSessionBase) session1, (SipSessionBase) session2);
        }

        // requests are not linked, lets link them...
        linkRequests((SipServletRequestImpl) origRequest,
            (SipServletRequestImpl) subRequest);

        return subRequest;
    }

    private void setHeader(String name, Set<String> values,
        SipServletRequestImpl req1) throws IllegalArgumentException {
        if (Header.isSingleLineHeader(name) && (values.size() > 1)) {
            throw new IllegalArgumentException("Header " + name +
                " should be a singel line header.");
        }

        Header header = Header.createFormated(name, req1);
        boolean isFirst = true;

        for (String value : values) {
            header.setValue(value, isFirst);
            isFirst = false;
        }

        // since the initial request can add new from and to header,
        // and also modify Contact header,
        // make sure they have proper address format...
        if ((Header.FROM.equalsIgnoreCase(name) ||
                Header.TO.equalsIgnoreCase(name)) ||
                Header.CONTACT.equalsIgnoreCase(name)) {
            try {
                header.getAddressValue();
            } catch (ServletParseException e) {
                throw new IllegalArgumentException("Header " + name +
                    " is not valid.", e);
            }
        }

        req1.setHeader(header);
    }

    private boolean isSystemHeader(Set<String> headers,
        SipServletMessageImpl message, boolean isInitial) {
        for (String name : headers) {
            // it's allowed to change From and To header of initial request
            if (!(isInitial &&
                    (Header.FROM.equalsIgnoreCase(name) ||
                    Header.TO.equalsIgnoreCase(name)))) {
                if (Header.isSystemHeader(name, message)) {
                    return true;
                }
            }
        }

        return false;
    }

    private boolean isAlreadyLinked(SipServletRequestImpl req1) {
        return req1.getLinkedRequest() != null;
    }

    private void linkRequests(SipServletRequestImpl origReq,
        SipServletRequestImpl linkedReq) {
        origReq.setLinkedRequest(linkedReq);
        linkedReq.setLinkedRequest(origReq);
        linkedReq.setB2buaHelper(this);
    }

    private void updateTransactionStack(SipServletResponseImpl resp) {
        // need to update transaction path since there are new nodes.
        // lets start by cleaning the top of the stack...
        PathNode p = null;
        Iterator<PathNode> i = resp.getDialog().getCaller2CalleePath();

        if (i.hasNext()) {
            // skip this since it is the uas, which is not on the stack
            i.next();
        }

        while (i.hasNext()) {
            resp.popDispatcher();
        }

        // ...and replace it with the new nodes.
        i = resp.getDialog().getCaller2CalleePath();

        if (i.hasNext()) {
            // skip this since it is the uas, which is not on the stack
            i.next();
        }

        while (i.hasNext()) {
            // add the rest of the nodes..
            p = i.next();
            resp.pushTransactionDispatcher(p);
        }
    }

    public synchronized SipServletResponse createResponseToOriginalRequest(
        SipSession session, int status, String reasonPhrase) {
        return createResponseToOriginalRequest((SipSessionBase) session,
            status, reasonPhrase);
    }

    private SipServletResponse createResponseToOriginalRequest(
        SipSessionBase session2, int status, String reasonPhrase) {
        // 1. create response with original request
        // 2. verify that only final response 200OK has been created before
        // otherwise throw IllegalStateException: included in createResponse
        if (_originalRequest == null) {
            throw new IllegalStateException(
                "Only allowed to create response to original request.");
        }

        SipSessionBase session = null;

        TempSession tempSession = null;

        // some trix to make jsr289 12.5.1 Cloning and Linking working.
        // see also getLinkedSession()
        if (session2 instanceof TempSession) {
            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE, "Convert FacadeSession -> session");
            }

            tempSession = (TempSession) session2;
            session = tempSession.getDerivedUACSessionToLink();
        } else {
            session = session2;
        }

        SipServletResponseImpl resp = null;

        if (session.isDerived()) {
            // maybe it's a subsequent response or retransmission
            // and hence sessions might already be linked
            SipSessionBase linkedSession = (SipSessionBase) getLinkedSession(session,
                    false);

            if (linkedSession != null) {
                DialogFragment dialog = _originalRequest.getDialog()
                                                        .getDialogSet()
                                                        .searchForDialog(linkedSession.getToTag(),
                        linkedSession.getFragmentId());

                if (dialog != null) {
                    // make a copy of the original request, can't change the original
                    SipServletRequestImpl clone = (SipServletRequestImpl) _originalRequest.clone();

                    // adjust the clone, needed because of new fragmentId,
                    // which changes contact
                    clone.setDialog(dialog);

                    resp = (SipServletResponseImpl) clone.createResponse(status,
                            reasonPhrase);

                    // point back again...
                    resp.setRequest(_originalRequest);

                    resp.setSession((SipSessionBase) linkedSession);

                    updateTransactionStack(resp);

                    if (_log.isLoggable(Level.FINE)) {
                        _log.log(Level.FINE,
                            "Session is derivied and linked, uac session = " +
                            session + ", uas session = " + linkedSession +
                            " , linked dialog =" + dialog + ", " + resp);
                    }
                } else {
                    throw new IllegalStateException(
                        "Derived session is already linked but dialog is not established.");
                }
            } else {
                // need to create a new structure (dialog, etc.)
                // of the derived linked session

                // fetch the original uas
                PathNode uas = _originalRequest.getDialog().getLast();

                // clone only until uas, i.e. exclude it
                DialogFragment d = _originalRequest.getDialog()
                                                   .cloneFromCallerToCalleeUntil(uas,
                        true);

                // create the new uas replacing the original
                uas = new UA(_originalRequest.getApplicationSessionImpl(), false);

                // lets add the current uas to the application & transaction path
                d.addToPath(uas);

                // make a copy of the original request, can't change the original
                SipServletRequestImpl clone = (SipServletRequestImpl) _originalRequest.clone();

                // adjust the clone, needed because of new fragmentId,
                // which changes contact
                clone.setDialog(d);

                // create response
                resp = (SipServletResponseImpl) clone.createResponse(status,
                        reasonPhrase);

                // point back again...
                resp.setRequest(_originalRequest);

                // need to update transaction stack
                updateTransactionStack(resp);

                // save session in response.
                // It will later be linked to new derived session.
                //
                // It is also used as signal of b2buaHelper usage.
                if (tempSession != null) {
                    resp.setB2buaHelperTempSession(tempSession);
                }

                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Session is derivied but not linked, uac session = " +
                        session + ", uas session = " + linkedSession +
                        " , linked dialog =" + d + ", " + resp);
                }
            }
        } else {
            resp = (SipServletResponseImpl) _originalRequest.createResponse(status,
                    reasonPhrase);

            if (!isAlreadyLinkedToEachOther(session,
                        _originalRequest.getSessionImpl())) {
                // lets link the sessions...
                linkSipSessions(session, _originalRequest.getSessionImpl());

                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Session is original and was not linked, uac session = " +
                        session + ", uas session = " +
                        _originalRequest.getSessionImpl() +
                        " , linked dialog =" + _originalRequest.getDialog() +
                        ", " + resp);
                }
            } else {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Session is original and already linked, uac session = " +
                        session + ", uas session = " +
                        _originalRequest.getSessionImpl() +
                        " , linked dialog =" + _originalRequest.getDialog() +
                        ", " + resp);
                }
            }
        }

        return resp;
    }

    public synchronized SipSession getLinkedSession(SipSession session) {
        return getLinkedSession((SipSessionBase) session, true);
    }

    /**
     * Special handling to support jsr289 12.5.1 Cloning and Linking. Furnish the
     * UAS-2 (a clone of UAS-1) when the getLinkedSipSession is invoked with
     * UAC-2 according to jsr289. Since lazy creation of sessions are enabled
     * only a wrapper session (FacadeSession) is returned. It will later be
     * replaced by the correct one.
     *
     * @param session
     * @param enableFacade -
     *           enables jsr289 12.5.1 Cloning and Linking
     * @return
     */
    private SipSession getLinkedSession(SipSessionBase session,
        boolean enableFacade) {
        SipApplicationSession app = session.getApplicationSession();
        SipSessionBase sb = (SipSessionBase) session;
        String id = sb.getLinkedSipSessionId();

        if (enableFacade && (id == null) && sb.isDerived()) {
            // jsr289 support 12.5.1 Cloning and Linking.
            // lazy creation is enforced, lets return a dummy wrapper...
            // NOTE: TempSession wrappes UAC session, not UAS session
            return new TempSession(sb);
        }

        return (id != null) ? app.getSipSession(id) : null;
    }

    public SipServletRequest getLinkedSipServletRequest(SipServletRequest req) {
        return getLinkedSipServletRequest((SipServletRequestImpl) req);
    }

    private synchronized SipServletRequestImpl getLinkedSipServletRequest(
        SipServletRequestImpl req1) {
        SipServletRequestImpl req2 = req1.getLinkedRequest();

        if (req2 != null) {
            SipSession session1 = req1.getSession();
            SipSession session2 = req2.getSession();

            if (!isAlreadyLinkedToEachOther((SipSessionBase) session1,
                        (SipSessionBase) session2)) {
                // since the sessions are not linked to each other,
                // the request can't be linked either...
                req2 = null;
            }
        }

        return req2;
    }

    public List<SipServletMessage> getPendingMessages(SipSession session,
        UAMode mode) {
        return ((SipSessionBase) session).getPendingMessages(mode);
    }

    private boolean isValidSessions(SipSessionBase session1,
        SipSessionBase session2) {
        return (session1.isValid() && session2.isValid()) &&
        (session1.getApplicationSession().getId()
                 .equals(session2.getApplicationSession().getId()));
    }

    private boolean isNotLinkedToAnyone(SipSessionBase session1,
        SipSessionBase session2) {
        return (session1.getLinkedSipSessionId() == null) &&
        (session2.getLinkedSipSessionId() == null);
    }

    private boolean isAlreadyLinkedToEachOther(SipSessionBase session1,
        SipSessionBase session2) {
        return session1.equals(session2) ||
        (((session1.getLinkedSipSessionId() != null) &&
        (session2.getLinkedSipSessionId() != null)) &&
        ((session1.getLinkedSipSessionId().equals(session2.getId())) &&
        (session2.getLinkedSipSessionId().equals(session1.getId()))));
    }

    public synchronized void linkSipSessions(SipSession session1,
        SipSession session2) {
        if ((session1 == null) || (session2 == null)) {
            throw new NullPointerException();
        }

        // silently ignore if already linked
        if (!isAlreadyLinkedToEachOther((SipSessionBase) session1,
                    (SipSessionBase) session2)) {
            if ((!isValidSessions((SipSessionBase) session1,
                        (SipSessionBase) session2)) &&
                    !isNotLinkedToAnyone((SipSessionBase) session1,
                        (SipSessionBase) session2)) {
                throw new IllegalArgumentException();
            }

            linkSipSessions((SipSessionBase) session1, (SipSessionBase) session2);
        }
    }

    private void linkSipSessions(SipSessionBase session1,
        SipSessionBase session2) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "linkSipSessions(" + session1.getId() + "," + session2.getId() +
                ")");
        }

        session1.setLinkedSipSessionId(session2.getId());
        session2.setLinkedSipSessionId(session1.getId());
    }

    public synchronized void unlinkSipSessions(SipSession session1) {
        SipSession session2 = getLinkedSession((SipSessionBase) session1, false);

        if ((session2 == null) || !session2.isValid()) {
            throw new IllegalArgumentException();
        }

        unlinkSipSessions((SipSessionBase) session1, (SipSessionBase) session2);
    }

    private void unlinkSipSessions(SipSessionBase session1,
        SipSessionBase session2) {
        session1.setLinkedSipSessionId(null);
        session2.setLinkedSipSessionId(null);
    }

    /**
     * Only a dummy wrapper for B2buaHelper support 12.5.1 Cloning and Linking
     * Support saving attributes and will be replaced by a real sip session after
     * send of message.
     *
     * @author ehsroha
     * @since 2007-apr-01
     *
     */
    class TempSession implements SipSessionBase {
        private static final long serialVersionUID = -5845365964095917542L;
        private ConcurrentHashMap<String, Object> _sessionAttributeMap = new ConcurrentHashMap<String, Object>();
        private final SipSessionBase _session;

        private TempSession(SipSessionBase derivedUACSessionToLink) {
            _session = derivedUACSessionToLink;
        }

        /**
         * Method MUST be kept as is.
         * Call this method when finally the derived UAC session should be
         * linked with the derived UAS session (which is not jet created).
         * @return
         */
        SipSessionBase getDerivedUACSessionToLink() {
            return _session;
        }

        public long getCreationTime() {
            return 0;
        }

        public String getId() {
            return "";
        }

        public long getLastAccessedTime() {
            return 0;
        }

        public void invalidate() {
        }

        public void setOutboundInterface(SipURI uri) {
        }

        public boolean isOngoingTransaction() {
            return true;
        }

        public SipSession.State getState() {
            throw new RuntimeException(" Not yet implemented ");
        }

        public SipApplicationSession getApplicationSession() {
            return null;
        }

        public String getCallId() {
            return "";
        }

        public Address getLocalParty() {
            return null;
        }

        public Address getRemoteParty() {
            return null;
        }

        public SipServletRequest createRequest(String method) {
            return null;
        }

        public void setHandler(String name) throws ServletException {
        }

        public Object getAttribute(String name) {
            return _sessionAttributeMap.get(name);
        }

        public void access() { 
        }

        public Enumeration<String> getAttributeNames() {
            // Since Map doesn't have Enumeration as interface need to convert via
            // private class EnumerationConverter.
            synchronized (_sessionAttributeMap) {
                return new EnumerationConverter<String>(_sessionAttributeMap.keySet());
            }
        }

        public void setAttribute(String name, Object attribute) {
            if (name.equals(SipFactoryImpl.REMOTE_TARGET)) {
                throw new IllegalStateException("reserved key.");
            }

            _sessionAttributeMap.put(name, attribute);
        }

        public void removeAttribute(String name) {
            _sessionAttributeMap.remove(name);
        }

        public String getRegion() {
            return "";
        }

        public URI getSubscriberURI() throws IllegalStateException {
            return null;
        }

        public boolean isValid() {
            return true;
        }

        public SipApplicationSessionImpl getApplicationSessionImpl() {
            return null;
        }

        public String getFragmentId() {
            return null;
        }

        public String getFromTag() {
            return "";
        }

        public String getHandler() {
            return "";
        }

        public String getLinkedSipSessionId() {
            return "";
        }

        public SipSessionBase getOriginalOrDerivedSessionAndRegisterDialog(
            SipServletRequestImpl req, DialogFragment d) {
            return null;
        }

        public SipSessionBase getOriginalOrDerivedSessionAndRegisterDialog(
            SipServletResponseImpl resp, DialogFragment d) {
            return null;
        }

        public URI getRemoteTarget() {
            return null;
        }

        public Address getTo() {
            return null;
        }

        public String getToTag() {
            return "";
        }

        public boolean hasNoToTag() {
            return false;
        }

        public void invalidate(boolean hasTimedOut) {
        }

        public boolean is1xxReliableOngoing() {
            return false;
        }

        public boolean is1xxReliableSDP() {
            return false;
        }

        public boolean isDerived() {
            return true;
        }

        public void reset1xxReliable() {
        }

        public void resetUpdateOngoing() {
        }

        public boolean set1xxReliableOngoing(boolean sdp) {
            return false;
        }

        public void setLinkedSipSessionId(String id) {
        }

        public void setRegion(String region) {
        }

        public void setRemoteTarget(URI contact) {
        }

        public void setSubscriberURI(URI subscriberURI) {
        }

        public void setType(Type type) {
        }

        public boolean setUpdateOngoing() {
            return false;
        }

        public boolean isReplicable() {
            return false;
        }

        public List<SipServletMessage> getPendingMessages(UAMode mode) {
            return null;
        }

        public void setShouldBePersisted() {
        }
        
        public void updateSipSessionState(SipServletRequest req, Type type) {            
        }

        public void updateSipSessionState(SipServletResponse resp, Type type) {
        }

    }
}
