/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.sip.PathNode.Type;

import java.io.Serializable;

import java.util.logging.Level;

import javax.servlet.Servlet;
import javax.servlet.sip.Proxy;


/**
 * Implementation of a ProxyInterface for a confirmed dialog. This class has only
 * a subset of the state that ProxyImpl and most important: it is Serializable.
 *
 * It merely implements the dispatch()-methods which does not need to be aware of
 * earlier responses.
 *
 * @author qmaghes
 *
 */
public class ConfirmedProxyImpl extends AbstractProxyImpl
    implements Serializable {
    private static final long serialVersionUID = -7258322652405154081L;

    public ConfirmedProxyImpl() {
        super();
    }

    /**
     * For deserialization only
     *
     */
    public ConfirmedProxyImpl(String applicationSessionName) {
        super(applicationSessionName);
    }

    public void dispatch(SipServletRequestImpl req, ProxyContext pc) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, req.toDebugString());
        }

        // update max forwards
        int maxForwards = req.getMaxForwards();

        if (maxForwards == 0) {
            if (_log.isLoggable(Level.INFO)) {
                _log.log(Level.INFO, "Too Many Hops for request = " + req);
            }

            SipServletResponseImpl response = req.createTerminatingResponse(483);

            if (response != null) {
                response.popDispatcher().dispatch(response);
            }

            return;
        } else if (maxForwards == -1) {
            maxForwards = 70;
        } else {
            maxForwards--;
        }

        req.setMaxForwards(maxForwards);

        if (req.getMethod().equals("CANCEL")) {
            // FIXME investigate handing of CANCEL on a confirmed dialog.
            // For now we keep the code moved from ProxyImpl.

            // lets answer cancel with 200 OK...
            SipServletResponseImpl resp = req.createTerminatingResponse(200);
            resp.setRemote(req.getRemote());
            resp.popDispatcher().dispatch(resp);

            try {
                // notify the servlet about the CANCEL...
                setDerivedOrOriginalSession(req, pc);

                Servlet s = getServlet(pc.getSipSession().getHandler());

                if (s != null) {
                    req.setProxyContext(pc);
                    req.getSessionImpl().updateSipSessionState(req, Type.Proxy);
                    s.service(req, null);
                } else {
                    if (_log.isLoggable(Level.INFO)) {
                        _log.log(Level.INFO,
                            "Could not find servlet name: " +
                            req.getSessionImpl().getHandler() +
                            " in application: " +
                            req.getSessionImpl().getApplicationSessionImpl()
                               .getName());
                    }
                }
            } catch (Exception e) {
                // problem in servlet, FIXME what to do?
            }
        } else { // not CANCEL

            if (getRecordRoute()) {
                try {
                    // support for dialog creational NOTIFY
                    setDerivedOrOriginalSession(req, pc);

                    // if record route is enabled invoke the servlet
                    Servlet s = getServlet(pc.getSipSession().getHandler());

                    if (s != null) {
                        req.setProxyContext(pc);
                        s.service(req, null);

                        if (req.getSupervised()) {
                            // only if the response will visit this proxy
                            // the request need to be cloned
                            SipServletRequestImpl clone = (SipServletRequestImpl) req.clone();
                            clone.pushTransactionDispatcher(pc);
                            clone.setTransactionRequest(req);
                            clone.popDispatcher().dispatch(clone);
                        } else {
                            req.popDispatcher().dispatch(req);
                        }
                    } else {
                        if (_log.isLoggable(Level.INFO)) {
                            _log.log(Level.INFO,
                                "Could not find servlet name: " +
                                req.getSessionImpl().getHandler() +
                                " in application: " +
                                req.getSessionImpl().getApplicationSessionImpl()
                                   .getName());
                        }
                    }
                } catch (Exception e) {
                    if (_log.isLoggable(Level.FINE)) {
                        _log.log(Level.FINE, "Problem in servlet ", e);
                    }

                    // problem in servlet, // FIXME
                }
            } else {
                req.popDispatcher().dispatch(req);
            }
        }
    }

    public void dispatch(SipServletResponseImpl resp, ProxyContext pc) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, resp.toDebugString());
        }

        invokeServletAndForward(resp, pc);
    }

    /**
     * Getting a Proxy is not applicable for a confirmed dialog.
     *
     */
    public Proxy getFacade(SipServletRequestImpl request) {
        throw new IllegalStateException(
            "getFacade not applicable for Confirmed Proxy");
    }

    /**
     * Only applicable before dialog is confirmed
     *
     */
    public void setVirtualProxyBranchRequest(SipServletResponseImpl resp) {
        throw new IllegalStateException(
            "setVirtualProxyBranchRequest not applicable for Confirmed Proxy");
    }

    /**
     * Only applicable before dialog is confirmed
     *
     */
    public void startVirtualProxyBranch(SipServletResponseImpl resp) {
        throw new IllegalStateException(
            "startVirtualProxyBranch not applicable for Confirmed Proxy");
    }
}
