/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.Servlet;
import javax.servlet.sip.SipServletMessage;


/**
 * This is a Singlton class handling all methods not handled by other FSM, which
 * includes for example: INFO, MESSAGE, OPTIONS, PUBLISH, REGISTER
 *
 * @author ehsroha
 */
public class GeneralSession extends FSM {
    private static final Logger m_Log = (Logger) Logger.getLogger(
            "SipContainer");
    private static GeneralSession m_Instance = new GeneralSession();

    private GeneralSession() {
    }

    /**
     * Will return a GeneralSession if the request matches this FSM.
     *
     * @param m
     *        the incoming message is the key to decide which FSM to return
     * @return a GeneralSession or null if no match occured
     */
    public static FSM createFSM(SipServletMessage m) {
        if (isStaticResponsible(m)) {
            return m_Instance;
        }

        return null;
    }

    public Object clone() {
        return m_Instance;
    }

    /**
     * This FSM is responsible if the message is non of the methods: ACK, BYE,
     * CANCEL, INVITE, PRACK, UPDATE, SUBSCRIBE, REFER or NOTIFY.
     *
     * @param m
     *        the incoming message is the key to decide whether this FSM will
     *        match or not.
     */
    private static boolean isStaticResponsible(SipServletMessage m) {
        String method = m.getMethod();

        // all methods not handled by other FSM should be handled
        // by this general FSM
        return ((method.equals("ACK") || method.equals("BYE") ||
        method.equals("CANCEL") || method.equals("INVITE") ||
        method.equals("PRACK") || method.equals("UPDATE") ||
        method.equals("SUBSCRIBE") || method.equals("REFER") ||
        method.equals("NOTIFY")) == false);
    }

    public boolean isResponsible(SipServletMessage m) {
        return isStaticResponsible(m);
    }

    /**
     * Returns always false since it is a singleton class.
     */
    public boolean isDeletable() {
        return false;
    }

    public void send(SipServletRequestImpl req, UA ua)
        throws IllegalStateException {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE, req.toDebugString());
        }

        // the request need to be cloned before sending
        SipServletRequestImpl clone = (SipServletRequestImpl) req.clone();
        clone.setTransactionRequest(req);
        // lets run in new thread...
        super.send(clone, ua);
    }

    public void send(SipServletResponseImpl resp, UA ua)
        throws IllegalStateException {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE, resp.toDebugString());
        }

        if (!resp.hasToTag()) {
            resp.createTag(Header.TO);
        }

        // the response need to be cloned before sending
        final SipServletResponseImpl clone = (SipServletResponseImpl) resp.clone();

        // if there is a transaction request, added it to the response...
        SipServletRequestImpl req = resp.getRequestImpl().getTransactionRequest();

        if (req != null) {
            clone.setRequest(req);
            clone.setSession(req.getSessionImpl());
        }

        // lets run in new thread...
        super.send(clone, ua);
    }

    public void dispatch(SipServletRequestImpl req, UA ua) {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE, req.toDebugString());
        }

        try {
            Servlet s = ua.getServlet(req.getSessionImpl().getHandler());

            if (s != null) {
                req.getSessionImpl().updateSipSessionState(req, ua.getType());
                s.service(req, null);
            } else {
                if (m_Log.isLoggable(Level.INFO)) {
                    m_Log.log(Level.INFO,
                        "Could not find servlet name: " +
                        req.getSessionImpl().getHandler() +
                        " in application: " +
                        req.getSessionImpl().getApplicationSessionImpl()
                           .getName());
                }
            }
        } catch (Exception e) {
            if (m_Log.isLoggable(Level.FINE)) {
                m_Log.log(Level.FINE, "Problem in servlet ", e);
            }

            // problem in servlet...
            // TR HH52078
            SipServletResponseImpl resp = req.createTerminatingResponse(500);

            if (resp == null) {
                return;
            }

            resp.popDispatcher().dispatch(resp);
        }
    }

    public void dispatch(SipServletResponseImpl resp, UA ua) {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE, resp.toDebugString());
        }

        try {
            Servlet s = ua.getServlet(resp.getSessionImpl().getHandler());

            if (s != null) {
                resp.getSessionImpl().updateSipSessionState(resp, ua.getType());
                s.service(null, resp);
            } else {
                if (m_Log.isLoggable(Level.INFO)) {
                    m_Log.log(Level.INFO,
                        "Could not find servlet name: " +
                        resp.getSessionImpl().getHandler() +
                        " in application: " +
                        resp.getSessionImpl().getApplicationSessionImpl()
                            .getName());
                }
            }
        } catch (Exception e) {
            // problem in servlet, lets drop response
            if (m_Log.isLoggable(Level.INFO)) {
                m_Log.log(Level.INFO, "Problem in servlet.", e);
            }
        }
    }
}
