/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.IOException;
import java.io.Serializable;


/**
 * The proxy represantation of the dialog structure.
 *
 * @author ehsroha
 * @since 2005-maj-30
 */
public class ProxyContext implements PathNode, Serializable {

	private static final long serialVersionUID = -8226217111004527819L;
	private transient ProxyInterface _proxy;
	private ProxyInterface _confirmedProxy = null;
    private transient SipSessionManager _sipSessionManager;
    private String _sessionId;
    private transient State _state; 
    

    /**
     * States a proxy context can be in. The only valid state transitions are
     * 
     * INITIAL => CONFIRMED => TRANSACTION_FINISHED
     * 
     * If there is a transaction timeout BEFORE it gets confirmed, it is 
     * an error in the stack calling transactionFinished(). Those DialogFragments
     * should instead be removed.
     * 
     * @author qmaghes
     *
     */
    public enum State {
    	INITIAL, 	// Before a 2xx has been received 
    	CONFIRMED,	// When a 2xx has been received, but the transacion is still active.
    	TRANSACTION_FINISHED; // When transaction has timed out
    	
    }

    // Used when deserializing
    public ProxyContext()
    {
    	_state = State.TRANSACTION_FINISHED;
    }
    
    public ProxyContext(ProxyInterface proxy, SipSessionBase session) {
        this(proxy,session.getId(),session.getApplicationSessionImpl().getSipSessionManager());
    }
    
    public ProxyContext(ProxyInterface proxy, String sesssionId,SipSessionManager manager) {
       _proxy = proxy;
       _sessionId = sesssionId;
       _state = State.INITIAL;
       _sipSessionManager = manager;
   }


    public ProxyInterface getProxy() {
        return _proxy;
    }

    public Type getType() {
        return Type.Proxy;
    }

    public Object clone() {
        return new ProxyContext(getProxy(), findSipSession());
    }

    public synchronized SipSessionBase getSipSession() {
        return findSipSession();
    }
    
    private SipSessionBase findSipSession()
    {
         if(_sessionId==null){
            return null;
         }
    	 return _sipSessionManager.findSipSession(_sessionId);
    }

    public synchronized void setSipSession(SipSessionBase session) {
        _sessionId = session.getId();
    }

    public SipApplicationSessionImpl getApplicationSession() {
        return findSipSession().getApplicationSessionImpl();
    }

    public void send(SipServletRequestImpl req)
        throws IOException, IllegalStateException {
        throw new IllegalStateException(
            "Illegal to send from ProxyImpl, use proxyTo instead.");
    }

    public void send(SipServletResponseImpl resp)
        throws IOException, IllegalStateException {
        throw new IllegalStateException(
            "Illegal to send from ProxyImpl, use proxyTo instead.");
    }

    public void dispatch(SipServletRequestImpl req) {
        _proxy.dispatch(req, this);
    }

    public void dispatch(SipServletResponseImpl resp) {
        _proxy.dispatch(resp, this);
    }

    /**
     * Returns true if this PathNode is replicable, false otherwise.
     *
     * @return true if this PathNode is replicable, false otherwise
     */
    public boolean isReplicable() {
        return findSipSession().isReplicable();
    }
    
    /**
     * To be called when 2xx is receive for this DialogFragment
     *
     */
    public void setConfirmed(){
       if(_state==State.TRANSACTION_FINISHED){
          throw new IllegalStateException("Not applicable to confirm a dialog in TRANSACTION_FINISHED state");
       }
    	this._confirmedProxy = _proxy.asConfirmed();
    	this._state = State.CONFIRMED;
    }

    /**
     * To be called when transaction times out
     *
     */
    public void transactionFinished()
    {
    	if(_state==State.INITIAL){
    		throw new IllegalStateException("Not applicable to put a non-confirmed dialog in TRANSACTION_FINISHED state");
    	}
    	this._proxy = _confirmedProxy;
    	this._state = State.TRANSACTION_FINISHED;
    }
    
    /**
     * Serialization
     */
    private void writeObject(java.io.ObjectOutputStream out) throws IOException {
    	switch(_state){
    	case TRANSACTION_FINISHED:
    	case CONFIRMED:
    		break;
    		default:
    			throw  new IllegalStateException("Not applicable to save a ProxyContext that is not confirmed");
    	}
    	out.writeObject(_confirmedProxy);
    	out.writeUTF(_sipSessionManager.getApplicationId());
    	out.writeUTF(_sessionId);
	}

    /**
     * Deserialization.
     * 
     * @param in
     * @throws IOException
     * @throws ClassNotFoundException
     */
	private void readObject(java.io.ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		_confirmedProxy = (ProxyInterface)in.readObject();
		_proxy = _confirmedProxy;
		_state = State.TRANSACTION_FINISHED;
		_sipSessionManager = SipSessionManagerBase.get(in.readUTF());
		_sessionId = in.readUTF();
	}
}
