/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.util.BitSet;


/**
 * This class is similar to the java.net.URLEncoder class.
 * Unfortunately, with URLEncoder there is no way to specify
 * safe characters. For SIP there is character which need not to
 * be encoded ("&", "=", "+", "$", ",", ";", "?", "/"). The encoding of
 * the character will be according to RFC 3261.
 *
 * @author eraayaz
 * @since 2005-nov-01
 *
 */
public class SipURIEncoder {
    protected static final char[] hexadecimal = {
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D',
            'E', 'F'
        };
    protected static final char[] unreserved_mark = {
            '-', '_', '.', '!', '~', '*', '', '(', ')'
        };
    protected static final char[] user_unreserved = {
            '&', '=', '+', '$', ',', ';', '?', '/'
        };
    protected static final char[] password = { '&', '=', '+', '$', ',' };
    protected static final char[] token = {
            '-', '.', '!', '%', '*', '_', '+', '', '', '~'
        };
    protected static final char[] param_unreserved = {
            '[', ']', '/', ':', '&', '+', '$'
        };
    protected static final char[] hnv_unreserved = {
            '[', ']', '/', '?', ':', '+', '$'
        };

    // Array containing the safe characters set.
    protected static BitSet userPartSafeCharacters = new BitSet(256);
    protected static BitSet passwordPartSafeCharacters = new BitSet(256);
    protected static BitSet parametersSafeCharacters = new BitSet(256);
    protected static BitSet headersSafeCharacters = new BitSet(256); // for both header name and value, it is the same.

    static {
        for (char i = 'a'; i <= 'z'; i++) {
            userPartSafeCharacters.set(i);
            passwordPartSafeCharacters.set(i);
            parametersSafeCharacters.set(i);
            headersSafeCharacters.set(i);
        }

        for (char i = 'A'; i <= 'Z'; i++) {
            userPartSafeCharacters.set(i);
            passwordPartSafeCharacters.set(i);
            parametersSafeCharacters.set(i);
            headersSafeCharacters.set(i);
        }

        for (char i = '0'; i <= '9'; i++) {
            userPartSafeCharacters.set(i);
            passwordPartSafeCharacters.set(i);
            parametersSafeCharacters.set(i);
            headersSafeCharacters.set(i);
        }

        for (int i = 0; i < unreserved_mark.length; i++) {
            userPartSafeCharacters.set(unreserved_mark[i]);
            passwordPartSafeCharacters.set(i);
            parametersSafeCharacters.set(i);
            headersSafeCharacters.set(i);
        }

        for (int i = 0; i < user_unreserved.length; i++) {
            userPartSafeCharacters.set(user_unreserved[i]);
        }

        for (int i = 0; i < password.length; i++) {
            passwordPartSafeCharacters.set(password[i]);
        }

        for (int i = 0; i < token.length; i++) {
            parametersSafeCharacters.set(token[i]);
        }

        for (int i = 0; i < param_unreserved.length; i++) {
            parametersSafeCharacters.set(param_unreserved[i]);
        }

        for (int i = 0; i < hnv_unreserved.length; i++) {
            headersSafeCharacters.set(hnv_unreserved[i]);
        }
    }

    /**
     * Default constructor.
     */
    public SipURIEncoder() {
    }

    public String encodeUserPart(String toEncode) {
        return encode(toEncode, userPartSafeCharacters);
    }

    public String encodePasswordPart(String toEncode) {
        return encode(toEncode, passwordPartSafeCharacters);
    }

    public String encodeParameter(String toEncode) {
        return encode(toEncode, parametersSafeCharacters);
    }

    public String encodeHeader(String toEncode) {
        return encode(toEncode, headersSafeCharacters);
    }

    /**
     * This method will encode the characters according to
     * RFC 3261, chapter 25.1. The default encoding used is UTF-8.
     *
     * @param type A character u->userpart, p->password, h->headers and a->parameters.
     * @param path A string that need encoding.
     * @return An encoded string.
     */
    private String encode(String path, BitSet safeCharacters) {
        if (path == null) {
            return null;
        }

        int maxBytesPerChar = 10;
        StringBuffer rewrittenPath = new StringBuffer(path.length());
        ByteArrayOutputStream buf = new ByteArrayOutputStream(maxBytesPerChar);
        OutputStreamWriter writer = null;

        try {
            writer = new OutputStreamWriter(buf, "UTF-8");
        } catch (Exception e) {
            e.printStackTrace();
            writer = new OutputStreamWriter(buf);
        }

        for (int i = 0; i < path.length(); i++) {
            int c = (int) path.charAt(i);

            if (safeCharacters.get(c)) {
                rewrittenPath.append((char) c);
            } else {
                // convert to external encoding before hex conversion
                try {
                    writer.write((char) c);
                    writer.flush();
                } catch (IOException e) {
                    buf.reset();

                    continue;
                }

                byte[] ba = buf.toByteArray();

                for (int j = 0; j < ba.length; j++) {
                    // Converting each byte in the buffer
                    byte toEncode = ba[j];
                    rewrittenPath.append('%');

                    int low = (int) (toEncode & 0x0f);
                    int high = (int) ((toEncode & 0xf0) >> 4);
                    rewrittenPath.append(hexadecimal[high]);
                    rewrittenPath.append(hexadecimal[low]);
                }

                buf.reset();
            }
        }

        return rewrittenPath.toString();
    }
}
