/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.dns;

import com.ericsson.ssa.config.annotations.Configuration;
import com.ericsson.ssa.container.TransportNotSupportedException;
import com.ericsson.ssa.container.callflow.CallflowResolver;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.dd.SessionCase;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.URIImpl;
import com.ericsson.ssa.sip.ViaImpl;

import java.util.ListIterator;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.sip.Address;
import javax.servlet.sip.ServletParseException;


/**
 * @author ekrigro TODO To change the template for this generated type comment
 *         go to Window - Preferences - Java - Code Style - Code Templates
 *
 * @reviewed epetstr 2007-feb-20
 *
 */
public class ResolverManager implements Layer {
    private static DnsResolver myDnsResolver = DnsResolver.getInstance();
    private static ResolverManager myResolverManager = new ResolverManager();
    private static TargetResolver myTargetResolver = TargetResolver.getInstance();
    private Layer _nextLayer;
    private Logger logger = (Logger) Logger.getLogger("SipContainer");
    private final int UNDEFINED_SIZE = -1; // messageLength for MTU_SIZE handling
    private boolean defaultTCPTransport;

    // Determines if the functionality described in RFC3263 4.3 Details of RFC
    // 2782 Process is disabled.
    private boolean _eas_503_disabled = false;
    private Reporter _reporter;

    private ResolverManager() {
    }

    public void setReporters(String reporters) {
        _reporter = CallflowResolver.getInstance().getReporter(reporters);
    }

    public Reporter getReporter() {
        return _reporter;
    }

    public static ResolverManager getInstance() {
        return myResolverManager;
    }

    /**
     * Configurable parameter.
     *
     * @return true only if the {@link #setDefaultTCPTransport(boolean)} has been
     *         called with value true.
     */
    public synchronized boolean isDefaultTCPTransport() {
        return defaultTCPTransport;
    }

    /**
     * Called via java reflection from
     * <code> SSAContainerListener.initializeServer()</code>
     *
     * @param defaultTCPTransport
     *            true
     */
    @Configuration(key = "DefaultTcpTransport", node = "/SipService/SipProtocol")
    public synchronized void setDefaultTCPTransport(Boolean defaultTCPTransport) {
        this.defaultTCPTransport = defaultTCPTransport;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Layer#next(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void next(SipServletRequestImpl req) {
        // receiving a request...
        // Update via with received+rport
        ViaImpl via = new ViaImpl(req.getHeader(Header.VIA));

        if (via == null) {
            throw new IllegalAccessError("Missing via in request"); // TODO
                                                                    // define
                                                                    // constants
                                                                    // somewhere
        }

        String host = via.getHost();
        String packetSourceAddress = req.getRemote().getIP();
        boolean isViaModified = false;

        if (TargetResolver.isNumericIp(host)) {
            // rfc 3261 18.2.1 compare with packetsource address
            if (host.equals(packetSourceAddress) == false) {
                via.setParameter(ViaImpl.PARAM_RECEIVED, packetSourceAddress);
                isViaModified = true;
            }
        } else // host contains a domain name
         {
            via.setParameter(ViaImpl.PARAM_RECEIVED, packetSourceAddress);
            isViaModified = true;
        }

        String rport = via.getParameter(ViaImpl.PARAM_RPORT);

        if (rport != null) {
            via.setParameter(ViaImpl.PARAM_RPORT,
                new Integer(req.getRemote().getPort()).toString());
            // PARAM_RECEIVED may already be set above but, in that case we set
            // again with same value
            via.setParameter(ViaImpl.PARAM_RECEIVED, packetSourceAddress); // rfc
                                                                           // 3581 MUST always set received

            isViaModified = true;
        }

        if (isViaModified) {
            // update the "real" header with the changes above
            Header v = req.getRawHeader(Header.VIA);
            v.setReadOnly(false);

            ListIterator<String> it = v.getValues();
            it.next();
            it.set(via.toString());
            v.setReadOnly(true);

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "Updating via = " + via.toString());
            }
        }

        checkRouteHeader(req, true);

        req.pushTransactionDispatcher(this);
        req.pushApplicationDispatcher(this);
        LayerHelper.next(req, this, _nextLayer);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Layer#next(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public void next(SipServletResponseImpl resp) {
        boolean shouldForwardResponseToApplication = true;

        // TODO check if should treat more responseCodes similar????
        // if (resp.getStatus() == 503) // TODO check if we should use defined
        // identifiers (SC_SERVICE_UNAVAILABLE)
        if ((resp.getStatus() == 503) && (_eas_503_disabled == false)) {
            int expireTime = TargetResolver.DEFAULT_RETRYTIME; // in secs
                                                               // check if the Retry-After header field is set&valid

            String retryAfterHeaderStr = resp.getHeader(Header.RETRY_AFTER);
            String retryAfter = getRetryAfterValue(retryAfterHeaderStr);

            if (retryAfter != null) {
                try {
                    expireTime = Integer.parseInt(retryAfter);
                } catch (NumberFormatException ignored) {
                }
            }

            try {
                TargetResolver resolver = TargetResolver.getInstance();
                SipServletRequestImpl req = (SipServletRequestImpl) resp.getRequest();
                resolver.setTargetFailed(req.getRemote(), expireTime);

                TargetTuple newDestination = resolver.resolveRequest(req,
                        req.getMessageSize());

                if (newDestination == null) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE,
                            "List of available targets is exhausted");
                    }

                    shouldForwardResponseToApplication = true; // nothing more to
                                                               // do.....
                } else {
                    req.setRemote(newDestination);
                    // send in new transaction....
                    req.popDispatcher().dispatch(req);
                    shouldForwardResponseToApplication = false; // Do not pass this
                                                                // response up
                }
            } catch (Exception e) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "failed during response processing:", e);
                }
            }
        }

        if (shouldForwardResponseToApplication) {
            LayerHelper.next(resp, this, _nextLayer);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Layer#registerNext(com.ericsson.ssa.sip.Layer)
     */
    public void registerNext(Layer layer) {
        _nextLayer = layer;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Dispatcher#dispatch(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void dispatch(SipServletRequestImpl req) {
        // sending a request...

        //
        try {
            if (doResolve(req, req.getMessageSize())) {
                req.popDispatcher().dispatch(req);
            }
        } catch (TransportNotSupportedException noSuppExc) {
            /*
             * The size of a message must be checked after the transaction layer
             * has added the via header Therefore this RuntimeException is thrown
             * by network layer when messageSize > MAX_MTU_SIZE this means we tried
             * tcp instead of udp however remote host does not support tcp....
             */
            if (doResolve(req, UNDEFINED_SIZE)) {
                req.popDispatcher().dispatch(req);
            }
        }
    }

    private boolean doResolve(SipServletRequestImpl req, int msgSize) {
        TargetTuple destination = null;

        try {
            destination = myTargetResolver.resolveRequest(req, msgSize);
        } catch (Throwable t) {
            // log but ignore exception, destination will be null anyway...
            logger.log(Level.INFO, "failed during request processing:", t);
        }

        if (destination == null) {
            handleResolvError(req);

            return false;
        }

        // hh29713, lets remove this loop detection since it causes problem when
        // spiral occurs.
        // jsr 289 will solve this.
        // Old correction removed: HG69331 Detect loop if ip-address points to us
        // and no more Route exist
        /*
         * if
         * (destination.getIP().equals(NetworkManager.getInstance().getPreferredIpAddress()) &&
         * (destination.getPort() == NetworkManager.getInstance().getDefaultPort() ||
         * destination.getPort() ==
         * NetworkManager.getInstance().getSubsequentPort()) &&
         * ((req.getHeader(Header.ROUTE)) == null)) { SipServletResponseImpl
         * loopResp = null; if (req.getMethod().equals("SUBSCRIBE") ||
         * req.getMethod().equals("NOTIFY")) { // RFC 3265: "489 Bad Event"
         * response should be returned to indicate that the // specified
         * event/event class is not understood by any application on this server
         * loopResp = req.createTerminatingResponse(489); } else { // return 404
         * Not Found loopResp = req.createTerminatingResponse(404); } if
         * (logger.isLoggable(Level.FINE)) { logger.log(Level.FINE, "loop detected, send " +
         * loopResp.getStatus() + " response"); }
         * loopResp.popDispatcher().dispatch(loopResp); return false; }
         */
        req.setRemote(destination);

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "resolving req:\n" + destination.toString());
        }

        return true;
    }

    private void handleResolvError(SipServletRequestImpl req) {
        SipServletResponseImpl failureResp = req.createTerminatingResponse(408);

        // TR HH52078
        if (failureResp == null) {
            return;
        }

        // incoming req (with via) --> send 408 to client
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "failed to resolve req, send 408 response");
        }

        failureResp.popDispatcher().dispatch(failureResp);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ericsson.ssa.sip.Dispatcher#dispatch(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public void dispatch(SipServletResponseImpl resp) {
        // sending a response...
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "sending response");
        }

        TargetTuple remote = null;

        try {
            remote = myTargetResolver.resolveResponse(resp);

            if (remote == null) {
                if (logger.isLoggable(Level.INFO)) {
                    logger.log(Level.INFO,
                        "Ignoring response. Cannot resolve target for:" + resp);

                    return;
                }
            }

            resp.setRemote(remote);
        } catch (Exception e) {
            // The only thing to do here is to ignore the response....
            logger.log(Level.INFO, "Exception when resolving via header:", e);

            return;
        }

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "resolving via:\n" + remote.toString());
        }

        resp.popDispatcher().dispatch(resp);
    }

    public void checkRouteHeader(SipServletRequestImpl req, boolean remove) {
        // loose route support: remove first route header which should refer to
        // this sipstack
        // TODO implement support for strict route
        Header r = req.getRawHeader(Header.ROUTE);

        if (r == null) {
            req.setFragmentId(((URIImpl) req.getRequestURI()).getFragmentId());

            // Session Case handling
            req.setSessionCase(SessionCase.EXTERNAL);

            // end Session Case handling

            // Populate Parameters from Request URI if initial request
            if (req.isInitial()) {
                req.setRequestParams((URIImpl) req.getRequestURI());
            }
        } else {
            r.setReadOnly(false);

            ListIterator<Address> it_a = null;

            try {
                it_a = r.getAddressValues();
            } catch (ServletParseException e) {
                // TR HH52078
                SipServletResponseImpl failureResp = req.createTerminatingResponse(400);

                if (failureResp == null) {
                    return;
                }

                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "failed to parse top route , send 400 response");
                }

                failureResp.popDispatcher().dispatch(failureResp);

                return;
            }

            Address a = it_a.next();
            URIImpl uri = (URIImpl) a.getURI();

            // Session Case handling
            // TODO move constants
            final String CALL_PARAM = "call";
            final String CALL_PARAM_ORIG = "orig";
            final String CALL_PARAM_TERMINATING = "term_registered";
            final String CALL_PARAM_TERMINATING_UNREG = "term_unregistered";

            String callParam = uri.getParameter(CALL_PARAM);

            // if(callParam == null) callParam = uri.getParameter(CALL_PARAM);
            if (callParam == null) {
                req.setSessionCase(SessionCase.EXTERNAL);
            } else {
                if (callParam.equals(CALL_PARAM_ORIG)) {
                    req.setSessionCase(SessionCase.ORIGINATING);
                } else if (callParam.equals(CALL_PARAM_TERMINATING)) {
                    req.setSessionCase(SessionCase.TERMINATING);
                } else if (callParam.equals(CALL_PARAM_TERMINATING_UNREG)) {
                    req.setSessionCase(SessionCase.TERMINATING_UNREGISTERED);
                } else {
                    req.setSessionCase(SessionCase.EXTERNAL);
                    logger.log(Level.WARNING,
                        "Assigned SessionCase external due to unknown value of " +
                        CALL_PARAM + " :" + callParam);
                }
            }

            // end Session Case handling
            if (uri.getLrParam()) {
                // Before removing the Route header extract the fragment-id.
                // This code needs BASIC TESTING in an extended
                // spiraling scenario, such as
                //
                // UAC/EAS --> proxy/external --> proxy/EAS --> UAS/external
                //
                // logger.log(Level.FINE, "+++ Route header with ;lr seen in resolver");
                // logger.log(Level.FINE, "+++ try setting frag info from the Route header
                // URI");
                // logger.log(Level.FINE, "+++ URI: "+(SipURIImpl)a.getURI());
                // logger.log(Level.FINE, "+++ and the id is:
                // "+((SipURIImpl)a.getURI()).getFragmentId());
                // logger.log(Level.FINE, "+++ let's set it in the request then");
                req.setFragmentId(((URIImpl) a.getURI()).getFragmentId());

                if (remove) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE, "Removing route = " + a);
                    }

                    it_a.remove();

                    // support of popped route added.
                    req.setPoppedRoute(a);
                    req.setRecordRouteURI(uri);
                }
            } else {
                throw new IllegalAccessError(
                    "we do not support strict routes yet");
            }

            r.setReadOnly(true);
        }
    }

    private String getRetryAfterValue(String retry_after) {
        if (retry_after == null) {
            return null;
        }

        int stopIndex = 0;
        int length = retry_after.length();

        // Fixed defect HG89704 by checking length before doing charAt.
        while ((stopIndex < length) &&
                Character.isDigit(retry_after.charAt(stopIndex)))
            stopIndex++;

        return retry_after.substring(0, stopIndex);
    }

    @Configuration(key = "enum.top.domain")
    public void setEnumTopDomain(String topDomain) {
        myDnsResolver.setEnumTopDomain(topDomain);
    }

    public String getEnumTopDomain() {
        return myDnsResolver.getEnumTopDomain();
    }

    /**
     * Returns whether the functionality described in RFC3263 4.3 Details of RFC
     * 2782 Process is disabled.
     *
     * @return
     */
    public Boolean isEas503Disabled() {
        return _eas_503_disabled;
    }

    /**
     * Sets whether the functionality described in RFC3263 4.3 Details of RFC
     * 2782 Process should be disabled.
     *
     * @param responseEnabled
     */
    @Configuration(key = "Eas503Disabled", node = "/SipService/SipProtocol")
    public void setEas503Disabled(Boolean disabled) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "Eas503Disabled property is set to " + disabled);
        }

        _eas_503_disabled = disabled;
    }

    public synchronized void start() {
        defaultTCPTransport = Boolean.getBoolean(
                "sip.resolver.defaultTCPTransport");

        if (defaultTCPTransport) {
            logger.warning(
                "TCP default instead of UDP as in rfc3261, this option is for jsr116 TCK compatibility only.");
        }
    }
}
