/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2007. All rights reserved.
 */
package com.ericsson.ssa.sip.persistence;

import java.util.Iterator;

import com.ericsson.ssa.container.callflow.CallflowResolver;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.PathNode;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.SipSessionBase;

import org.jvnet.glassfish.comms.util.LogUtil;


/**
 * Replication layer intercepting message flow
 *
 * @author epakdsz
 */
public class ReplicationManager implements Layer {
    private static final ReplicationManager singletonInstance = new ReplicationManager();
    private LogUtil logger = LogUtil.SIP_LOGGER;
    private Layer nextLayer = null;
    private Reporter reporter;

    // Enforce Singleton pattern
    private ReplicationManager() {
        // Empty
    }

    public void setReporters(String reporters) {
        reporter = CallflowResolver.getInstance().getReporter(reporters);
    }

    public Reporter getReporter() {
        return reporter;
    }

    public static ReplicationManager getInstance() {
        return singletonInstance;
    }

    /**
     * Delegate to next layer.
     * Incoming UAS request.
     */
    public void next(SipServletRequestImpl req) {
        req.pushTransactionDispatcher(this);
        req.pushApplicationDispatcher(this);

        // Send onwards
        LayerHelper.next(req, this, nextLayer);

        replicateOnRequest(req);
    }

    /**
     * Delegate to next layer.
     * Incoming responses for UAC.
     */
    public void next(SipServletResponseImpl resp) {
        // Send onwards
        LayerHelper.next(resp, this, nextLayer);
        replicateOnResponse(resp);
    }

    private void replicateOnResponse(SipServletResponseImpl resp) {
        DialogFragment df = resp.getDialog();

        if (df != null) {
            if (SipFactoryImpl.isDialogCreational(resp.getRequest().getMethod())) {
                if (df.isConfirmed()) {
                    PersistenceUtil.getInstance().saveDialogueFragmentDeeply(df);
                }
            } else {
                for (Iterator<PathNode> pathIter = df.getCaller2CalleePath(); pathIter.hasNext();) {
                    PathNode pn = pathIter.next();
                    SipSessionBase ss = pn.getSipSession();
                    if (ss != null) {
                        SipApplicationSessionImpl sas = ss.getApplicationSessionImpl();
                        if (sas != null) {
                            PersistenceUtil.getInstance().saveSipApplicationSessionDeeply(sas);
                        }
                    }
                }
            }
        }
    }

    public void registerNext(Layer layer) {
        nextLayer = layer;
    }

    /**
     * Delegate to next layer, UAC
     */
    public void dispatch(SipServletRequestImpl req) {
        replicateOnRequest(req);

        // Send onwards
        req.popDispatcher().dispatch(req);
    }

    private void replicateOnRequest(SipServletRequestImpl req) {
        DialogFragment df = req.getDialog();

        if (df != null) {
            if ("ACK".equalsIgnoreCase(req.getMethod()) && df.isConfirmed()) {
                PersistenceUtil.getInstance().saveDialogueFragmentDeeply(df);
            }
        }
    }

    /**
     * Delegate to next layer.
     * Replicate on UAS outgoing responses.
     */
    public void dispatch(SipServletResponseImpl resp) {
        replicateOnResponse(resp);

        // Send onwards
        resp.popDispatcher().dispatch(resp);
    }
}
