/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.timer;

import java.util.concurrent.ScheduledFuture;


/**
 * Implementation of GeneralTimer, i.e. implements trivial backing
 * of abstract getters and setters in GeneralTimerBase.
 *
 * @author qmaghes
 * @since 2007 nov 21
 *
 */
public class GeneralTimerImpl extends GeneralTimerBase {
    /**
     * A future delayed scheduled action to be run
     */
    protected ScheduledFuture<?> future;

    /**
     * Information about what this timer is.
     */
    private Object info;

    /**
     * Absolute time in milliseconds for next execution.
     */
    protected long scheduledExecutionTime = 0;

    /**
     * Delay from creation of timer to execution
     */
    protected long delay = 0;

    /**
     * Period between executions. Only applicable for repeating timers.
     */
    protected long period = 0;

    /**
     * Number of times execution has happened.
     */
    protected long numInvocations = 0;

    /**
     * Absolute time for first execution.
     */
    protected long firstExecution = 0;

    /**
     * Whether executions should be scheduled with fixed delay.
     *
     * @see java.util.Timer for semantics.
     */
    protected boolean fixedDelay = false;

    /**
     * Registered listener that will get a timeout event when executed.
     */
    private GeneralTimerListener listener;

    /**
     * Whether execution should be repeated.
     */
    protected boolean isRepeatingTimer = true;

    /**
     * Constructor for non-repeating timer.
     *
     * @param info Information about the timer
     * @param delay Delay until execution
     * @param listener Listener that will get timeout events.
     */
    public GeneralTimerImpl(Object info, long delay,
        GeneralTimerListener listener) {
        this(info, delay, false, 0, listener);
        isRepeatingTimer = false;
    }

    /**
     * Constructor for repeating times
     *
     * @param info Information about the timer
     * @param delay Delay until first execution
     * @param fixedDelay Whether fixed delay mode should be used
     * @param period Period between execution
     * @param listener Listener that will get timeout events.
     */
    public GeneralTimerImpl(Object info, long delay, boolean fixedDelay,
        long period, GeneralTimerListener listener) {
        this.info = info;
        this.delay = delay;
        scheduledExecutionTime = delay + System.currentTimeMillis();
        this.fixedDelay = fixedDelay;
        this.period = period;
        this.listener = listener;
    }

    protected ScheduledFuture<?> getPFieldFuture() {
        return future;
    }

    protected void setPFieldFuture(ScheduledFuture<?> future) {
        this.future = future;
    }

    protected boolean getPFieldIsRepeatingTimer() {
        return isRepeatingTimer;
    }

    protected long getPFieldFirstExecution() {
        return firstExecution;
    }

    protected void setPFieldFirstExecution(long firstExecutionTime) {
        this.firstExecution = firstExecutionTime;
    }

    protected boolean getPFieldFixedDelay() {
        return fixedDelay;
    }

    protected long getPFieldPeriod() {
        return period;
    }

    protected long getPFieldScheduledExecutionTime() {
        return scheduledExecutionTime;
    }

    protected void setPFieldScheduledExecutionTime(long scheduledExecutionTime) {
        this.scheduledExecutionTime = scheduledExecutionTime;
    }

    protected long incrementAndGetPFieldNumInvocations() {
        return ++numInvocations;
    }

    protected long getPFieldDelay() {
        return delay;
    }

    @Override
    protected void timeout() {
        listener.timeout(this);
    }

    public Object getInfo() {
        return info;
    }

    public GeneralTimerListener getListener() {
        return listener;
    }

    public void setListener(GeneralTimerListener listener) {
        this.listener = listener;
    }
}
