/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.container.SipContainerThreadPool;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipSessionManager;
import com.ericsson.ssa.sip.wrapper.WrapperFactorySet;

import java.io.Serializable;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import javax.servlet.sip.ServletTimer;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.TimerListener;
import javax.servlet.sip.TimerService;


/**
 * @reviewed ejoelbi 2007-jan-17
 */
public class TimerServiceImpl implements TimerService, GeneralTimerListener {
    private static TimerServiceImpl _instance = new TimerServiceImpl();
    private static Logger logger = Logger.getLogger("SipContainer");
    private SipContainerThreadPool _schedPool = null;

    private TimerServiceImpl() {
    }

    public static TimerServiceImpl getInstance() {
        return _instance;
    }

    /*
     * (non-Javadoc)
     * @see javax.servlet.sip.TimerService#createTimer(javax.servlet.sip.SipApplicationSession, long, boolean, java.io.Serializable)
     */
    public ServletTimer createTimer(SipApplicationSession appSession,
        long delay, boolean isPersistent, Serializable info) {
        SipApplicationSession sasIncoming = appSession;
        appSession = WrapperFactorySet.unwrap(appSession);

        SipApplicationSessionImpl as = (SipApplicationSessionImpl) appSession;
        SipSessionManager ssm = as.getSipSessionManager();

        if ((null == ssm) || (as.isValid() == false)) {
            throw new IllegalStateException("Invalid SipApplicationSession");
        }

        TimerListener tl = as.getTimerListener();

        if (tl == null) {
            throw new IllegalStateException("No TimerListener is configured ");
        }

        ServletTimerImpl st = createTimer(ssm, as, tl, delay, isPersistent, info);

        return WrapperFactorySet.wrap(st, sasIncoming);
    }

    /*
     * (non-Javadoc)
     * @see javax.servlet.sip.TimerService#createTimer(javax.servlet.sip.SipApplicationSession, long, long, boolean, boolean, java.io.Serializable)
     */
    public ServletTimer createTimer(SipApplicationSession appSession,
        long delay, long period, boolean fixedDelay, boolean isPersistent,
        Serializable info) {
        if (period < 1) {
            throw new IllegalArgumentException(
                "Period should be greater than 0");
        }

        SipApplicationSession sasIncoming = appSession;
        appSession = WrapperFactorySet.unwrap(appSession);

        SipApplicationSessionImpl as = (SipApplicationSessionImpl) appSession;
        SipSessionManager ssm = as.getSipSessionManager();

        if ((null == ssm) || (as.isValid() == false)) {
            throw new IllegalStateException("Invalid SipApplicationSession");
        }

        TimerListener tl = as.getTimerListener();

        if (tl == null) {
            throw new IllegalStateException("No TimerListener is configured ");
        }

        ServletTimerImpl st = createTimer(ssm, as, tl, delay, period,
                fixedDelay, isPersistent, info);

        return WrapperFactorySet.wrap(st, sasIncoming);
    }

    /**
     * Non SSA interface for creation of GeneralTimer to be used internally in the container.
     *
     * @param listener
     * @param delay
     * @param isPersistent
     * @param info
     * @return
     */
    public GeneralTimer createTimer(GeneralTimerListener listener, long delay,
        boolean isPersistent, Object info) {
        setupScheduller(); // TODO only for test

        GeneralTimerImpl st = new GeneralTimerImpl(info, delay, listener);

        ScheduledFuture<?> f = _schedPool.schedule(st, delay,
                TimeUnit.MILLISECONDS);
        st.setFuture(f);

        return st;
    }

    public GeneralTimer createTimer(GeneralTimerListener listener, long delay,
        long period, boolean fixedDelay, boolean isPersistent, Object info) {
        setupScheduller(); // TODO only for test

        GeneralTimerImpl st = null;
        st = new GeneralTimerImpl(info, delay, fixedDelay, period, listener);

        ScheduledFuture<?> f = null;

        if (fixedDelay) {
            f = _schedPool.scheduleWithFixedDelay(st, delay, period,
                    TimeUnit.MILLISECONDS);
        } else {
            f = _schedPool.scheduleAtFixedRate(st, delay, period,
                    TimeUnit.MILLISECONDS);
        }

        st.setFuture(f);

        return st;
    }

    /**
     * Non SSA interface for creation of GeneralTimer to be used internally in the container.
     * @param ssm
     * @param sas
     * @param listener
     * @param delay
     * @param isPersistent
     * @param info
     * @return
     */
    private ServletTimerImpl createTimer(SipSessionManager ssm,
        SipApplicationSessionImpl sas, TimerListener listener, long delay,
        boolean isPersistent, Serializable info) {
        setupScheduller(); // TODO only for test

        ServletTimerImpl st = null;

        if ((ssm != null) && (sas != null)) {
            st = ssm.createServletTimer(sas, info, delay, listener);
        } else {
            throw new IllegalStateException("SipApplicationSession is null");
        }

        ScheduledFuture<?> f = _schedPool.schedule(st, delay,
                TimeUnit.MILLISECONDS);
        st.setFuture(f);

        st.setPersistent(isPersistent);

        return st;
    }

    private ServletTimerImpl createTimer(SipSessionManager ssm,
        SipApplicationSessionImpl sas, TimerListener listener, long delay,
        long period, boolean fixedDelay, boolean isPersistent, Serializable info) {
        setupScheduller(); // TODO only for test

        ServletTimerImpl st = null;

        if ((ssm != null) && (sas != null)) {
            st = ssm.createServletTimer(sas, info, delay, fixedDelay, period,
                    listener);
        } else {
            throw new IllegalStateException("SipApplicationSession is null");
        }

        ScheduledFuture<?> f = null;

        if (fixedDelay) {
            f = _schedPool.scheduleWithFixedDelay(st, delay, period,
                    TimeUnit.MILLISECONDS);
        } else {
            f = _schedPool.scheduleAtFixedRate(st, delay, period,
                    TimeUnit.MILLISECONDS);
        }

        st.setFuture(f);

        st.setPersistent(isPersistent);

        return st;
    }

    public void rescheduleTimer(ServletTimerImpl timer) {
        final ServletTimerImpl st = timer;
        setupScheduller();

        long now = System.currentTimeMillis();
        RescheduleAlgorithm alg = new RescheduleAlgorithm(timer, now);

        // Fetch what the RescheduleAlgorithm believe it should do.
        // NOTE!!! Important to remeber those values before doint anything
        // with the timer as the rescedul algorithm might answer something
        // else if servlet fires.
        boolean fireAtOnceSeparately = alg.fireAtOnceSeparately();
        boolean needRescheduling = alg.needReschedule(); // Next time to do a periodic reschedule
        long initDelay = alg.delayUntilNextExecution();

        if (fireAtOnceSeparately) {
            _schedPool.execute(st);
        }

        ScheduledFuture<?> f = null;

        if (needRescheduling) {
            if (!st.isRepeatingTimer()) {
                f = _schedPool.schedule(st, initDelay, TimeUnit.MILLISECONDS);
            } else if (st.isFixedDelay()) {
                f = _schedPool.scheduleWithFixedDelay(st, initDelay,
                        st.getPeriod(), TimeUnit.MILLISECONDS);
            } else {
                f = _schedPool.scheduleAtFixedRate(st, initDelay,
                        st.getPeriod(), TimeUnit.MILLISECONDS);
            }
        }

        st.setFuture(f);
    }

    private void setupScheduller() {
        // Could be configured in dispatcher.xml so that TM is unaware of NM
        if (_schedPool == null) {
            _schedPool = SipContainerThreadPool.getInstance();

            if (_schedPool == null) {
                throw new IllegalStateException(
                    "Failed to get access to the ThreadPool");
            }

            createTimer((GeneralTimerListener) this, 10000, 10000, false,
                false, null);
        }
    }

    /*
     * // Only for unit test public void
     * setScheduledExecutorService(ScheduledExecutorService schedPool) {
     * _schedPool = schedPool; }
     */
    public void timeout(GeneralTimer timer) {
        // logger.log(Level.FINE, "Purging Thread pool");
        _schedPool.purge();
    }
}
