/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip.transaction;

import com.ericsson.ssa.container.startup.SipMonitoring;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.timer.GeneralTimerListener;
import com.ericsson.ssa.sip.timer.TimerServiceImpl;

import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author ekrigro TODO To change the template for this generated type comment
 *         go to Window - Preferences - Java - Code Style - Code Templates
 */
enum TransactionState {TRYING,
    CALLING,
    PROCEEDING,
    CONFIRMED,
    COMPLETED,
    TERMINATED,
    ESTABLISHED;

    // ESTABLISHED, state entered after CALLING or PROCEEDING upon 2xx arrival
    // see also TimerRemoveTransaction
}
;
enum TransactionTimer {TimerA,
    TimerB,
    TimerC,
    TimerD,
    TimerE,
    TimerF,
    TimerG,
    TimerH,
    TimerI,
    TimerJ,
    TimerK,
    TimerRemoveTransaction;

    // TimerRemoveTransaction is an extra timer to support RFC 13.2.2.4 2xx
    // Responses, the transaction is removed 64*T1 seconds after the first
    // received 2xx.
}
;


public abstract class Transaction implements GeneralTimerListener {
    public static final String MAGIC_COOKIE = "z9hG4bK";
    protected static TimerServiceImpl _timerService;

    // TODO - UDP transport is defined someother place too i guess
    private final static String _udpTransport = "udp";
    protected static final Logger _log = Logger.getLogger("SipContainer");
    protected static TransactionManager _transactionManager = TransactionManager.getInstance();
    static long T1 = _transactionManager.getTimerT1();
    static long T2 = _transactionManager.getTimerT2();
    static long T4 = _transactionManager.getTimerT4();
    protected TransactionState _state;
    protected String _transactionId;
    protected SipServletRequestImpl _request;
    protected boolean _reliableTransport = true;
    private long _transactionStartedTimestamp = 0;
    private long _lastAccessTimestamp = 0;

    public Transaction(String id, TransactionState state,
        SipServletRequestImpl req) {
        if (SipMonitoring.isEnabled(SipMonitoring.TRANSACTION_MANAGER)) {
            _transactionStartedTimestamp = System.currentTimeMillis();

            // We are very careful here not to affect performance:-)
        }

        _transactionId = id;
        _state = state;
        _request = req;

        if (req.getTransport().equalsIgnoreCase(_udpTransport)) {
            _reliableTransport = false;
        }

        _timerService = TimerServiceImpl.getInstance();

        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "Created Transaction = " +
                getClass().getName());
        }
    }

    public static String generateBranch() {
        // Generate an UUID
        StringBuilder sb = new StringBuilder(MAGIC_COOKIE).append(UUID.randomUUID()
                                                                      .toString());

        return sb.toString();
    }

    public String getTransactionId() {
        return _transactionId;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("TransactionID = ").append(_transactionId);

        return sb.toString();
    }

    protected void updateLastAccessTimestamp() {
        _lastAccessTimestamp = System.currentTimeMillis();
    }

    protected synchronized void terminate() {
        if (SipMonitoring.isEnabled(SipMonitoring.TRANSACTION_MANAGER)) {
            if (_lastAccessTimestamp == 0) {
                // i.e has reached terminated with no retransmissions and not
                // toCompleted/toConfirmed
                _lastAccessTimestamp = System.currentTimeMillis();
            }

            long transactionTime = _lastAccessTimestamp -
                _transactionStartedTimestamp;
            _transactionManager.recordTransactionTime(transactionTime);

            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE, "TransactionTime:" + transactionTime);
            }
        }

        _request = null;
    }
}
