/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.admin.gui.extensions.handlers;

import com.sun.appserv.management.config.AccessLogConfig;
import com.sun.appserv.management.config.ClusteredServerConfig;
import com.sun.appserv.management.config.ConnectionPoolConfig;
import com.sun.appserv.management.config.KeepAliveConfig;
import com.sun.appserv.management.config.RequestProcessingConfig;
import com.sun.appserv.management.config.StandaloneServerConfig;
import com.sun.enterprise.tools.admingui.handlers.ConfigurationHandlers;
import com.sun.enterprise.tools.admingui.util.AMXUtil;
import com.sun.enterprise.tools.admingui.util.GuiUtil;
import com.sun.enterprise.tools.admingui.util.JMXUtil;

import com.sun.jsftemplating.annotation.Handler;
import com.sun.jsftemplating.annotation.HandlerInput;
import com.sun.jsftemplating.annotation.HandlerOutput;
import com.sun.jsftemplating.layout.descriptors.handler.HandlerContext;

import org.jvnet.glassfish.comms.admin.gui.extensions.util.SipUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.faces.model.SelectItem;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.ObjectName;

import javax.net.ssl.SSLServerSocketFactory;


/**
 *  These are the handlers for sip-service element and its sub-elements
 *  @author irfan
 */
public class SipServiceHandlers {
    /*  sip-listener attribute names */
    public static final String[] listenerAttributeNames = new String[] {
            "id", "port", "address", "transport", "enabled"
        };

    /*  sip-listener.ssl attribute names */
    public static final String[] sslAttributeNames = new String[] {
            "cert-nickname", "ssl2-enabled", "ssl2-ciphers", "ssl3-enabled",
            "ssl3-tls-ciphers", "tls-enabled", "tls-rollback-enabled",
            "client-auth-enabled"
        };
    public static List sipServiceSkipList = new ArrayList();
    
    static {
        sipServiceSkipList.add("accessLoggingEnabled");
        sipServiceSkipList.add("accessLogWriteInterval");
        sipServiceSkipList.add("accessLogBufferSize");
    }

    /**
     *  <p> This handler saves the sip-service properties.</p>
     *  <p> Input value:  "ConfigName"         -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "newProps"         -- Type: <code>java.util.Map</code></p>
     *  <p> Input value:  "accessLoggingEnabled"         -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "accessLogWriteInterval"         -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "accessLogBufferSize"         -- Type: <code>java.lang.String</code></p>
     */
    @Handler(id = "saveSipService", input =  {
        @HandlerInput(name = "ConfigName", type = String.class, required = true)
        , @HandlerInput(name = "newProps", type = Map.class)
        , @HandlerInput(name = "accessLoggingEnabled", type = String.class)
        , @HandlerInput(name = "accessLogWriteInterval", type = String.class)
        , @HandlerInput(name = "accessLogBufferSize", type = String.class)
    }
    )
    public static void saveSipService(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);

        HashMap props = (HashMap) handlerContext.getInputValue("newProps");

        SipUtil.updateProperties(sipService, props, sipServiceSkipList);

        // Change the props that are always shown on the page.
        SipUtil.changeProperty(sipService, "accessLoggingEnabled",
            (String) handlerContext.getInputValue("accessLoggingEnabled"));
        SipUtil.changeProperty(sipService, "accessLogWriteInterval",
            (String) handlerContext.getInputValue("accessLogWriteInterval"));
        SipUtil.changeProperty(sipService, "accessLogBufferSize",
            (String) handlerContext.getInputValue("accessLogBufferSize"));
    }

    /**
     *  <p> This handler gets the sip-service properties.</p>
     *  <p> Input value: "ConfigName"    --Type: <code>java.lang.String</code> </p>
     *  <p> Output value: "accessLogBufferSize"    --Type: <code>java.lang.String</code> </p>
     *  <p> Output value: "accessLogWriteInterval"    --Type: <code>java.lang.String</code> </p>
     *  <p> Output value: "accessLoggingEnabled"    --Type: <code>java.lang.Boolean</code> </p>
     *  <p> Output value: "Properties"    --Type: <code>java.util.Map</code> </p>)
     */
    @Handler(id = "getSipService", input =  {
        @HandlerInput(name = "ConfigName", type = String.class, required = true)
    }
    , output =  {
        @HandlerOutput(name = "accessLogBufferSize", type = String.class)
        , @HandlerOutput(name = "accessLogWriteInterval", type = String.class)
        , @HandlerOutput(name = "accessLoggingEnabled", type = Boolean.class)
        , @HandlerOutput(name = "Properties", type = Map.class)
    }
    )
    public static void getSipService(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        Map props = SipUtil.getProperties(sipService);

        try {
            handlerContext.setOutputValue("accessLogBufferSize",
                props.remove("accessLogBufferSize"));
            handlerContext.setOutputValue("accessLogWriteInterval",
                props.remove("accessLogWriteInterval"));

            String alog = (String) props.remove("accessLoggingEnabled");
            Boolean accessLoggingEnabled = true;

            if (GuiUtil.isEmpty(alog)) {
                accessLoggingEnabled = true;
            } else {
                accessLoggingEnabled = (alog.equals("true")) ? true : false;
            }

            handlerContext.setOutputValue("accessLoggingEnabled",
                accessLoggingEnabled);
            handlerContext.setOutputValue("Properties", props);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }

    /**
     *  <p> This handler gets the sip listeners within a sip service.</p>
     *  <p> Input value:  "ConfigName"  -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "selectedRows"  -- Type: <code>java.util.List</code></p>
     *  <p> Output value: "Result"    -- Type: <code>java.util.List</code></p>
     */
    @Handler(id = "getSipListenersList", input =  {
	@HandlerInput(name="ConfigName", type=String.class, required=true),
	@HandlerInput(name="selectedRows", type=List.class),
	@HandlerInput(name="instanceName", type=String.class)
    },
    output =  {
	@HandlerOutput(name="Result", type=List.class),
	@HandlerOutput(name="sipPorts", type=String.class)
    })
    
    public static void getSipListenersList(HandlerContext handlerContext) {
	List result = new ArrayList();
	StringBuffer sipPorts = new StringBuffer();
	try {
	    String instanceName = (String) handlerContext.getInputValue("instanceName");
	    String configName = (String) handlerContext.getInputValue("ConfigName");
	    if (configName == null) {
		configName = AMXUtil.getConfigName(instanceName);
	    }
	    ObjectName sipService = SipUtil.getSipService(configName);
	    String[] listenerNames = (String[]) JMXUtil.invoke(sipService,
		    "getSipListenerNamesList", null, null);
	    for (String id : listenerNames) {
		ObjectName sipListener = SipUtil.getSipListener(sipService, id);
		HashMap row = SipUtil.getAttributeValues(sipListener, listenerAttributeNames);
		if (instanceName != null) {
		    String port = (String) row.get("port");
		    if (port.startsWith("$")) {
			port = SipUtil.resolveToken((port.substring(2, port.length() - 1)),
				configName, instanceName);
		    }
		    sipPorts.append(",").append(port);
		}
		row.put("selected", false);
		result.add(row);
	    }
	} catch (Exception ex) {
	    GuiUtil.handleException(handlerContext, ex);
	}

	handlerContext.setOutputValue("Result", result);
	if (sipPorts.toString().startsWith(",")) {
	    sipPorts.deleteCharAt(0);  //remove the first ','
	}
	handlerContext.setOutputValue("sipPorts", sipPorts.toString());
    }

    /**
     *  <p> The handler gets the various transport protocols available</p>
     *  <p> Input value:  "ConfigName" -- Type: <code>java.lang.String</code></p>
     *  <p> Output value: "Protocols"  -- Type: <code>java.lang.String</code></p>
     */
    @Handler(id = "getSipListenerTransportProtocols", input =  {
        @HandlerInput(name = "ConfigName", type = String.class, required = true)
    }
    , output =  {
        @HandlerOutput(name = "Protocols", type = SelectItem[].class)
    }
    )
    public static void getSipListenerTransportProtocols(
        HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        String[] labels = { "UDP and TCP", "TLS" };
        String[] values = { "udp_tcp", "tls" };
        SelectItem[] options = ConfigurationHandlers.getOptions(values, labels);
        handlerContext.setOutputValue("Protocols", options);
    }

    /**
     *  <p> This handler is used to get the sip listener attributes and properties. </p>
     *  <p> Input value:  "Edit"  -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "FromStep2"  -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "ConfigName"  -- Type: <code>java.lang.String</code></p>
     *  <p> Input value:  "SipName"  -- Type: <code>java.lang.String</code></p>
     *
     *  <p> Output value: "Listener"  -- Type: <code>java.lang.String</code></p>
     *  <p> Output value: "Address"  -- Type: <code>java.lang.String</code></p>
     *  <p> Output value: "Port"  -- Type: <code>java.lang.String</code></p>
     *  <p> Output value: "Transport"  -- Type: <code>java.lang.String</code></p>
     *  <p> Output value: "Properties"  -- Type: <code>java.lang.String</code></p>
     */
    @Handler(id = "getSipListenerValues", input =  {
        @HandlerInput(name = "Edit", type = Boolean.class, required = true)
        , @HandlerInput(name = "FromStep2", type = Boolean.class, required = true)
        , @HandlerInput(name = "ConfigName", type = String.class, required = true)
        , @HandlerInput(name = "SipName", type = String.class, required = true)
    }
    , output =  {
        @HandlerOutput(name = "Listener", type = Boolean.class)
        , @HandlerOutput(name = "Address", type = String.class)
        , @HandlerOutput(name = "Port", type = String.class)
        , @HandlerOutput(name = "Transport", type = String.class)
        , @HandlerOutput(name = "Properties", type = Map.class)
    }
    )
    public static void getSipListenerValues(HandlerContext handlerContext) {
        try {
            Boolean edit = (Boolean) handlerContext.getInputValue("Edit");
            Boolean fromStep2 = (Boolean) handlerContext.getInputValue(
                    "FromStep2");

            if (!edit) {
                if ((fromStep2 == null) || (!fromStep2)) {
                    handlerContext.getFacesContext().getExternalContext()
                                  .getSessionMap().put("sipProps", new HashMap());
                    handlerContext.getFacesContext().getExternalContext()
                                  .getSessionMap().put("sslProps", null);
                    //TODO need a better way to get the default values
                    handlerContext.setOutputValue("Listener", true);
                    handlerContext.setOutputValue("Transport", "udp_tcp");
                } else {
                    Map props = (Map) handlerContext.getFacesContext()
                                                    .getExternalContext()
                                                    .getSessionMap()
                                                    .get("sipProps");
                    handlerContext.setOutputValue("Listener",
                        props.get("enabled"));
                    handlerContext.setOutputValue("Address",
                        props.get("address"));
                    handlerContext.setOutputValue("Port", props.get("port"));
                    handlerContext.setOutputValue("Transport",
                        props.get("transport"));
                    handlerContext.setOutputValue("Properties",
                        props.get("options"));
                }

                return;
            }

            // Edit page
            String listenerName = (String) handlerContext.getInputValue("SipName");
            String configName = (String) handlerContext.getInputValue("ConfigName");
            ObjectName sipListener = SipUtil.getSipListener(SipUtil.getSipService(configName),
                    listenerName);
            HashMap listenerProps = SipUtil.getAttributeValues(sipListener,
                    listenerAttributeNames);

            handlerContext.setOutputValue("Listener",
                listenerProps.get("enabled"));
            handlerContext.setOutputValue("Address",
                listenerProps.get("address"));
            handlerContext.setOutputValue("Port", listenerProps.get("port"));
            handlerContext.setOutputValue("Transport",
                listenerProps.get("transport"));

            Map<String, String> pMap = SipUtil.getProperties(sipListener);

            handlerContext.setOutputValue("Properties", pMap);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }

    /**
     *  <p> This handler saves the sip listener attribute values and properties. The function is called
     *  from both the edit sip listener as well as the new sip listener tabs. In case of new the code
     *  updates the session sipProps map with the values which have been furnished by the user.</p>
     *
     *  <p> Input value: "ConfigName"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "SipName"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Edit"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "Address"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Port"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Listener"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "Transport"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "AddProps"   --Type: <code>java.util.Map</code>
     *  <p> Input value: "RemoveProps"   --Type: <code>java.util.ArrayList</code>
     */
    @Handler(id = "saveSipListenerValues", input =  {
        @HandlerInput(name = "ConfigName", type = String.class, required = true)
        , @HandlerInput(name = "SipName", type = String.class, required = true)
        , @HandlerInput(name = "Edit", type = Boolean.class, required = true)
        , @HandlerInput(name = "Address", type = String.class, required = true)
        , @HandlerInput(name = "Port", type = String.class, required = true)
        , @HandlerInput(name = "Listener", type = Boolean.class)
        , @HandlerInput(name = "Transport", type = String.class)
        , @HandlerInput(name = "AddProps", type = Map.class)
        , @HandlerInput(name = "RemoveProps", type = ArrayList.class)
    }
    )
    public static void saveSipListenerValues(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        String objectName = "com.sun.appserv:type=config,category=config,name=" + configName;
        String listenerName = (String) handlerContext.getInputValue("SipName");
        String address = (String) handlerContext.getInputValue("Address");
        String transport = (String) handlerContext.getInputValue("Transport");
        String port = (String) handlerContext.getInputValue("Port");
        Boolean enabled = (Boolean) handlerContext.getInputValue("Listener");

        Boolean edit = (Boolean) handlerContext.getInputValue("Edit");

        if (!edit) {
            // Creating a new sip listener
            Map sipPropsMap = new HashMap();
            sipPropsMap.put("listenerName", listenerName);
            sipPropsMap.put("address", address);
            sipPropsMap.put("port", port);
            sipPropsMap.put("properties",
                (Map) handlerContext.getInputValue("AddProps"));
            sipPropsMap.put("enabled", enabled);
            sipPropsMap.put("transport", transport);
            handlerContext.getFacesContext().getExternalContext().getSessionMap()
                          .put("sipProps", sipPropsMap);

            //the actual creation is in step 2 of the wizard.
        } else {
            try {
                ObjectName sipListener = SipUtil.getSipListener(SipUtil.getSipService(configName), 
                        listenerName);

                HashMap attributeMap = new HashMap();
                attributeMap.put("port", handlerContext.getInputValue("Port"));
                attributeMap.put("address",
                    (String) handlerContext.getInputValue("Address"));
                attributeMap.put("transport",
                    (String) handlerContext.getInputValue("Transport"));
                attributeMap.put("enabled",
                    (Boolean) handlerContext.getInputValue("Listener"));
                SipUtil.setAttributeValues(sipListener, attributeMap);

                HashMap addProps = (HashMap) handlerContext.getInputValue(
                        "AddProps");
                ArrayList removeProps = (ArrayList) handlerContext.getInputValue(
                        "RemoveProps");
                SipUtil.setProperties(sipListener, addProps, removeProps);
            } catch (Exception ex) {
                GuiUtil.handleException(handlerContext, ex);
            }
        }
    }

    /**
     *  <p> The handler is called from the sip listener SSL edit and new screens. The handler gets the
     *  sip listener SSL properties if any in case of edit. In case of new, the handler sets the default
     *  sip listener SSL properties.</p>
     *  <p> Input value: "ConfigName"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Name"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Type"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Edit"   --Type: <code>java.lang.Boolean</code>
     *
     *  <p> Output value: "ClientAuth"   --Type: <code>java.lang.Boolean</code>
     *  <p> Output value: "CertNickname"   --Type: <code>java.lang.String</code>
     *  <p> Output value: "SSL3Prop"   --Type: <code>java.lang.Boolean</code>
     *  <p> Output value: "SSL2Prop"   --Type: <code>java.lang.Boolean</code>
     *  <p> Output value: "TLSProp"   --Type: <code>java.lang.Boolean</code>
     *  <p> Output value: "CommonCiphersList",    type=SelectItem[]
     *  <p> Output value: "EphemeralCiphersList", type=SelectItem[]
     *  <p> Output value: "OtherCiphersList",     type=SelectItem[]
     *  <p> Output value: "EccCiphersList",       type=SelectItem[]
     *  <p> Output value: "SelectedCommon"   --Type: <code>java.lang.String</code>[]
     *  <p> Output value: "SelectedEph"   --Type: <code>java.lang.String</code>[]
     *  <p> Output value: "SelectedOther"   --Type: <code>java.lang.String</code>[]
     *  <p> Output value: "SelectedEcc"   --Type: <code>java.lang.String</code>[]
     */
    @Handler(id = "getSipSSLProperties", input =  {
        @HandlerInput(name = "ConfigName", type = String.class, required = true)
        , @HandlerInput(name = "Name", type = String.class, required = true)
        , @HandlerInput(name = "Type", type = String.class, required = true)
        , @HandlerInput(name = "Edit", type = Boolean.class, required = true)
    }
    , output =  {
        @HandlerOutput(name = "ClientAuth", type = Boolean.class)
        , @HandlerOutput(name = "CertNickname", type = String.class)
        , @HandlerOutput(name = "SSL3Prop", type = Boolean.class)
        , @HandlerOutput(name = "SSL2Prop", type = Boolean.class)
        , @HandlerOutput(name = "TLSProp", type = Boolean.class)
        , @HandlerOutput(name = "CommonCiphersList", type = SelectItem[].class)
        , @HandlerOutput(name = "EphemeralCiphersList", type = SelectItem[].class)
        , @HandlerOutput(name = "OtherCiphersList", type = SelectItem[].class)
        , @HandlerOutput(name = "EccCiphersList", type = SelectItem[].class)
        , @HandlerOutput(name = "SelectedCommon", type = String[].class)
        , @HandlerOutput(name = "SelectedEph", type = String[].class)
        , @HandlerOutput(name = "SelectedOther", type = String[].class)
        , @HandlerOutput(name = "SelectedEcc", type = String[].class)
    }
    )
    public static void getSipSSLProperties(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        String listenerName = ((String) handlerContext.getInputValue("Name"));
        String selectedCiphers = null;
        String type = (String) handlerContext.getInputValue("Type");

        boolean ssl3Prop = false;
        boolean ssl2Prop = false;
        boolean tlsProp = false;
        boolean clientAuth = false;
        boolean isEdit = ((Boolean) handlerContext.getInputValue("Edit")).booleanValue();

        Map sslProps = (Map) handlerContext.getFacesContext()
                                           .getExternalContext().getSessionMap()
                                           .get("sslProps");

        if (isEdit) {
            try {
                ObjectName listener = SipUtil.getSipListener(sipService, listenerName);
                ObjectName ssl = (ObjectName) SipUtil.getChildByFunction(listener, "getSsl");

                if (ssl != null) {
                    HashMap sslAttrMap = SipUtil.getAttributeValues(ssl,
                            sslAttributeNames);

                    if ((sslAttrMap != null) && (sslAttrMap.size() > 0)) {
                        clientAuth = SipUtil.getBooleanValue(sslAttrMap.get(
                                    "client-auth-enabled"));
                        handlerContext.setOutputValue("CertNickname",
                            sslAttrMap.get("cert-nickname"));
                        ssl3Prop = SipUtil.getBooleanValue(sslAttrMap.get(
                                    "ssl3-enabled"));
                        ssl2Prop = SipUtil.getBooleanValue(sslAttrMap.get(
                                    "ssl2-enabled"));
                        tlsProp = SipUtil.getBooleanValue(sslAttrMap.get(
                                    "tls-enabled"));
                        selectedCiphers = (String) sslAttrMap.get(
                                "ssl3-tls-ciphers");
                    }
                }
            } catch (Exception ex) {
                GuiUtil.handleException(handlerContext, ex);
            }
        } else {
            if (sslProps != null) {
                clientAuth = (Boolean) sslProps.get("clientAuth");
                ssl3Prop = (Boolean) sslProps.get("ssl3Prop");
                ssl2Prop = (Boolean) sslProps.get("ssl2Prop");
                tlsProp = (Boolean) sslProps.get("tlsProp");
                handlerContext.setOutputValue("CertNickname",
                    sslProps.get("certNickname"));
            }
        }

        SSLServerSocketFactory factory = (SSLServerSocketFactory) SSLServerSocketFactory.getDefault();
        String[] supportedCiphers = factory.getDefaultCipherSuites();
        Vector ciphers = SipUtil.getCiphersVector(supportedCiphers);

        SelectItem[] commonCiphers = ConfigurationHandlers.getOptions(SipUtil.getCommonCiphers(
                    ciphers));
        SelectItem[] ephemeralCiphers = ConfigurationHandlers.getOptions(SipUtil.getEphemeralCiphers(
                    ciphers));
        SelectItem[] otherCiphers = ConfigurationHandlers.getOptions(SipUtil.getOtherCiphers(
                    ciphers));
        SelectItem[] eccCiphers = ConfigurationHandlers.getOptions(SipUtil.getEccCiphers(
                    ciphers));

        handlerContext.setOutputValue("ClientAuth", clientAuth);
        handlerContext.setOutputValue("SSL3Prop", ssl3Prop);
        handlerContext.setOutputValue("SSL2Prop", ssl2Prop);
        handlerContext.setOutputValue("TLSProp", tlsProp);
        handlerContext.setOutputValue("CommonCiphersList", commonCiphers); //NOI18N
        handlerContext.setOutputValue("EphemeralCiphersList", ephemeralCiphers); //NOI18N
        handlerContext.setOutputValue("OtherCiphersList", otherCiphers); //NOI18N
        handlerContext.setOutputValue("EccCiphersList", eccCiphers); //NOI18N

        if (sslProps == null) {
            String[] ciphersArr = SipUtil.getSelectedCiphersList(selectedCiphers);
            Vector selValues = SipUtil.getCiphersVector(ciphersArr);
            handlerContext.setOutputValue("SelectedCommon",
                SipUtil.getCommonCiphers(selValues)); //NOI18N
            handlerContext.setOutputValue("SelectedEph",
                SipUtil.getEphemeralCiphers(selValues)); //NOI18N
            handlerContext.setOutputValue("SelectedOther",
                SipUtil.getOtherCiphers(selValues)); //NOI18N
            handlerContext.setOutputValue("SelectedEcc",
                SipUtil.getEccCiphers(selValues)); //NOI18N
        } else {
            handlerContext.setOutputValue("SelectedCommon",
                sslProps.get("selectedCommon")); //NOI18N
            handlerContext.setOutputValue("SelectedEph",
                sslProps.get("selectedEph")); //NOI18N
            handlerContext.setOutputValue("SelectedOther",
                sslProps.get("selectedOther")); //NOI18N
            handlerContext.setOutputValue("SelectedEcc",
                sslProps.get("selectedEcc")); //NOI18N
        }
    }

    /**
     *  <p> The handler saves the sip SSL properties for the sip listener. The function is called for
     *  both the edit as well as new sip listener. In case of new sip listener screen, the handler
     *  extracts the sip listener properties and creates a sip listener first. It then sets the SSL
     *  properties on the newly created listener.</p>
     *  <p> Input value: "ConfigName"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Name"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Type"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Edit"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "ClientAuth"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "CertNickname"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "SSL3Prop"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "SSL2Prop"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "TLSProp"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "CommonCiphersList",     type=SelectItem[]
     *  <p> Input value: "EphemeralCiphersList",  type=SelectItem[]
     *  <p> Input value: "OtherCiphersList",      type=SelectItem[]
     *  <p> Input value: "EccCiphersList",        type=SelectItem[]
     *  <p> Input value: "SelectedCommon"   --Type: <code>java.lang.String</code>[]
     *  <p> Input value: "SelectedEph"   --Type: <code>java.lang.String</code>[]
     *  <p> Input value: "SelectedOther"   --Type: <code>java.lang.String</code>[]
     *  <p> Input value: "SelectedEcc"   --Type: <code>java.lang.String</code>[]
     */
    @Handler(id = "saveSipSSLProperties", input =  {
        @HandlerInput(name = "ConfigName", type = String.class, required = true)
        , @HandlerInput(name = "Name", type = String.class, required = true)
        , @HandlerInput(name = "Type", type = String.class, required = true)
        , @HandlerInput(name = "Edit", type = Boolean.class, required = true)
        , @HandlerInput(name = "ClientAuth", type = Boolean.class)
        , @HandlerInput(name = "CertNickname", type = String.class)
        , @HandlerInput(name = "SSL3Prop", type = Boolean.class)
        , @HandlerInput(name = "SSL2Prop", type = Boolean.class)
        , @HandlerInput(name = "TLSProp", type = Boolean.class)
        , @HandlerInput(name = "CommonCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "EphemeralCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "OtherCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "EccCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "SelectedCommon", type = String[].class)
        , @HandlerInput(name = "SelectedEph", type = String[].class)
        , @HandlerInput(name = "SelectedOther", type = String[].class)
        , @HandlerInput(name = "SelectedEcc", type = String[].class)
    }
    )
    public static void saveSipSSLProperties(HandlerContext handlerContext) {
        try {
            String listenerName = (String) handlerContext.getInputValue("Name");
            String configName = (String) handlerContext.getInputValue("ConfigName");
            ObjectName sipService = SipUtil.getSipService(configName);
            ObjectName ssl = null;
            ObjectName sipListener = null;
            String certNickname = (String) handlerContext.getInputValue("CertNickname");
            String type = (String) handlerContext.getInputValue("Type");

            boolean isEdit = ((Boolean) handlerContext.getInputValue("Edit")).booleanValue();

            if (isEdit) {
                // remove the ssl element from the listener.
                sipListener = SipUtil.getSipListener(sipService, listenerName);
                ssl = (ObjectName) SipUtil.getChildByFunction(sipListener, "getSsl");
                if (ssl != null) {
                    //Remove the SSL element so that it can be recreated below
                    JMXUtil.invoke(sipListener, "removeSsl", null, null);
                    ssl = null;
                }
            } else {
                // New Listener. Need to create the listener and then set the SSL props
                Map props = (Map) handlerContext.getFacesContext()
                                                .getExternalContext()
                                                .getSessionMap().get("sipProps");
                AttributeList list = new AttributeList();
                list.add(new Attribute("id", props.get("listenerName")));
                list.add(new Attribute("port", props.get("port")));
                list.add(new Attribute("address", props.get("address")));
                list.add(new Attribute("transport", props.get("transport")));
                list.add(new Attribute("enabled", props.get("enabled")));

                // create the listener
                sipListener = (ObjectName) JMXUtil.invoke(sipService, "createSipListener", 
                        new Object[] {list}, new String[] {list.getClass().getName()});
                SipUtil.setProperties(sipListener, (HashMap) props.get("properties"), null);
            }

            if ((certNickname != null) && (!certNickname.equals(""))) {
                HashMap sslProps = new HashMap();

                if (ssl == null) {
                    //SSL Element for this listener does not exist create one
                    Attribute attribute = new Attribute("cert-nickname",
                            certNickname);
                    AttributeList attrList = new AttributeList(1);
                    attrList.add(attribute);
                    ssl = (ObjectName) JMXUtil.invoke(sipListener, "createSsl",
                            new Object[] { attrList },
                            new String[] { attrList.getClass().getName() });
                }

                boolean clientAuthEnabled = ((Boolean) handlerContext.getInputValue(
                        "ClientAuth")).booleanValue();
                boolean ssl3Prop = ((Boolean) handlerContext.getInputValue(
                        "SSL3Prop")).booleanValue();
                boolean ssl2Prop = ((Boolean) handlerContext.getInputValue(
                        "SSL2Prop")).booleanValue();
                boolean tlsProp = ((Boolean) handlerContext.getInputValue(
                        "TLSProp")).booleanValue();
                sslProps.put("cert-nickname", certNickname);
                sslProps.put("client-auth-enabled", clientAuthEnabled);
                sslProps.put("ssl2-enabled", ssl2Prop);
                sslProps.put("ssl3-enabled", ssl3Prop);
                sslProps.put("tls-enabled", tlsProp);

                if (ssl3Prop || tlsProp) {
                    SSLServerSocketFactory factory = (SSLServerSocketFactory) SSLServerSocketFactory.getDefault();
                    String[] supportedCiphers = factory.getDefaultCipherSuites();
                    Vector ciphersVector = SipUtil.getCiphersVector(supportedCiphers);
                    String[] selectedCiphers = SipUtil.getSelectedCiphersList((String) JMXUtil.getAttribute(
                                ssl, "ssl3-tls-ciphers"));
                    String[] selectedCommon = (String[]) handlerContext.getInputValue(
                            "SelectedCommon");
                    String[] selectedEph = (String[]) handlerContext.getInputValue(
                            "SelectedEph");
                    String[] selectedOther = (String[]) handlerContext.getInputValue(
                            "SelectedOther");
                    String[] selectedEcc = (String[]) handlerContext.getInputValue(
                            "SelectedEcc");

                    //TODO SSL Ciphers settings has issues. Has errors and does not save
                    // properly when selections are removed
                    String ciphers = SipUtil.processSelectedCiphers(selectedCommon, "");
                    ciphers = SipUtil.processSelectedCiphers(selectedEph, ciphers);
                    ciphers = SipUtil.processSelectedCiphers(selectedOther, ciphers);
                    ciphers = SipUtil.processSelectedCiphers(selectedEcc, ciphers);

                    sslProps.put("ssl3-tls-ciphers", ciphers);
                }

                SipUtil.setAttributeValues(ssl, sslProps);
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }

    /**
     *  <p> The handler is called when a user clicks the previous button whilst creating a new sip
     *  listener. The handler sets the selected SSL properties into the sslProps map.</p>
     *  <p> Input value: "ClientAuth"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "CertNickname"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "SSL3Prop"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "SSL2Prop"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "TLSProp"   --Type: <code>java.lang.Boolean</code>
     *  <p> Input value: "CommonCiphersList",     type=SelectItem[]
     *  <p> Input value: "EphemeralCiphersList",  type=SelectItem[]
     *  <p> Input value: "OtherCiphersList",      type=SelectItem[]
     *  <p> Input value: "EccCiphersList",        type=SelectItem[]
     *  <p> Input value: "SelectedCommon"   --Type: <code>java.lang.String</code>[]
     *  <p> Input value: "SelectedEph"   --Type: <code>java.lang.String</code>[]
     *  <p> Input value: "SelectedOther"   --Type: <code>java.lang.String</code>[]
     *  <p> Input value: "SelectedEcc"   --Type: <code>java.lang.String</code>[]
     */
    @Handler(id = "updateSipSSLProperties", input =  {
        @HandlerInput(name = "ClientAuth", type = Boolean.class)
        , @HandlerInput(name = "CertNickname", type = String.class)
        , @HandlerInput(name = "SSL3Prop", type = Boolean.class)
        , @HandlerInput(name = "SSL2Prop", type = Boolean.class)
        , @HandlerInput(name = "TLSProp", type = Boolean.class)
        , @HandlerInput(name = "CommonCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "EphemeralCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "OtherCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "EccCiphersList", type = SelectItem[].class)
        , @HandlerInput(name = "SelectedCommon", type = String[].class)
        , @HandlerInput(name = "SelectedEph", type = String[].class)
        , @HandlerInput(name = "SelectedOther", type = String[].class)
        , @HandlerInput(name = "SelectedEcc", type = String[].class)
    }
    )
    public static void updateSipSSLProperties(HandlerContext handlerContext) {
        Map sslPropsMap = new HashMap();
        sslPropsMap.put("certNickname",
            (String) handlerContext.getInputValue("CertNickname"));
        sslPropsMap.put("clientAuth",
            ((Boolean) handlerContext.getInputValue("ClientAuth")).booleanValue());
        sslPropsMap.put("ssl3Prop",
            ((Boolean) handlerContext.getInputValue("SSL3Prop")).booleanValue());
        sslPropsMap.put("ssl2Prop",
            ((Boolean) handlerContext.getInputValue("SSL2Prop")).booleanValue());
        sslPropsMap.put("tlsProp",
            ((Boolean) handlerContext.getInputValue("TLSProp")).booleanValue());
        sslPropsMap.put("selectedCommon",
            (String[]) handlerContext.getInputValue("SelectedCommon"));
        sslPropsMap.put("selectedEph",
            (String[]) handlerContext.getInputValue("SelectedEph"));
        sslPropsMap.put("selectedOther",
            (String[]) handlerContext.getInputValue("SelectedOther"));
        sslPropsMap.put("selectedEcc",
            (String[]) handlerContext.getInputValue("SelectedEcc"));
        handlerContext.getFacesContext().getExternalContext().getSessionMap()
                      .put("sslProps", sslPropsMap);
    }

    /**
     *  <p> The handler deletes the selected sip listeners.</p>
     *  <p> Input value: "selectedRows"   --Type: <code>java.util.List</code>
     *  <p> Input value: "ConfigName"   --Type: <code>java.lang.String</code>
     *  <p> Input value: "Type"   --Type: <code>java.lang.String</code>
     */
    @Handler(id = "deleteSipListeners", input =  {
        @HandlerInput(name = "selectedRows", type = List.class, required = true)
        , @HandlerInput(name = "ConfigName", type = String.class, required = true)
        , @HandlerInput(name = "Type", type = String.class, required = true)
    }
    )
    public static void deleteSipListeners(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        
        List obj = (List) handlerContext.getInputValue("selectedRows");
        String type = (String) handlerContext.getInputValue("Type");
        List<Map> selectedRows = (List) obj;

        try {
            for (Map oneRow : selectedRows) {
                String name = (String) oneRow.get("id");
                JMXUtil.invoke(sipService, "removeSipListenerById", 
                    new Object[] {name}, new String[] {"java.lang.String"});
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }

    @Handler(id="getSipAccessLogSettings",
        input = {
            @HandlerInput(name="ConfigName", type=String.class, required=true)
        },
        output = {
            @HandlerOutput(name="Rotation", type=Boolean.class),
            @HandlerOutput(name="Policy", type=String.class),
            @HandlerOutput(name="Suffix", type=String.class),
            @HandlerOutput(name="Interval", type=String.class),
            @HandlerOutput(name="Format", type=String.class)
        }
    )
    public static void getSipAccessLogSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        try {
            ObjectName al = (ObjectName) SipUtil.getChildByFunction(sipService, "getAccessLog");
            String policy="";
            String interval="";
            String suffix="";
            String format="";
            boolean rotation = true;

            if (al != null){
                rotation = Boolean.parseBoolean(JMXUtil.getAttribute(al, "rotation-enabled").toString());
                policy = (String) JMXUtil.getAttribute(al, "rotation-policy");
                interval = (String) JMXUtil.getAttribute(al, "rotation-interval-in-minutes");
                suffix = (String) JMXUtil.getAttribute(al, "rotation-suffix");
                format = (String) JMXUtil.getAttribute(al, "format");
            }else{
                //Default values for sip access log are same as http access log
                Map defaultMap = AMXUtil.getDomainConfig().getDefaultAttributeValues(AccessLogConfig.J2EE_TYPE);
                policy = (String) defaultMap.get("rotation-policy");
                interval = (String)defaultMap.get("rotation-interval-in-minutes");
                suffix = (String)defaultMap.get("rotation-suffix");
                format = (String)defaultMap.get("format");
                String rotationKey = (String) defaultMap.get("rotation-enabled");
                rotation = (rotationKey == null) ? false : Boolean.valueOf(rotationKey);
            }
            handlerContext.setOutputValue("Rotation", rotation);
            handlerContext.setOutputValue("Policy", policy);
            handlerContext.setOutputValue("Interval", interval);
            handlerContext.setOutputValue("Suffix", suffix);
            handlerContext.setOutputValue("Format", format);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }

    @Handler(id="saveSipAccessLogSettings",
        input = {
            @HandlerInput(name="ConfigName", type=String.class, required=true),
            @HandlerInput(name="Rotation", type=Boolean.class),
            @HandlerInput(name="Policy", type=String.class),
            @HandlerInput(name="Interval", type=String.class),
            @HandlerInput(name="Suffix", type=String.class),
            @HandlerInput(name="Format", type=String.class)
        }
    )
    public static void saveSipAccessLogSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        try {
            ObjectName accessLog = (ObjectName) SipUtil.getChildByFunction(sipService, "getAccessLog");
            if (accessLog == null) {
                //create access log element
                accessLog = SipUtil.createChildByFunction(sipService, "createAccessLog");
            }
            HashMap attributes = new HashMap();
            Boolean temp = (Boolean) handlerContext.getInputValue("Rotation");
            attributes.put("rotation-enabled", ((temp == null) ? false : temp));
            attributes.put("rotation-policy", (String)handlerContext.getInputValue("Policy"));
            attributes.put("rotation-interval-in-minutes", (String)handlerContext.getInputValue("Interval"));
            attributes.put("rotation-suffix", (String)handlerContext.getInputValue("Suffix"));
            attributes.put("format", (String)handlerContext.getInputValue("Format"));
            SipUtil.setAttributeValues(accessLog, attributes);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    @Handler(id="getSipRequestProcessingSettings",
        input = {
            @HandlerInput(name="ConfigName", type=String.class, required=true)
        },
        output = {
            @HandlerOutput(name="Count", type=String.class),
            @HandlerOutput(name="Initial", type=String.class),
            @HandlerOutput(name="Increment", type=String.class),
            @HandlerOutput(name="Timeout", type=String.class),
            @HandlerOutput(name="Buffer", type=String.class)
        }
    )
    public static void getSipRequestProcessingSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        try {
            String count = "";
            String initial = "";
            String increment = "";
            String timeout = "";
            String buffer = "";
            
            ObjectName reqProcessing = (ObjectName) SipUtil.getChildByFunction(
                    SipUtil.getSipService(configName), "getRequestProcessing");
            
            if (reqProcessing == null) {
                Map defaultMap = AMXUtil.getDomainConfig().getDefaultAttributeValues(RequestProcessingConfig.J2EE_TYPE);
                count = (String) defaultMap.get(SipUtil.RequestProcessingKeys.THREAD_COUNT_KEY);
                initial = (String) defaultMap.get(SipUtil.RequestProcessingKeys.INITIAL_THREAD_COUNT_KEY);
                increment = (String) defaultMap.get(SipUtil.RequestProcessingKeys.THREAD_INCREMENT_KEY);
                timeout = (String) defaultMap.get(SipUtil.RequestProcessingKeys.REQUEST_TIMEOUT_IN_SECONDS_KEY);
                buffer = (String) defaultMap.get(SipUtil.RequestProcessingKeys.HEADER_BUFFER_LENGTH_IN_BYTES_KEY);
            } else {
                count = (String) JMXUtil.getAttribute(reqProcessing, SipUtil.RequestProcessingKeys.THREAD_COUNT_KEY);
                initial = (String) JMXUtil.getAttribute(reqProcessing, SipUtil.RequestProcessingKeys.INITIAL_THREAD_COUNT_KEY);
                increment = (String) JMXUtil.getAttribute(reqProcessing, SipUtil.RequestProcessingKeys.THREAD_INCREMENT_KEY);
                timeout = (String) JMXUtil.getAttribute(reqProcessing, SipUtil.RequestProcessingKeys.REQUEST_TIMEOUT_IN_SECONDS_KEY);
                buffer = (String) JMXUtil.getAttribute(reqProcessing, SipUtil.RequestProcessingKeys.HEADER_BUFFER_LENGTH_IN_BYTES_KEY);
            }
            
            handlerContext.setOutputValue("Count", count);
            handlerContext.setOutputValue("Initial", initial);
            handlerContext.setOutputValue("Increment", increment);
            handlerContext.setOutputValue("Timeout", timeout);
            handlerContext.setOutputValue("Buffer", buffer);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    @Handler(id="saveSipRequestProcessingSettings",
    input={
        @HandlerInput(name="ConfigName", type=String.class, required=true),
        @HandlerInput(name="Count",  type=String.class),
        @HandlerInput(name="Initial",    type=String.class),
        @HandlerInput(name="Increment",       type=String.class),
        @HandlerInput(name="Timeout", type=String.class),
        @HandlerInput(name="Buffer", type=String.class)
    })

    public static void saveSipRequestProcessingSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        try{
            ObjectName rp = (ObjectName) SipUtil.getChildByFunction(sipService, "getRequestProcessing");
            if (rp == null) {
                rp = SipUtil.createChildByFunction(sipService, "createRequestProcessing");
            }
            HashMap attributes = new HashMap();
            attributes.put(SipUtil.RequestProcessingKeys.THREAD_COUNT_KEY, (String)handlerContext.getInputValue("Count"));
            attributes.put(SipUtil.RequestProcessingKeys.INITIAL_THREAD_COUNT_KEY, (String)handlerContext.getInputValue("Initial"));
            attributes.put(SipUtil.RequestProcessingKeys.THREAD_INCREMENT_KEY, (String)handlerContext.getInputValue("Increment"));
            attributes.put(SipUtil.RequestProcessingKeys.REQUEST_TIMEOUT_IN_SECONDS_KEY, 
                    (String)handlerContext.getInputValue("Timeout"));
            attributes.put(SipUtil.RequestProcessingKeys.HEADER_BUFFER_LENGTH_IN_BYTES_KEY, 
                    (String)handlerContext.getInputValue("Buffer"));
            SipUtil.setAttributeValues(rp, attributes);
        }catch(Exception ex){
            GuiUtil.handleException(handlerContext, ex);
        }
        
    }
    
    
    @Handler(id="getSipKeepAliveSettings",
    input={
        @HandlerInput(name="ConfigName", type=String.class, required=true)},
    output={
        @HandlerOutput(name="Count",  type=String.class),
        @HandlerOutput(name="Connections",    type=String.class),
        @HandlerOutput(name="Timeout",       type=String.class)
    })

    public static void getSipKeepAliveSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        try {
            ObjectName keepAlive = (ObjectName) SipUtil.getChildByFunction(
                    SipUtil.getSipService(configName), "getKeepAlive");
            String count = "";
            String maxConnections = "";
            String timeout = "";

            if (keepAlive == null) {
                Map defaultMap = AMXUtil.getDomainConfig().getDefaultAttributeValues(KeepAliveConfig.J2EE_TYPE);
                count = (String) defaultMap.get(SipUtil.KeepAliveKeys.THREAD_COUNT);
                maxConnections = (String) defaultMap.get(SipUtil.KeepAliveKeys.MAX_CONNECTIONS);
                timeout = (String) defaultMap.get(SipUtil.KeepAliveKeys.TIMEOUT_IN_SECONDS);
            } else {
                count = (String) JMXUtil.getAttribute(keepAlive, SipUtil.KeepAliveKeys.THREAD_COUNT);
                maxConnections = (String) JMXUtil.getAttribute(keepAlive, SipUtil.KeepAliveKeys.MAX_CONNECTIONS);
                timeout = (String) JMXUtil.getAttribute(keepAlive, SipUtil.KeepAliveKeys.TIMEOUT_IN_SECONDS);
            }
            handlerContext.setOutputValue("Count", count);
            handlerContext.setOutputValue("Connections", maxConnections);
            handlerContext.setOutputValue("Timeout", timeout);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    @Handler(id="saveSipKeepAliveSettings",
    input={
        @HandlerInput(name="ConfigName", type=String.class, required=true),
        @HandlerInput(name="Count",  type=String.class),
        @HandlerInput(name="Connections",    type=String.class),
        @HandlerInput(name="Timeout",       type=String.class)
    })

    public static void saveSipKeepAliveSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        try {
            ObjectName sipService = SipUtil.getSipService(configName);
            ObjectName keepAlive = (ObjectName) SipUtil.getChildByFunction(sipService, "getKeepAlive");

            if (keepAlive == null) {
                keepAlive = SipUtil.createChildByFunction(sipService, "createKeepAlive");
            }
            
            HashMap map = new HashMap();
            map.put(SipUtil.KeepAliveKeys.THREAD_COUNT, (String)handlerContext.getInputValue("Count"));
            map.put(SipUtil.KeepAliveKeys.MAX_CONNECTIONS, (String)handlerContext.getInputValue("Connections"));
            map.put(SipUtil.KeepAliveKeys.TIMEOUT_IN_SECONDS, (String)handlerContext.getInputValue("Timeout"));
            
            SipUtil.setAttributeValues(keepAlive, map);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        
    }
    
    
    @Handler(id="getSipConnectionPoolSettings",
    input={
        @HandlerInput(name="ConfigName", type=String.class, required=true)},
    output={
        @HandlerOutput(name="Count",  type=String.class),
        @HandlerOutput(name="Queue",    type=String.class),
        @HandlerOutput(name="Receive",       type=String.class),
        @HandlerOutput(name="Send",       type=String.class)
    })

    public static void getSipConnectionPoolSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        try {
            ObjectName connPool = (ObjectName) SipUtil.getChildByFunction(
                    SipUtil.getSipService(configName), "getConnectionPool");
            String count = "";
            String queue = "";
            String receive = "";
            String send = "";
            
            if (connPool == null) {
                Map defaultMap = AMXUtil.getDomainConfig().getDefaultAttributeValues(ConnectionPoolConfig.J2EE_TYPE);
                count = (String) defaultMap.get(SipUtil.ConnectionPoolKeys.MAX_PENDING_COUNT);
                queue = (String) defaultMap.get(SipUtil.ConnectionPoolKeys.QUEUE_SIZE_IN_BYTES);
                receive = (String) defaultMap.get(SipUtil.ConnectionPoolKeys.RECEIVE_BUFFER_SIZE_IN_BYTES);
                send = (String) defaultMap.get(SipUtil.ConnectionPoolKeys.SEND_BUFFER_SIZE_IN_BYTES);
            } else {
                count = (String) JMXUtil.getAttribute(connPool, (SipUtil.ConnectionPoolKeys.MAX_PENDING_COUNT));
                queue = (String) JMXUtil.getAttribute(connPool, (SipUtil.ConnectionPoolKeys.QUEUE_SIZE_IN_BYTES));
                receive = (String) JMXUtil.getAttribute(connPool, (SipUtil.ConnectionPoolKeys.RECEIVE_BUFFER_SIZE_IN_BYTES));
                send = (String) JMXUtil.getAttribute(connPool, (SipUtil.ConnectionPoolKeys.SEND_BUFFER_SIZE_IN_BYTES));
            }
            handlerContext.setOutputValue("Count", count);
            handlerContext.setOutputValue("Queue", queue);
            handlerContext.setOutputValue("Receive", receive);
            handlerContext.setOutputValue("Send", send);
                    
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }     
    
    @Handler(id="saveSipConnectionPoolSettings",
    input={
        @HandlerInput(name="ConfigName", type=String.class, required=true),
        @HandlerInput(name="Count",  type=String.class),
        @HandlerInput(name="Queue",    type=String.class),
        @HandlerInput(name="Receive",       type=String.class),
        @HandlerInput(name="Send",       type=String.class)
    })

    public static void saveConnectionPoolSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        ObjectName sipService = SipUtil.getSipService(configName);
        try{
            ObjectName connPool = (ObjectName) SipUtil.getChildByFunction(sipService, "getConnectionPool");
            if (connPool == null) {
                connPool = SipUtil.createChildByFunction(sipService, "createConnectionPool");
            }
            HashMap map = new HashMap();
            map.put(SipUtil.ConnectionPoolKeys.MAX_PENDING_COUNT, (String)handlerContext.getInputValue("Count"));
            map.put(SipUtil.ConnectionPoolKeys.QUEUE_SIZE_IN_BYTES, (String)handlerContext.getInputValue("Queue"));
            map.put(SipUtil.ConnectionPoolKeys.RECEIVE_BUFFER_SIZE_IN_BYTES, 
                    (String)handlerContext.getInputValue("Receive"));
            map.put(SipUtil.ConnectionPoolKeys.SEND_BUFFER_SIZE_IN_BYTES, 
                    (String)handlerContext.getInputValue("Send"));
            SipUtil.setAttributeValues(connPool, map);
        } catch (Exception ex){
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    
    @Handler(id="getSipProtocolSettings",
    input = {
        @HandlerInput(name="ConfigName", type=String.class, required=true)
    },
    output = {
        @HandlerOutput(name="ErrorResponse", type=Boolean.class),
        @HandlerOutput(name="DefaultTransport", type=Boolean.class),
        @HandlerOutput(name="ConnectionTimeout", type=String.class),
        @HandlerOutput(name="MaxQueueLength", type=String.class),
        @HandlerOutput(name="WriteTimeout", type=String.class),
        @HandlerOutput(name="WriteTimeoutRetries", type=String.class),
        @HandlerOutput(name="T1", type=String.class),
        @HandlerOutput(name="T2", type=String.class),
        @HandlerOutput(name="T4", type=String.class),
        @HandlerOutput(name="Properties", type=Map.class)
    })
    public static void getSipProtocolSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        try {
            Boolean errorResponse = false;
            Boolean defaultTransport = false;
            String connectionTimeout = "";
            String maxQueueLength = "";
            String writeTimeout = "";
            String writeTimeoutRetries = "";
            String t1 = "";
            String t2 = "";
            String t4 = "";
            HashMap props = new HashMap();
            
            ObjectName sipProtocol = (ObjectName) SipUtil.getChildByFunction(
                    SipUtil.getSipService(configName), "getSipProtocol");
            
            if (sipProtocol == null) {
                getSipProtocolDefaultSettings(handlerContext);
                return;
            } else {
                errorResponse = Boolean.parseBoolean((String)JMXUtil.getAttribute(sipProtocol, 
                        SipUtil.SipProtocolKeys.ERROR_RESPONSE_ENABLED));
                defaultTransport = Boolean.parseBoolean((String)JMXUtil.getAttribute(sipProtocol, 
                        SipUtil.SipProtocolKeys.DEFAULT_TCP_TRANSPORT));
                props = SipUtil.getProperties(sipProtocol);
                
                ObjectName sipLink = (ObjectName) SipUtil.getChildByFunction(sipProtocol, "getSipLink");
                if (sipLink == null) {
                    HashMap defaultMap = SipUtil.getDefaultAttributeValues("sip-link", null);
                    connectionTimeout = (String) defaultMap.get(SipUtil.SipProtocolKeys.CONNECTION_ALIVE_TIMEOUT_IN_SECONDS);
                    maxQueueLength = (String) defaultMap.get(SipUtil.SipProtocolKeys.MAX_QUEUE_LENGTH);
                    writeTimeout = (String) defaultMap.get(SipUtil.SipProtocolKeys.WRITE_TIMEOUT_IN_MILLIS);
                    writeTimeoutRetries = (String) defaultMap.get(SipUtil.SipProtocolKeys.WRITE_TIMEOUT_RETRIES);
                } else {
                    connectionTimeout = (String) JMXUtil.getAttribute(sipLink, SipUtil.SipProtocolKeys.CONNECTION_ALIVE_TIMEOUT_IN_SECONDS);
                    maxQueueLength = (String) JMXUtil.getAttribute(sipLink, SipUtil.SipProtocolKeys.MAX_QUEUE_LENGTH);
                    writeTimeout = (String) JMXUtil.getAttribute(sipLink, SipUtil.SipProtocolKeys.WRITE_TIMEOUT_IN_MILLIS);
                    writeTimeoutRetries = (String) JMXUtil.getAttribute(sipLink, SipUtil.SipProtocolKeys.WRITE_TIMEOUT_RETRIES);
                }
                
                ObjectName sipTimers = (ObjectName) SipUtil.getChildByFunction(sipProtocol, "getSipTimers");
                if (sipTimers == null) {
                    HashMap defaultMap = SipUtil.getDefaultAttributeValues("sip-timers", null);
                    t1 = (String) defaultMap.get(SipUtil.SipProtocolKeys.T1_IN_MILLIS);
                    t2 = (String) defaultMap.get(SipUtil.SipProtocolKeys.T2_IN_MILLIS);
                    t4 = (String) defaultMap.get(SipUtil.SipProtocolKeys.T4_IN_MILLIS);
                } else {
                    t1 = (String) JMXUtil.getAttribute(sipTimers, SipUtil.SipProtocolKeys.T1_IN_MILLIS);
                    t2 = (String) JMXUtil.getAttribute(sipTimers, SipUtil.SipProtocolKeys.T2_IN_MILLIS);
                    t4 = (String) JMXUtil.getAttribute(sipTimers, SipUtil.SipProtocolKeys.T4_IN_MILLIS);
                }
            }
            
            handlerContext.setOutputValue("ErrorResponse", errorResponse);
            handlerContext.setOutputValue("DefaultTransport", defaultTransport);
            handlerContext.setOutputValue("ConnectionTimeout", connectionTimeout);
            handlerContext.setOutputValue("MaxQueueLength", maxQueueLength);
            handlerContext.setOutputValue("WriteTimeout", writeTimeout);
            handlerContext.setOutputValue("WriteTimeoutRetries", writeTimeoutRetries);
            handlerContext.setOutputValue("T1", t1);
            handlerContext.setOutputValue("T2", t2);
            handlerContext.setOutputValue("T4", t4);
            handlerContext.setOutputValue("Properties", props);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    @Handler(id="getSipProtocolDefaultSettings",
    input = {
        @HandlerInput(name="ConfigName", type=String.class, required=true)
    },
    output = {
        @HandlerOutput(name="ErrorResponse", type=Boolean.class),
        @HandlerOutput(name="DefaultTransport", type=Boolean.class),
        @HandlerOutput(name="ConnectionTimeout", type=String.class),
        @HandlerOutput(name="MaxQueueLength", type=String.class),
        @HandlerOutput(name="WriteTimeout", type=String.class),
        @HandlerOutput(name="WriteTimeoutRetries", type=String.class),
        @HandlerOutput(name="T1", type=String.class),
        @HandlerOutput(name="T2", type=String.class),
        @HandlerOutput(name="T4", type=String.class)
    })
    public static void getSipProtocolDefaultSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        HashMap defaultMap = SipUtil.getDefaultAttributeValues("sip-protocol", null);
        Boolean errorResponse = Boolean.parseBoolean((String)defaultMap.get(SipUtil.SipProtocolKeys.ERROR_RESPONSE_ENABLED));
        Boolean defaultTransport = Boolean.parseBoolean((String)defaultMap.get(SipUtil.SipProtocolKeys.DEFAULT_TCP_TRANSPORT));

        defaultMap = SipUtil.getDefaultAttributeValues("sip-link", null);
        String connectionTimeout = (String) defaultMap.get(SipUtil.SipProtocolKeys.CONNECTION_ALIVE_TIMEOUT_IN_SECONDS);
        String maxQueueLength = (String) defaultMap.get(SipUtil.SipProtocolKeys.MAX_QUEUE_LENGTH);
        String writeTimeout = (String) defaultMap.get(SipUtil.SipProtocolKeys.WRITE_TIMEOUT_IN_MILLIS);
        String writeTimeoutRetries = (String) defaultMap.get(SipUtil.SipProtocolKeys.WRITE_TIMEOUT_RETRIES);

        defaultMap = SipUtil.getDefaultAttributeValues("sip-timers", null);
        String t1 = (String) defaultMap.get(SipUtil.SipProtocolKeys.T1_IN_MILLIS);
        String t2 = (String) defaultMap.get(SipUtil.SipProtocolKeys.T2_IN_MILLIS);
        String t4 = (String) defaultMap.get(SipUtil.SipProtocolKeys.T4_IN_MILLIS);
        
        handlerContext.setOutputValue("ErrorResponse", errorResponse);
        handlerContext.setOutputValue("DefaultTransport", defaultTransport);
        handlerContext.setOutputValue("ConnectionTimeout", connectionTimeout);
        handlerContext.setOutputValue("MaxQueueLength", maxQueueLength);
        handlerContext.setOutputValue("WriteTimeout", writeTimeout);
        handlerContext.setOutputValue("WriteTimeoutRetries", writeTimeoutRetries);
        handlerContext.setOutputValue("T1", t1);
        handlerContext.setOutputValue("T2", t2);
        handlerContext.setOutputValue("T4", t4);
    }
    
    @Handler(id="saveSipProtocolSettings",
    input = {
        @HandlerInput(name="ConfigName", type=String.class, required=true),
        @HandlerInput(name="ErrorResponse", type=Boolean.class),
        @HandlerInput(name="DefaultTransport", type=Boolean.class),
        @HandlerInput(name="ConnectionTimeout", type=String.class),
        @HandlerInput(name="MaxQueueLength", type=String.class),
        @HandlerInput(name="WriteTimeout", type=String.class),
        @HandlerInput(name="WriteTimeoutRetries", type=String.class),
        @HandlerInput(name="T1", type=String.class),
        @HandlerInput(name="T2", type=String.class),
        @HandlerInput(name="T4", type=String.class),
        @HandlerInput(name="AddProps", type=Map.class),
        @HandlerInput(name="RemoveProps", type=ArrayList.class)
    })
    public static void saveSipProtocolSettings(HandlerContext handlerContext) {
        String configName = (String) handlerContext.getInputValue("ConfigName");
        HashMap addProps = (HashMap) handlerContext.getInputValue("AddProps");
        ArrayList removeProps = (ArrayList) handlerContext.getInputValue("RemoveProps");
        
        try {
            ObjectName sipService = SipUtil.getSipService(configName);
            // sip-protocol
            ObjectName sipProtocol = (ObjectName) SipUtil.getChildByFunction(sipService, "getSipProtocol");
            if (sipProtocol == null) {
                sipProtocol = SipUtil.createChildByFunction(sipService, "createSipProtocol");
            }
            HashMap props = new HashMap();
            props.put(SipUtil.SipProtocolKeys.ERROR_RESPONSE_ENABLED, Boolean.parseBoolean(
                    handlerContext.getInputValue("ErrorResponse").toString()));
            props.put(SipUtil.SipProtocolKeys.DEFAULT_TCP_TRANSPORT, Boolean.parseBoolean(
                    handlerContext.getInputValue("DefaultTransport").toString()));
            SipUtil.setAttributeValues(sipProtocol, props);
            SipUtil.setProperties(sipProtocol, addProps, removeProps);
            
            // sip-link
            ObjectName sipLink = (ObjectName) SipUtil.getChildByFunction(sipProtocol, "getSipLink");
            if (sipLink == null) {
                sipLink = SipUtil.createChildByFunction(sipService, "createSipLink");
            }
            props = new HashMap();
            props.put(SipUtil.SipProtocolKeys.CONNECTION_ALIVE_TIMEOUT_IN_SECONDS, 
                    (String) handlerContext.getInputValue("ConnectionTimeout"));
            props.put(SipUtil.SipProtocolKeys.WRITE_TIMEOUT_IN_MILLIS, 
                    (String) handlerContext.getInputValue("WriteTimeout"));
            props.put(SipUtil.SipProtocolKeys.WRITE_TIMEOUT_RETRIES, 
                    (String) handlerContext.getInputValue("WriteTimeoutRetries"));
            props.put(SipUtil.SipProtocolKeys.MAX_QUEUE_LENGTH, 
                    (String) handlerContext.getInputValue("MaxQueueLength"));
            SipUtil.setAttributeValues(sipLink, props);
            
            //sip timers
            ObjectName sipTimers = (ObjectName) SipUtil.getChildByFunction(sipProtocol, "getSipTimers");
            if (sipTimers == null) {
                sipTimers = SipUtil.createChildByFunction(sipService, "createSipTimers");
            }
            props = new HashMap();
            props.put(SipUtil.SipProtocolKeys.T1_IN_MILLIS, 
                    (String) handlerContext.getInputValue("T1"));
            props.put(SipUtil.SipProtocolKeys.T2_IN_MILLIS, 
                    (String) handlerContext.getInputValue("T2"));
            props.put(SipUtil.SipProtocolKeys.T4_IN_MILLIS, 
                    (String) handlerContext.getInputValue("T4"));
            SipUtil.setAttributeValues(sipTimers, props);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }    
    
}
