/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.admin.mbeans.extensions;

import com.sun.enterprise.admin.mbeans.ConfigsMBean;
import com.sun.enterprise.admin.server.core.AdminService;
import com.sun.enterprise.admin.target.ConfigTarget;
import com.sun.enterprise.admin.target.Target;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.ApplicationRef;
import com.sun.enterprise.config.serverbeans.AvailabilityService;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.ConfigAPIHelper;
import com.sun.enterprise.config.serverbeans.Configs;
import com.sun.enterprise.config.serverbeans.ConvergedLbClusterRef;
import com.sun.enterprise.config.serverbeans.ConvergedLbConfig;
import com.sun.enterprise.config.serverbeans.ConvergedLbConfigs;
import com.sun.enterprise.config.serverbeans.ConvergedLbPolicy;
import com.sun.enterprise.config.serverbeans.ConvergedLoadBalancer;
import com.sun.enterprise.config.serverbeans.Domain;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.config.serverbeans.Proxy;
import com.sun.enterprise.config.serverbeans.Server;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.ServerRef;
import com.sun.enterprise.config.serverbeans.Clusters;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.util.SystemPropertyConstants;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.util.i18n.StringManagerBase;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.admin.servermgmt.pe.PEFileLayout;
import org.jvnet.glassfish.comms.util.LogUtil;
import com.sun.enterprise.admin.common.MBeanServerFactory;
import com.sun.enterprise.admin.common.Status;
import com.sun.enterprise.admin.servermgmt.RuntimeStatus;
import com.sun.enterprise.admin.servermgmt.RuntimeStatusList;
import org.jvnet.glassfish.comms.admin.clbadmin.ClbAdminEventHelper;
import java.io.File;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.*;


public class ConvergedLbConfigsMBean extends ConfigsMBean {
    private static final StringManager _strMgr = StringManager.getManager(ConvergedLbConfigsMBean.class);
    private static Logger _logger = LogUtil.SIP_LOGGER.getLogger();
    private static final StringManagerBase _sMgr = StringManagerBase.getStringManager(_logger.getResourceBundleName());

    public ConvergedLbConfigsMBean() {
        super();
    }

    private ElementProperty[] convertPropertiesToElementProperties(
        Properties props) {
        ArrayList list = new ArrayList();
        Enumeration keys = props.keys();

        while (keys.hasMoreElements()) {
            final String key = (String) keys.nextElement();
            ElementProperty property = new ElementProperty();
            property.setName(key);
            property.setValue((String) props.get(key));
            list.add(property);
        }

        return (ElementProperty[]) list.toArray(new ElementProperty[list.size()]);
    }

    protected ConvergedLbConfigs getConvergedLbConfigs() {
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
            Domain domain = (Domain) ctx.getRootConfigBean();
            ConvergedLbConfigs clbConfigs = domain.getConvergedLbConfigs();

            if (clbConfigs == null) {
                _logger.log(Level.INFO, _strMgr.getString("ClbConfigsNull"));
                clbConfigs = new ConvergedLbConfigs();
                domain.setConvergedLbConfigs(clbConfigs);
            } else {
                _logger.log(Level.INFO, _strMgr.getString("ClbConfigsNotNull"));
            }

            return clbConfigs;
        } catch (ConfigException ce) {
            return null;
        }
    }

    protected ConvergedLbConfig getConvergedLbConfig(String configName)
        throws MBeanException {
        ConvergedLbConfigs clbConfigs = getConvergedLbConfigs();

        if (clbConfigs == null) {
                _logger.log(Level.INFO, _strMgr.getString("ClbConfigsNull"));
            _logger.log(Level.FINE, _sMgr.getString("sip.clbadmin.NoLbConfigs"));

            String msg = _strMgr.getString("NoLbConfigsElement");
            throw new MBeanException(new ConfigException(msg));
        }

        ConvergedLbConfig clbConfig = clbConfigs.getConvergedLbConfigByName(configName);

        if (clbConfig == null) {
            _logger.log(Level.FINE, _sMgr.getString("sip.clbadmin.NoLbConfigs"));

            String msg = _strMgr.getString("InvalidLbConfigName", configName);
            throw new MBeanException(new ConfigException(msg));
        }

        return clbConfig;
    }

    public String[] getConvergedLbConfigsForServer(String serverName)
        throws MBeanException {
        ArrayList sList = new ArrayList();
        ConvergedLbConfigs clbConfigs = getConvergedLbConfigs();

        if (clbConfigs == null) {
            // return, no lb configs
            return null;
        }

        ConvergedLbConfig[] clbConfigArray = clbConfigs.getConvergedLbConfig();

        for (int i = 0; i < clbConfigArray.length; i++) {
            ServerRef[] sRefs = clbConfigArray[i].getServerRef();

            for (int j = 0; j < sRefs.length; j++) {
                if (sRefs[j].getRef().equals(serverName)) {
                    sList.add(clbConfigArray[i].getName());
                }
            }
        }

        String[] strList = new String[sList.size()];

        return (String[]) sList.toArray(strList);
    }

    public String[] getConvergedLbConfigsForCluster(String clusterName)
        throws MBeanException {
        ArrayList sList = new ArrayList();
        ConvergedLbConfigs clbConfigs = getConvergedLbConfigs();

        if (clbConfigs == null) {
            // return, no lb configs
            return null;
        }

        ConvergedLbConfig[] clbConfigArray = clbConfigs.getConvergedLbConfig();

        for (int i = 0; i < clbConfigArray.length; i++) {
            ConvergedLbClusterRef[] cRefs = clbConfigArray[i].getConvergedLbClusterRef();

            for (int j = 0; j < cRefs.length; j++) {
                if (cRefs[j].getRef().equals(clusterName)) {
                    sList.add(clbConfigArray[i].getName());
                }
            }
        }

        String[] strList = new String[sList.size()];

        return (String[]) sList.toArray(strList);
    }

    public void createConvergedLbConfig(
        String dcrFileName, String httpLbPolicy, String sipLbPolicy,
        String targetName, Properties props, String clbConfigName) throws MBeanException {
        ConvergedLbConfigs clbConfigs = null;
        ConvergedLbConfig clbConfig = null;
        
        if (targetName == null && clbConfigName == null) {
            String msg = _strMgr.getString("CLBConfigSpecifytargetOrCfgName");
            throw new MBeanException(new ConfigException(msg));
        }

        if (httpLbPolicy != null && !(httpLbPolicy.equals("round-robin") || httpLbPolicy.equals("weighted-round-robin") || httpLbPolicy.equals("user-defined"))) {
            String msg = _strMgr.getString("InvalidHttpValue");
            throw new MBeanException(new ConfigException(msg));
        }

        if (sipLbPolicy != null && !(sipLbPolicy.equals("from-tag,to-tag,call-id"))) {
                String msg = _strMgr.getString("InvalidSipValue");
                throw new MBeanException(new ConfigException(msg));
        }

        try {
            clbConfigs = getConvergedLbConfigs();
            
            // if clbConfigName is not specified, generate the same
            if (clbConfigName == null)
                clbConfigName = generateClbCfgName(clbConfigs);

            clbConfig = new ConvergedLbConfig();
            clbConfig.setName(clbConfigName);

            ConvergedLbPolicy clbPolicy = new ConvergedLbPolicy();
            clbPolicy.setHttp(httpLbPolicy);
            clbPolicy.setSip(sipLbPolicy);
            clbPolicy.setDcrFile(dcrFileName);
            clbConfig.setConvergedLbPolicy(clbPolicy);
 
            //properties
            if (null != props)
                clbConfig.setElementProperty(convertPropertiesToElementProperties(props));

            clbConfigs.addConvergedLbConfig(clbConfig);
            
            if (targetName != null)
                createConvergedLbRef(null, clbConfigName, true, false, targetName);
        } catch (Exception ce) {
            clbConfigs.removeConvergedLbConfig(clbConfig);
            throw new MBeanException(ce);
        }
    } 
   
    public String generateClbCfgName(ConvergedLbConfigs cfgs) {
        String name = null;
        ConvergedLbConfig clbConfig = null;
        for (int i=0; ; i++) {
            name = "CLB_CONFIG_" + i;
            clbConfig = cfgs.getConvergedLbConfigByName(name);
            if (clbConfig == null)
                break;
        }
        return name;
    }

    public boolean deleteConvergedLbConfig(String configName)
        throws MBeanException {
        ConvergedLbConfigs clbConfigs = getConvergedLbConfigs();

        if (clbConfigs == null) {
            String msg = _strMgr.getString("NoLbConfigsElement");
            throw new MBeanException(new ConfigException(msg));
        }

        ConvergedLbConfig clbConfig = clbConfigs.getConvergedLbConfigByName(configName);

        if (clbConfig == null) {
            // Nothing to be deleted
            String msg = _strMgr.getString("InvalidLbConfigName", configName);
            throw new MBeanException(new ConfigException(msg));
        }

        if (((clbConfig.getServerRef() == null) ||
                (clbConfig.getServerRef().length == 0)) &&
                ((clbConfig.getConvergedLbClusterRef() == null) ||
                (clbConfig.getConvergedLbClusterRef().length == 0))) {
            clbConfigs.removeConvergedLbConfig(clbConfig);
            _logger.log(Level.INFO,
                _sMgr.getString("sip.clbadmin.LbConfigDeleted", configName));
        } else {
            String msg = _strMgr.getString("LbConfigNotEmpty", configName);
            throw new MBeanException(new ConfigException(msg));
        }

        return true;
    }

    public String[] listConvergedLbConfigs(String target)
        throws MBeanException {
        _logger.log(Level.FINE, "[LBAdminMBean] listLBConfigs called");

        ConvergedLbConfigs clbConfigs = getConvergedLbConfigs();
        if (clbConfigs == null) {
            _logger.log(Level.INFO, _sMgr.getString("sip.clbadmin.NoLbConfigs"));

            return null;
        }

        String[] names = null;

        if (target == null) {
            ConvergedLbConfig[] clbConfigArray = clbConfigs.getConvergedLbConfig();

            if (clbConfigArray.length == 0) {
                _logger.log(Level.INFO,
                    _sMgr.getString("sip.clbadmin.NoLbConfigs"));

                return null;
            }

            names = new String[clbConfigArray.length];

            for (int i = 0; i < clbConfigArray.length; i++) {
                names[i] = clbConfigArray[i].getName();
            }
        } else {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();

            try {
                // target is a cluster
                if (ClusterHelper.isACluster(ctx, target)) {
                    names = getConvergedLbConfigsForCluster(target);

                    // target is a server
                } else if (ServerHelper.isAServer(ctx, target)) {
                    names = getConvergedLbConfigsForServer(target);
                } else {
                    String msg = _strMgr.getString("TargetDoesNotExist", target);
                    throw new MBeanException(new ConfigException(msg));
                }
            } catch (ConfigException ce) {
                throw new MBeanException(ce);
            }
        }

        return names;
    }

    public String[] createConvergedLoadBalancer(String clbConfigName,
        AttributeList attrList,
        String clbConfigFile, // location of converged-load-balancer.xml
        Boolean autocommit, // if true, apply changes immediately
        String targetName, // target name (cluster or server)
        Properties props, // proxy's properties
        String clbName) throws MBeanException {
        String returnMsg[] = null;
        try {
            String dcrFileName = null;
            String httplbPolicy = null;
            String siplbPolicy = null;
            String lbtargetName = null;
            String selfLoadbalance = null;
            String lbenableallInstances = null;
            
            // check to see if clbName is unique across domain
            _logger.log(Level.FINE, "Checking if CLB Name is unique across domain");
            String[] clbs = listConvergedLoadBalancers();
            for (int i=0; clbs != null && i < clbs.length; i++) {
                if (clbs[i].equals(clbName)) {
                    String msg = _strMgr.getString("ClbNameAlreadyExists");
                    throw new MBeanException(new ConfigException(msg));
                }
            }
        
            for (int i = 0; i <attrList.size(); i++) {
                Attribute attr = (Attribute)attrList.get(i);
                if (isAttrNameMatch(attr, "dcrfile"))
                    dcrFileName = (String)attr.getValue();
                else if (isAttrNameMatch(attr, "httplbpolicy"))
                    httplbPolicy = (String)attr.getValue();
                else if (isAttrNameMatch(attr, "siplbpolicy"))
                    siplbPolicy = (String)attr.getValue();
                else if (isAttrNameMatch(attr, "lbtarget"))
                    lbtargetName = (String)attr.getValue();
                else if (isAttrNameMatch(attr, "selfloadbalance"))
                    selfLoadbalance = (String)attr.getValue();
                else if (isAttrNameMatch(attr, "lbenableallinstances"))
                    lbenableallInstances = (String)attr.getValue();
            }
            final Target target = getTarget(targetName);
            final ConfigTarget configTarget = target.getConfigTarget();

            Config config = ConfigAPIHelper.getConfigByName(getConfigContext(),
                    configTarget.getName());
            AvailabilityService availabilityService = config.getAvailabilityService();
            
            _logger.log(Level.FINE, "Checking if Clb Command is valid");
            if (clbConfigName != null && (dcrFileName != null || httplbPolicy != null || siplbPolicy != null || lbtargetName != null || lbenableallInstances.equals("false"))) {
                String msg = _strMgr.getString("InvalidClbCommand");
                throw new MBeanException(new ConfigException(msg));
            }

            if (availabilityService == null) {
                availabilityService = new AvailabilityService();
                config.setAvailabilityService(availabilityService);
            }

            ConvergedLoadBalancer clb = availabilityService.getConvergedLoadBalancer();

            if (clb != null) {
                String msg = _strMgr.getString("CLBAlreadyConfigured", targetName);
                throw new MBeanException(new ConfigException(msg));
            }

            Boolean selfLb = Boolean.valueOf(selfLoadbalance);
            if (selfLb.booleanValue() && lbtargetName == null)
                lbtargetName = targetName;
            
            _logger.log(Level.FINE, "Checking if selfloadbalance option and target provided are valid");
            if (selfLb.booleanValue() && !lbtargetName.equals(targetName)) {
                String msg = _strMgr.getString("ClbTargetSameHostingTarget");
                throw new MBeanException(new ConfigException(msg));
            }
            else if (!selfLb.booleanValue()) {
                if (lbtargetName != null && lbtargetName.equals(targetName)) {
                    String msg = _strMgr.getString("ClbTargetNotSameHostingTarget");
                    throw new MBeanException(new ConfigException(msg));
                }
            }
            
            if (clbConfigName == null) {
                // create clb config name with given clbName
                clbConfigName = clbName + "_CLB_CONFIG";
                _logger.log(Level.FINE, "Creating CLB Config " + clbConfigName + "as part of creation of CLB");
                createConvergedLbConfig(dcrFileName, httplbPolicy, siplbPolicy, lbtargetName, null, clbConfigName);
            } 
            else {
                // check if given clb config exists
                getConvergedLbConfig(clbConfigName);
            }
                    
            // we have to create it
            clb = new ConvergedLoadBalancer();
            clb.setName(clbName);
            clb.setConfigFile(clbConfigFile);
            clb.setConvergedLbConfigName(clbConfigName);
            clb.setAutoCommit(autocommit);

            Proxy proxy = new Proxy();

            //properties
            if (null != props) {
                proxy.setElementProperty(convertPropertiesToElementProperties(
                        props));
            }

            clb.setProxy(proxy);
            availabilityService.setConvergedLoadBalancer(clb);
            
            // If lbenableallinstances is true, set lb-enabled flag to true
            if ((lbenableallInstances != null) && (lbenableallInstances.equals("true"))) {
                ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
                Domain domain = (Domain) ctx.getRootConfigBean();
                Clusters clusters = domain.getClusters();
                ConvergedLbConfig clbConfig = domain.getConvergedLbConfigs().getConvergedLbConfigByName(clbConfigName);
                ConvergedLbClusterRef[] clbRef = clbConfig.getConvergedLbClusterRef();
                for (int ref=0; ref < clbRef.length; ref++) {
                    String clusterName = clbRef[ref].getRef();
                    if (clusterName != null ) {
                        Cluster cluster = clusters.getClusterByName(clusterName);
                        ServerRef[] serverRefs = cluster.getServerRef();
                        for (int i=0; i < serverRefs.length; i++) {
                            if (!serverRefs[i].isLbEnabled())
                                serverRefs[i].setLbEnabled(true);
                        }
                    }
                }
            }
            
            // Issue - 800, if cluster is running report to the user
            // cluster needs to be restarted
            if (isTargetRunning(targetName))
                returnMsg = new String[] {_strMgr.getString("ClusterRestartRequired")};
        } catch (Exception ce) {
            throw new MBeanException(ce);
        }

        return returnMsg;
    }

    private boolean isTargetRunning(String targetName) throws MBeanException{
        String status = "";
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext().getAdminConfigContext();
            ObjectName on = new ObjectName("com.sun.appserv:type=cluster,name=" + targetName + 
		    ",category=config");
            if (ServerHelper.isAServer(ctx, targetName)) {
                on = new ObjectName("com.sun.appserv:type=server,name=" + targetName + 
		    ",category=config");
                RuntimeStatus rtStatus = (RuntimeStatus) MBeanServerFactory.getMBeanServer().invoke(
		    on, "getRuntimeStatus", null, null);
		if ((rtStatus != null) && (rtStatus.getStatus() != null)) {
		    status = rtStatus.getStatus().getStatusString();
		}
            } else {
                RuntimeStatusList rtStatus = (RuntimeStatusList) MBeanServerFactory.getMBeanServer().invoke(
			on, "getRuntimeStatus", null, null);
		if (rtStatus != null) {
		    status = rtStatus.toString();
		    if (Status.getStatusString(Status.kClusterPartiallyRunningCode).equals(status)) {
			status = Status.getStatusString(Status.kInstanceRunningCode);
		    }
		}
            }
        } catch (Exception ce) {
            throw new MBeanException(ce);
        }
        return Status.getStatusString(Status.kInstanceRunningCode).equals(status);
    }
    
    public boolean deleteConvergedLoadBalancer(String clbName)
        throws MBeanException {
        _logger.log(Level.FINE, "Deleting CLB Config Name as part of deletion of CLB");
        
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                           .getAdminConfigContext();
            Domain domain = (Domain) ctx.getRootConfigBean();
            Configs configs = domain.getConfigs();
            String targetName = null;

            if (configs != null) {
                boolean found = false;
                Config[] configArray = configs.getConfig();
                for (int i = 0, j = 0; i < configArray.length; i++) {
                    AvailabilityService avSvc = configArray[i].getAvailabilityService();
                    if (avSvc != null) {
                        ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                  .getConvergedLoadBalancer();

                        if (clb != null && clb.getName().equals(clbName)) {
                            // Issue 803, check if Cluster is running before deleting CLB
                            String configName = configArray[i].getName();
                            Cluster[] cluster = domain.getClusters().getCluster();
                            Server[] server = domain.getServers().getServer();
                            boolean isClusterConfig = false;
                            // In the case if clb is created in cluster config
                            for (int cnt=0; cnt < cluster.length; cnt++) {
                                String configRef = cluster[cnt].getConfigRef();
                                if (configRef.equals(configName)) {
                                    targetName = cluster[cnt].getName();
                                    isClusterConfig = true;
                                    if (isTargetRunning(targetName)) {
                                        String msg = _strMgr.getString("ClusterStopRequired");
                                        throw new MBeanException(new ConfigException(msg));
                                    }
                                }
                            }
                            // In the case if clb is created in server config
                            if (!isClusterConfig) {
                                for (int cnt=0; cnt < server.length; cnt++) {
                                    String configRef = server[cnt].getConfigRef();
                                    if (configRef.equals(configName)) {
                                        targetName = server[cnt].getName();
                                        if (isTargetRunning(targetName)) {
                                            String msg = _strMgr.getString("ServerStopRequired");
                                            throw new MBeanException(new ConfigException(msg));
                                        }
                                    }
                                }
                            }
                            avSvc.setConvergedLoadBalancer(null);
                            found = true;
                            break;
                        }
                    }
                }
                if (!found) {
                    String msg = _strMgr.getString("ClbNameDoesNotExist");
                    throw new MBeanException(new ConfigException(msg));
                }
            }
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }
	
	try {
            ConvergedLbConfigs clbConfigs = getConvergedLbConfigs();
            ConvergedLbConfig clbConfig = clbConfigs.getConvergedLbConfigByName(clbName + "_CLB_CONFIG");
            if (clbConfig != null) {
                _logger.log(Level.FINE, "Before deleting CLB Checking if it contains cluster references");
                ConvergedLbClusterRef[] clbClusterRefs = clbConfig.getConvergedLbClusterRef();
                if (clbClusterRefs.length != 0) {
                    String msg = _strMgr.getString("ClbContainsClusterRefs");
                    throw new MBeanException(new ConfigException(msg));
                }
                clbConfigs.removeConvergedLbConfig(clbConfig);
             }
        } catch (Exception ce) {
            throw new MBeanException(ce);
        }
        return true;
    }

    public String[] listConvergedLoadBalancers() throws MBeanException {
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
            Domain domain = (Domain) ctx.getRootConfigBean();
            Configs configs = domain.getConfigs();
            HashSet names = new HashSet(0);

            if (configs != null) {
                Config[] configArray = configs.getConfig();
                for (int i = 0, j = 0; i < configArray.length; i++) {
                    AvailabilityService avSvc = configArray[i].getAvailabilityService();
                    if (avSvc != null) {
                        ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                  .getConvergedLoadBalancer();

                        if (clb != null) {
                            names.add(clb.getName());
                        }
                    }
                }
            }
            if (names.size() > 0) {
                return (String[]) names.toArray(new String[names.size()]);
            } 
        } catch (ConfigException ce) {
        }
        return null;
    }

    private void addServerToConvergedLbConfig(String configName,
        String serverName) throws MBeanException {
        ConvergedLbConfig clbConfig = getConvergedLbConfig(configName);

        // Check if there is already cluster-ref
        ConvergedLbClusterRef[] clbRefs = clbConfig.getConvergedLbClusterRef();
        if (clbRefs != null && clbRefs.length > 0) {
            String msg = _strMgr.getString("ClusterServerRefCannotCoExist");
            throw new MBeanException(new ConfigException(msg));
        }
        
        // check if there is already server-ref
        ServerRef[] sRefs = clbConfig.getServerRef();
        if (sRefs != null && sRefs.length > 0) {
            String msg = _strMgr.getString("ClusterServerRefCannotCoExist", serverName);
            throw new MBeanException(new ConfigException(msg));
        }

        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();

            if (!ServerHelper.isServerStandAlone(ctx, serverName)) {
                String msg = _strMgr.getString("NotStandAloneInstance",
                        serverName);
                throw new MBeanException(new ConfigException(msg));
            }

            ServerRef sRef = new ServerRef();
            sRef.setRef(serverName);
            clbConfig.addServerRef(sRef);
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }
    }

    private void addClusterToConvergedLbConfig(String configName,
        String clusterName, boolean selfloadbalance) throws MBeanException {
        ConvergedLbConfig clbConfig = getConvergedLbConfig(configName);

        // check if there is already server-ref
        ServerRef[] sRefs = clbConfig.getServerRef();
        if (sRefs != null && sRefs.length > 0) {
            String msg = _strMgr.getString("ClusterServerRefCannotCoExist");
            throw new MBeanException(new ConfigException(msg));
        }
        
        ConvergedLbClusterRef[] clbRefs = clbConfig.getConvergedLbClusterRef();
        if (clbRefs != null) {
            for (int i=0; i < clbRefs.length; i++) {
                // check if there is already cluster-ref with same name
                if (clbRefs[i].getRef().equals(clusterName)) {
                    String msg = _strMgr.getString("CLBRefExist", clusterName);
                    throw new MBeanException(new ConfigException(msg));
                } else {
                    // check if there is already cluster-ref with selfloadbalance as true
                    boolean selfLb = clbRefs[i].isSelfLoadbalance();
                    if (selfLb) {
                        String msg = _strMgr.getString("CLBConfigAsSelfLB");
                        throw new MBeanException(new ConfigException(msg));
                    } else if (!selfLb && selfloadbalance) { // check if there is already cluster-ref with self loadbalance as false and current selfloadbalance is true
                        String msg = _strMgr.getString("CLBConfigAsSelfLBMoreRef");
                        throw new MBeanException(new ConfigException(msg));
                    }
                }
            }
        }
            
        ConvergedLbClusterRef clbRef = new ConvergedLbClusterRef();
        clbRef.setRef(clusterName);
        clbRef.setSelfLoadbalance(selfloadbalance);

        try {
            clbConfig.addConvergedLbClusterRef(clbRef);
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }
    }

    public void createConvergedLbRef(String clbName,
        String clbConfigName, Boolean selfloadbalance,
        Boolean lbEnableInstances, String target) throws MBeanException {
        _logger.log(Level.FINE,
            "[LBAdminMbean] createLBRef called for target " + target);

        _logger.log(Level.FINE, "Checking if either CLB Config Name or CLB is provided");
        if ((clbConfigName != null && clbName != null) || (clbConfigName == null && clbName == null)) {
            String msg = _strMgr.getString("LbConfigNameNotValid");
            throw new MBeanException(new ConfigException(msg));
        }
        
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                        .getAdminConfigContext();

        try {
            // get the clbconfig name if clb name is provided
            _logger.log(Level.FINE, "Checking if CLB Name provided contains valid CLB Config Name");
            if (clbName != null) {
                Domain domain = (Domain) ctx.getRootConfigBean();
                Configs configs = domain.getConfigs();
                
                if (configs != null) {
                    Config[] configArray = configs.getConfig();
                    for (int i = 0, j = 0; i < configArray.length; i++) {
                        AvailabilityService avSvc = configArray[i].getAvailabilityService();
                        if (avSvc != null) {
                            ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                      .getConvergedLoadBalancer();

                            if (clb != null && clb.getName().equals(clbName))
                                clbConfigName = clb.getConvergedLbConfigName();
                        }
                    }
                }
                if (clbConfigName == null) {
                    String msg = _strMgr.getString("ReqdValidLbConfigName");
                    throw new MBeanException(new ConfigException(msg));
                }
            }
            
            // target is a cluster
            if (ClusterHelper.isACluster(ctx, target)) {
                addClusterToConvergedLbConfig(clbConfigName, target,
                    selfloadbalance);
                _logger.log(Level.INFO,
                    _sMgr.getString("sip.clbadmin.AddClusterToConfig", target,
                        clbConfigName));

                // target is a server
            } else if (ServerHelper.isAServer(ctx, target)) {
                addServerToConvergedLbConfig(clbConfigName, target);
                _logger.log(Level.INFO,
                    _sMgr.getString("sip.clbadmin.AddServerToConfig", target,
                        clbConfigName));
            } else {
                String msg = _strMgr.getString("InvalidTarget", target);
                throw new MBeanException(new RuntimeException(msg));
            }

            ApplicationRef[] appRefs = null;
            ConvergedLbConfig clbConfig = getConvergedLbConfig(clbConfigName);

            if (ClusterHelper.isACluster(ctx, target)) {
                Cluster c = ClusterHelper.getClusterByName(ctx, target);

                if (c == null) {
                    String msg = _strMgr.getString("ClusterNotDefined", target);
                    throw new MBeanException(new ConfigException(msg));
                }

                appRefs = c.getApplicationRef();

                ServerRef[] sRefs = c.getServerRef();

                for (int i = 0; i < sRefs.length; i++)
                    sRefs[i].setLbEnabled(lbEnableInstances);
            } else {
                Server s = ServerHelper.getServerByName(ctx, target);

                if (s == null) {
                    String msg = _strMgr.getString("InstanceNotFound", target);
                    throw new MBeanException(new ConfigException(msg));
                }

                appRefs = s.getApplicationRef();

                ServerRef sRef = clbConfig.getServerRefByRef(target);
                sRef.setLbEnabled(lbEnableInstances);
            }
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }
    }

    public void deleteConvergedLbRef(String clbConfigName, String clbName, String target)
        throws MBeanException {
        _logger.log(Level.FINE,
            "[LBAdminMBean] deleteLBRef called for target " + target);

        _logger.log(Level.FINE, "Checking if either CLB Config Name or CLB is provided");
        if ((clbConfigName != null && clbName != null) || (clbConfigName == null && clbName == null)) {
            String msg = _strMgr.getString("LbConfigNameNotValid");
            throw new MBeanException(new ConfigException(msg));
        }
        
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                        .getAdminConfigContext();

        try {
            _logger.log(Level.FINE, "Checking if CLB Name provided contains valid CLB Config Name");
            if (clbName != null) {
                Domain domain = (Domain) ctx.getRootConfigBean();
                Configs configs = domain.getConfigs();
                
                if (configs != null) {
                    Config[] configArray = configs.getConfig();
                    for (int i = 0, j = 0; i < configArray.length; i++) {
                        AvailabilityService avSvc = configArray[i].getAvailabilityService();
                        if (avSvc != null) {
                            ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                      .getConvergedLoadBalancer();

                            if (clb != null && clb.getName().equals(clbName))
                                clbConfigName = clb.getConvergedLbConfigName();
                        }
                    }
                }
                if (clbConfigName == null) {
                    String msg = _strMgr.getString("ReqdValidLbConfigName");
                    throw new MBeanException(new ConfigException(msg));
                }
            }
            
            // target is a cluster
            if (ClusterHelper.isACluster(ctx, target)) {
                deleteClusterFromConvergedLbConfig(clbConfigName, target);
                _logger.log(Level.INFO,
                    _sMgr.getString("sip.clbadmin.DeleteClusterFromConfig",
                        target, clbConfigName));

                // target is a server
            } else if (ServerHelper.isAServer(ctx, target)) {
                deleteServerFromConvergedLbConfig(clbConfigName, target);
                _logger.log(Level.INFO,
                    _sMgr.getString("sip.clbadmin.DeleteServerFromConfig",
                        target, clbConfigName));
            } else {
                String msg = _strMgr.getString("InvalidTarget", target);
                throw new MBeanException(new RuntimeException(msg));
            }
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }
    }

    private void deleteServerFromConvergedLbConfig(String configName,
        String serverName) throws MBeanException {
        ConvergedLbConfig clbConfig = getConvergedLbConfig(configName);

        ServerRef sRef = clbConfig.getServerRefByRef(serverName);

        if (sRef == null) {
            // does not exist, just return from here
            _logger.log(Level.FINEST,
                " server " + serverName +
                " does not exist in any cluster in the domain");

            String msg = _strMgr.getString("ServerNotDefined", serverName);
            throw new MBeanException(new RuntimeException(msg));
        }

        if (sRef.isLbEnabled()) {
            String msg = _strMgr.getString("ServerNeedsToBeDisabled", serverName);
            throw new MBeanException(new ConfigException(msg));
        }

        // check if its applications are LB disabled.
        Server s = null;

        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
            s = ServerHelper.getServerByName(ctx, sRef.getRef());
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }

        if (s == null) {
            String msg = _strMgr.getString("ServerNotDefined", serverName);
            throw new MBeanException(new ConfigException(msg));
        }

        ApplicationRef[] appRef = s.getApplicationRef();

        if (appRef == null) {
            String msg = _strMgr.getString("AppRefsNotDefined", serverName);
            throw new MBeanException(new ConfigException(msg));
        }

        int i = 0;

        for (i = 0; i < appRef.length; i++) {
            if (appRef[i].isLbEnabled()) {
                break;
            }
        }

        if (i < appRef.length) {
            String msg = _strMgr.getString("AppsNotDisabled");
            throw new MBeanException(new ConfigException(msg));
        }

        clbConfig.removeServerRef(sRef);
    }

    private void deleteClusterFromConvergedLbConfig(String configName,
        String clusterName) throws MBeanException {
        ConvergedLbConfig clbConfig = getConvergedLbConfig(configName);

        ConvergedLbClusterRef clbcRef = clbConfig.getConvergedLbClusterRefByRef(clusterName);

        if (clbcRef == null) {
            String msg = _strMgr.getString("ClusterNotDefinedInClbConfig", clusterName);
            throw new MBeanException(new ConfigException(msg));
        }

        Cluster c = null;

        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
            c = ClusterHelper.getClusterByName(ctx, clusterName);
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }

        if (c == null) {
            String msg = _strMgr.getString("ClusterNotDefined", clusterName);
            throw new MBeanException(new ConfigException(msg));
        }

        ServerRef[] sRefs = c.getServerRef();

        for (int i = 0; i < sRefs.length; i++) {
            if (sRefs[i].isLbEnabled()) {
                String msg = _strMgr.getString("ServerNeedsToBeDisabled",
                        clusterName);
                throw new MBeanException(new ConfigException(msg));
            }
        }

        clbConfig.removeConvergedLbClusterRef(clbcRef);
    }

    public void setDcrFile(String dcrFileName,
                     String clbConfigName, String clbName) throws MBeanException {

        // check if either of one option is provided
        if ((clbConfigName != null && clbName != null) || (clbConfigName == null && clbName == null)) {
            String msg = _strMgr.getString("DCRInvalidOption");
            throw new MBeanException(new ConfigException(msg));
        }
         
        // if clbname is specified get the clbconfig
        if (clbConfigName == null)
            clbConfigName = getCLBConfigUsingCLB(clbName);
        
        File file = new File(dcrFileName);
        String dcrFile = file.getName();
                    
        // check if dcr file is already set
        String dcrNewFile = null;
        ConvergedLbConfigs clbConfigs = null;
        ConvergedLbConfig clbConfig = null;
        try {
            clbConfigs = getConvergedLbConfigs();
            if (clbConfigs == null) {
                String msg = _strMgr.getString("NoLbConfigsElement");
                throw new MBeanException(new ConfigException(msg));
            }

            clbConfig = clbConfigs.getConvergedLbConfigByName(clbConfigName);
            if (clbConfig == null) {
                String msg = _strMgr.getString("InvalidLbConfigName", clbConfigName);
                throw new MBeanException(new ConfigException(msg));
            }

            MBeanServer mbs = AdminService.getAdminService().getAdminContext().getMBeanServer();
            String[] types = new String[] {(new String[]{}).getClass().getName()};
            // Get the value of config-file
            Object[] returnValues = (Object[])mbs.invoke(
                                        new ObjectName("com.sun.appserv:name=dotted-name-get-set,type=dotted-name-support"),
                                        "dottedNameGet",
                                        new Object[] {new String[] {"domain.converged-lb-configs." + clbConfigName + ".converged-lb-policy.dcr-file"}},
                                        types);
            Attribute attr = (Attribute) returnValues[0];
            String dcrOldFile = (String) attr.getValue();
            dcrNewFile = dcrFile;
            if (dcrOldFile == null || (dcrOldFile != null && dcrOldFile.equals("")))
                dcrOldFile = dcrFile;
            
            String dcrOldFileWOVersion = null;
            if (dcrOldFile.lastIndexOf(".v") != -1) {
                int versionNumPos = dcrOldFile.lastIndexOf(".v");
                dcrOldFileWOVersion = dcrOldFile.substring(0, versionNumPos);
            } else
                dcrOldFileWOVersion = dcrOldFile;
            
            String iRoot = System.getProperty(SystemPropertyConstants.INSTANCE_ROOT_PROPERTY);
            String sDcrOldFile = iRoot + File.separator + PEFileLayout.CONFIG_DIR + File.separator + dcrFile;
            File oldFileName = new File(sDcrOldFile);
            Object[] configsUsingCLBConfig = getConfigsUsingCLBConfig(clbConfigName);
            // If clb config is not used by any CLB,
            // do not copy the dcr file from the config directory to cluster config
            if (isClbCfgUsed(clbConfigName)) {
                if (dcrOldFileWOVersion.equals(dcrFile)) {
                    // Following gets executed when the uploaded dcr file name is same as previous dcr file name
                    dcrNewFile = ClbAdminEventHelper.getConfigFileNewValue(dcrOldFile);
                    try {
                        if (configsUsingCLBConfig != null) {
                            for (int i=0; i < configsUsingCLBConfig.length; i++) {
                                // Following copies the dcr file from config directory to config/<cluster-specific-config> directory
                                String sDcrNewFile = iRoot + File.separator + PEFileLayout.CONFIG_DIR + File.separator + configsUsingCLBConfig[i] 
                                                                                                                            + File.separator + dcrNewFile;
                                File newFileName = new File(sDcrNewFile);
                                FileUtils.copy(oldFileName, newFileName);

                                // Following deletes the dcr file under config/<cluster-specific-config> directory
                                // Do not delete the old dcr files if debug is set as true
                                if (!ClbAdminEventHelper.isClbDebug()) {
                                    String tmp = iRoot + File.separator + PEFileLayout.CONFIG_DIR + File.separator + configsUsingCLBConfig[i] 
                                                                                                                            + File.separator + dcrOldFile;
                                    File tmpFile = new File(tmp);
                                    if (tmpFile.exists())
                                        tmpFile.delete();
                                }
                            }
                        }
                    } catch (Exception e) {
                        throw new MBeanException(e);
                    }

                } else { 
                    // Following gets executed when the uploaded dcr file name is different from previous dcr file name
                    if (configsUsingCLBConfig != null) {
                        for (int i=0; i < configsUsingCLBConfig.length; i++) {
                            String sDcrNewFile = iRoot + File.separator + PEFileLayout.CONFIG_DIR + File.separator + configsUsingCLBConfig[i] 
                                                                                                                        + File.separator + dcrNewFile;
                            File newFileName = new File(sDcrNewFile);
                            FileUtils.copy(oldFileName, newFileName);

                            // Following deletes the dcr file under config/<cluster-specific-config> directory
                            // Do not delete the old dcr files if debug is set as true
                            if (!ClbAdminEventHelper.isClbDebug()) {
                                String tmp = iRoot + File.separator + PEFileLayout.CONFIG_DIR + File.separator + configsUsingCLBConfig[i]
                                                                                                                        + File.separator + dcrOldFile;
                                File tmpFile = new File(tmp);
                                if (tmpFile.exists())
                                    tmpFile.delete();
                            }
                        }
                    }
                }
                // Following deletes the dcr file under the domains/domain/config directory
                // sDcrOldFile = iRoot + File.separator + PEFileLayout.CONFIG_DIR + File.separator + dcrFile;
                // File tmpFile = new File(sDcrOldFile);
                if (oldFileName.exists())
                    oldFileName.delete();
            }
        } catch (Exception e) {
            throw new MBeanException(e);
        }

        ConvergedLbPolicy clbPolicy = clbConfig.getConvergedLbPolicy();
        clbPolicy.setDcrFile(dcrNewFile);
    }

    public ObjectName getAvailabilityService(String targetName)
        throws MBeanException {
        return getChild("availability-service", null, targetName);
    }

    protected ObjectName getAvailabilityServiceMBean(Target target)
        throws MBeanException {
        final ObjectName configMBean = getConfigMBean(target);
        ObjectName ret = (ObjectName) invoke0(configMBean,
                "getAvailabilityService");

        return ret;
    }
    
    private boolean doesClbConfigExist(String clbConfigName) throws MBeanException {
        String[] convergedLbConfigs = listConvergedLbConfigs(null);
        boolean clbConfigExists = false;
        if (convergedLbConfigs != null) {
            for (int i=0; i < convergedLbConfigs.length; i++) {
                if (convergedLbConfigs[i].equals(clbConfigName)) {
                    clbConfigExists = true;
                    break;
                }
            }
        }
        return clbConfigExists;
    }
    
    private String getCLBConfigUsingCLB(String clbName) throws MBeanException {
        String clbConfigName = null;
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
        try {
            Domain domain = (Domain) ctx.getRootConfigBean();
            Configs configs = domain.getConfigs();
            if (configs != null) {
                Config[] configArray = configs.getConfig();
                for (int i = 0; i < configArray.length; i++) {
                    AvailabilityService avSvc = configArray[i].getAvailabilityService();
                    if (avSvc != null) {
                        ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                  .getConvergedLoadBalancer();
                        if (clb != null) {
                            if (clb.getName().equals(clbName)) {
                                clbConfigName = clb.getConvergedLbConfigName();
                                break;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new MBeanException(e);
        }
        return clbConfigName;
    }
    
    private Object[] getConfigsUsingCLBConfig(String clbConfigName) throws MBeanException {
        ArrayList configsUsingCLBConfig = new ArrayList();
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                            .getAdminConfigContext();
        try {
            Domain domain = (Domain) ctx.getRootConfigBean();
            Configs configs = domain.getConfigs();
            if (configs != null) {
                Config[] configArray = configs.getConfig();
                for (int i = 0; i < configArray.length; i++) {
                    AvailabilityService avSvc = configArray[i].getAvailabilityService();
                    if (avSvc != null) {
                        ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                  .getConvergedLoadBalancer();
                        if (clb != null) {
                            if (clb.getConvergedLbConfigName().equals(clbConfigName))
                                configsUsingCLBConfig.add(configArray[i].getName());
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new MBeanException(e);
        }
        return configsUsingCLBConfig.toArray();
    }
    
    private static boolean isAttrNameMatch(Attribute attr, String name)
    {
        // for now we supporting both "dashed" and "underscored" names
        return attr.getName().replace('_','-').equals(name.replace('_','-'));
    }
    
    private boolean isClbCfgUsed(String clbConfigName) {
        // check if the specified CLB Config is used by any CLB
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                        .getAdminConfigContext();
        boolean clbConfigUsed = false;
        try {
            Domain domain = (Domain) ctx.getRootConfigBean();
            Configs configs = domain.getConfigs();
            if (configs != null) {
                Config[] configArray = configs.getConfig();
                for (int i = 0; i < configArray.length; i++) {
                    AvailabilityService avSvc = configArray[i].getAvailabilityService();
                    if (avSvc != null) {
                        ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                  .getConvergedLoadBalancer();
                        if (clb != null) {
                            if (clb.getConvergedLbConfigName().equals(clbConfigName)) {
                                clbConfigUsed = true;
                                break;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            // ignore exception
        }
        return clbConfigUsed;
    }
    
    public void unsetDcrFile(String clbName) throws MBeanException {
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                           .getAdminConfigContext();
            Domain domain = (Domain) ctx.getRootConfigBean();
            Configs configs = domain.getConfigs();

            String clbCfgName = null;
            String cfgName = null;
            // check if provided clbName is valid
            if (configs != null) {
                boolean found = false;
                Config[] configArray = configs.getConfig();
                for (int i = 0, j = 0; i < configArray.length; i++) {
                    AvailabilityService avSvc = configArray[i].getAvailabilityService();
                    if (avSvc != null) {
                        ConvergedLoadBalancer clb = configArray[i].getAvailabilityService()
                                                                  .getConvergedLoadBalancer();

                        if (clb != null && clb.getName().equals(clbName)) {
                            // get the clb config name and config name
                            clbCfgName = clb.getConvergedLbConfigName();
                            cfgName = configArray[i].getName();
                            found = true;
                            break;
                        }
                    }
                }
                if (!found) {
                    String msg = _strMgr.getString("ClbNameDoesNotExist");
                    throw new MBeanException(new ConfigException(msg));
                }
            }
            ConvergedLbConfig lbCfg = getConvergedLbConfig(clbCfgName);
            ConvergedLbPolicy clbLbPolicy = lbCfg.getConvergedLbPolicy();
            String dcrFileName = clbLbPolicy.getDcrFile();
            // Before unsetting dcr file delete the physical file
            String iRoot = System.getProperty(SystemPropertyConstants.INSTANCE_ROOT_PROPERTY);
            String dcrFilePath = iRoot + File.separator + PEFileLayout.CONFIG_DIR 
                                                + File.separator + cfgName + File.separator + dcrFileName;
            File dcrFile = new File(dcrFilePath);
            if (dcrFile.exists())
                dcrFile.delete();
            //setting dcr file to empty string
            clbLbPolicy.setDcrFile("");
        } catch (ConfigException ce) {
            throw new MBeanException(ce);
        }
    }
    
    public void disableServer(String target, String time) throws MBeanException {
        
        String msg = null;
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                .getAdminConfigContext();

        try {
            if (ClusterHelper.isACluster(ctx, target)) {
                // disables cluster
                disableCluster(target, time);
            } else if (ServerHelper.isAServer(ctx, target)) {
                // disable server
                boolean found = false;
                Domain domain = (Domain) ctx.getRootConfigBean();
                Cluster[] clusters = domain.getClusters().getCluster();
                for (int i=0; i < clusters.length; i++) {
                    Cluster cluster = clusters[i];
                    ServerRef[] sRefs = cluster.getServerRef();
                    for (int j=0; j < sRefs.length; j++) {
                        if (sRefs[i].getRef().equals(target)) {
                            sRefs[i].setLbEnabled(false);
                            sRefs[i].setDisableTimeoutInMinutes(time);
                            found = true;
                            break;
                        }
                    }
                    if (found)
                        break;
                }
                if (!found) {
                    msg = _strMgr.getString("InstanceNotPartOfCluster");
                    throw new MBeanException (new ConfigException(msg));
                }
            } else {
                msg = _strMgr.getString("InvalidTarget");
                throw new MBeanException(new ConfigException(msg));
            }
        } catch(ConfigException ce) {
            throw new MBeanException(ce);
        }
    }

    private void disableCluster(String clusterName, String time) throws MBeanException {          

        Cluster c = null;
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                    .getAdminConfigContext();
            c = ClusterHelper.getClusterByName(ctx, clusterName);
        } catch(ConfigException ce) {
            throw new MBeanException(ce);
        }
        
        ServerRef[] sRefs = c.getServerRef();
        for (int i=0; i < sRefs.length; i++) {
            sRefs[i].setLbEnabled(false);
            sRefs[i].setDisableTimeoutInMinutes(time);
        }
    }
    
    public void enableServer(String target) throws MBeanException {
        
        ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                .getAdminConfigContext();

        try {
            if (ClusterHelper.isACluster(ctx, target)) {
                // enable cluster
                enableCluster(target);
            } else if (ServerHelper.isAServer(ctx, target)) {
                // enable server
                boolean found = false;
                Domain domain = (Domain) ctx.getRootConfigBean();
                Cluster[] clusters = domain.getClusters().getCluster();
                for (int i=0; i < clusters.length; i++) {
                    Cluster cluster = clusters[i];
                    ServerRef[] sRefs = cluster.getServerRef();
                    for (int j=0; j < sRefs.length; j++) {
                        if (sRefs[i].getRef().equals(target)) {
                            sRefs[i].setLbEnabled(true);
                            found = true;
                            break;
                        }
                    }
                    if (found)
                        break;
                }
                if (!found) {
                    String msg = _strMgr.getString("InstanceNotPartOfCluster");
                    throw new MBeanException (new ConfigException(msg));
                }
            } else {
                String msg = _strMgr.getString("InvalidTarget");
                throw new MBeanException(new ConfigException(msg));
            }
        } catch(ConfigException ce) {
            throw new MBeanException(ce);
        }
    }

    private void enableCluster(String clusterName) throws MBeanException {

        Cluster c = null;
        try {
            ConfigContext ctx = AdminService.getAdminService().getAdminContext()
                                    .getAdminConfigContext();
            c = ClusterHelper.getClusterByName(ctx, clusterName);
        } catch(ConfigException ce) {
            throw new MBeanException(ce);
        }
        
        ServerRef[] sRefs = c.getServerRef();
        for (int i=0; i < sRefs.length; i++)
            sRefs[i].setLbEnabled(true);
    }
}
