/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.admin;

import org.jvnet.glassfish.comms.clb.core.CLBRuntimeException;
import org.jvnet.glassfish.comms.clb.core.ClusterHealthMonitorFactory;
import org.jvnet.glassfish.comms.clb.core.Controller;
import org.jvnet.glassfish.comms.clb.core.ControllerInitializer;
import org.jvnet.glassfish.comms.util.LogUtil;
import org.jvnet.glassfish.comms.admin.event.extensions.clb.ConvergedLbEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.clb.ConvergedLbEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.clb.ConvergedLbPolicyEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.clb.ConvergedLbPolicyEventListener;

import org.w3c.dom.DOMException;

import org.xml.sax.SAXException;

import java.io.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.bind.JAXBException;
import javax.xml.parsers.ParserConfigurationException;

import com.sun.enterprise.admin.event.AdminEventListenerRegistry;

/**
 * This class is responsible for monitoring for configuration changes both through DAS changes and
 * by polling and periodic checking for changes to converged-loadbalancer.xml.
 */
public class CLBConfigurator {

    private static final Logger logger = LogUtil.CLB_LOGGER.getLogger();
    static CLBConfigurator sigletonCLBConfigurator = null;
    private static boolean adminEventListenersRegistered = false;
    File lbConfig = null;
    Date lastModifiedDate = null;
    Loadbalancer lb;
    ObjectFactory configFactory;
    ControllerInitializer sipControllerInitializer;
    ControllerInitializer httpControllerInitializer;
    private String configName;

    /**
     * public constructor to create a new CLBConfigurator instance
     *
     */
    private CLBConfigurator() {
        configFactory = new ObjectFactory();
    }

    public void init(String configName, String lbConfigFileName,
        ControllerInitializer sipControllerInitializer,
        ControllerInitializer httpControllerInitializer) {
        logger.log(
            Level.INFO, "Starting the initialization. ConveredLoadbalancer.xml location ... " +
            lbConfigFileName);
        registerAdminEventListeners();
        this.sipControllerInitializer = sipControllerInitializer;
        this.httpControllerInitializer = httpControllerInitializer;
        this.configName = configName;
        this.lbConfig = new File(configName + File.separator + lbConfigFileName);
        reload(lbConfigFileName);
        logger.log(Level.INFO, "Initialization completed sucessfully");
    }

    /**
     * Method to load the configration from the inputStream
     */
    public void load(InputStream in)
        throws JAXBException, DOMException, ParserConfigurationException,
        IOException, SAXException {
        lb = configFactory.load(in);
    }

    /**
     * Method to force a reload of the configration from the XML file
     */
    public void reload(String newLBConfigFileName) {
        InputStream in = null;
        boolean reconfig = false;
        try {
            lbConfig = new File(configName + File.separator + newLBConfigFileName);
            in = new FileInputStream(lbConfig);
            load(in);
            reconfig = true;
        } catch (FileNotFoundException e) {
            logger.log(Level.SEVERE, e.getMessage());
        } catch (Exception e) {
            logger.log(Level.SEVERE,
                "Exception occured while trying to load the xml file. " +
                e.getMessage());
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
            //no op
            }
        }
        if(reconfig)
            reconfig();
    }

    private void reconfig() {
        assert (sipControllerInitializer != null) &&
            (httpControllerInitializer != null);

        try {
            if (lb != null) {
                List<Cluster> clusters = lb.getCluster();

                if ((clusters == null) || (clusters.size() == 0)) {
                    logger.log(Level.WARNING,
                        "There are no clusters defined in this configuration ... Exiting reconfig");

                    return;
                }

                ClusterHealthMonitorFactory.createClusterHealthMonitors(clusters);

                Controller sipController = sipControllerInitializer.createController();
                Controller httpController = httpControllerInitializer.createController();
                logger.log(Level.INFO, "Adding cluster info ... There are " +
                    clusters.size() + " clusters in the configuration");

                for (int i = 0; i < clusters.size(); i++) {
                    Cluster cluster = clusters.get(i);
                    addCluster(cluster.getName(), cluster.getSelfLoadbalance(),
                        sipController, httpController);

                    List instances = cluster.getInstance();
                    logger.log(Level.INFO, "Adding Instance info ... There are " +
                        instances.size() + " instances in the configuration");
                    addInstances(cluster.getName(), instances, sipController,
                        httpController);

                    List webModules = cluster.getWebModule();
                    logger.log(Level.INFO, "Adding webModules info ... There are " +
                        webModules.size() + " webModules in the configuration");
                    addWebModules(cluster.getName(), webModules, sipController,
                        httpController);

                    logger.log(Level.INFO, "Setting the Policies ... HTTP Policy " +
                        lb.getHttpPolicy() + " SIP Policy " +
                        lb.getSipPolicy());
                    setPolicies(lb.getHttpPolicy(), lb.getSipPolicy(),
                        sipController, httpController);

                    logger.log(Level.INFO, "Setting the DCR filename " +
                        lb.getDcrFile());
                    if (lb.getDcrFile() == null || lb.getDcrFile().length() == 0) {
                        setDCRFileName(null, sipController, httpController);
                    } else {
                        setDCRFileName(configName + File.separator + lb.getDcrFile(),
                            sipController, httpController);
                    }

                    logger.log(Level.INFO,
                        "Configuration Successful. Setting the SIP and HTTP Controllers");
                    sipControllerInitializer.setController(sipController);
                    httpControllerInitializer.setController(httpController);
                }
            }
        } catch (CLBRuntimeException e) {
            logger.log(Level.SEVERE,
                "Configuration failed. Exiting without applying changes");
        }
    }

    private void addCluster(String clusterName, String selfLoadbalance,
        Controller sipController, Controller httpController)
        throws CLBRuntimeException {
        boolean bselfLoadbalance = Boolean.parseBoolean(selfLoadbalance);
        logger.log(
            Level.INFO, "Adding Cluster " + clusterName +
            " with selfloadbalance option " + selfLoadbalance);

        try {
            sipController.addCluster(clusterName, bselfLoadbalance);
            httpController.addCluster(clusterName, bselfLoadbalance);
        } catch (CLBRuntimeException e) {
            logger.log(Level.SEVERE,
                "CLBRuntimeException occured while trying to add cluster " +
                clusterName + " " + e.getMessage());
            throw e;
        }
    }

    private void addInstances(String clusterName, List instances,
        Controller sipController, Controller httpController)
        throws CLBRuntimeException {
        if (instances == null) {
            return;
        }

        for (int i = 0; i < instances.size(); i++) {
            Instance instance = (Instance) instances.get(i);
            logger.log(
                Level.INFO, "Parsing instance " + instance.getName());

            String strListeners = instance.getListeners();
            String[] listeners = parseListeners(strListeners);
            addInstance(clusterName, instance.getName(), instance.enabled,
                instance.getDisableTimeoutInMinutes(), listeners,
                sipController, httpController);
        }
    }

    private String[] parseListeners(String listeners) {
        if (listeners == null) {
            return new String[0];
        }

        StringTokenizer tokenizer = new StringTokenizer(listeners, " ");
        List<String> listenersList = new ArrayList<String>();

        while (tokenizer.hasMoreTokens()) {
            String listener = tokenizer.nextToken();
            logger.log(
                Level.INFO, "Parsing listener " + listener);
            listenersList.add(listener);
        }

        return listenersList.toArray(new String[listenersList.size()]);
    }

    private void addInstance(String clusterName, String instanceName,
        String strEnabled, String strTimeoutInMinutes, String[] listeners,
        Controller sipController, Controller httpController)
        throws CLBRuntimeException {
        boolean enabled = Boolean.parseBoolean(strEnabled);
        int timeoutInMinutes = Integer.parseInt(strTimeoutInMinutes);
        logger.log(
            Level.INFO, "Adding Instance - " + instanceName + " to Cluster " +
            clusterName);

        try {
            sipController.addInstance(clusterName, instanceName, enabled,
                timeoutInMinutes, listeners);
            httpController.addInstance(clusterName, instanceName, enabled,
                timeoutInMinutes, listeners);
        } catch (CLBRuntimeException e) {
            logger.log(Level.SEVERE,
                "CLBRuntimeException occured while trying to add instance " +
                instanceName + " " + e.getMessage());
            throw e;
        }
    }

    private void addWebModules(String clusterName, List webModules,
        Controller sipController, Controller httpController)
        throws CLBRuntimeException {
        assert webModules != null;

        for (int i = 0; i < webModules.size(); i++) {
            WebModule webModule = (WebModule) webModules.get(i);
            addWebModule(clusterName, webModule.getContextRoot(),
                sipController, httpController);
        }
    }

    private void addWebModule(String clusterName, String contextRoot,
        Controller sipController, Controller httpController)
        throws CLBRuntimeException {
        try {
            logger.log(
                Level.INFO, "Adding web module with context root " +
                contextRoot + " to cluster " + clusterName);
            httpController.addWebModule(clusterName, contextRoot);
        } catch (CLBRuntimeException e) {
            logger.log(Level.SEVERE,
                "CLBRuntimeException occured while trying to add web module " +
                contextRoot + " " + e.getMessage());
            throw e;
        }
    }

    private void setPolicies(String HTTPPolicy, String SIPPolicy,
        Controller sipController, Controller httpController) {
        httpController.setPolicy(HTTPPolicy);
        sipController.setPolicy(SIPPolicy);
    }

    private void setDCRFileName(String name, Controller sipController,
        Controller httpController) {
        httpController.setDCRFileName(name);
        sipController.setDCRFileName(name);
    }

    public String getDCRFileName() {
        if (lb == null) {
            return null;
        }

        return lb.getDcrFile();
    }

    public boolean isSelfLoadbalanced() {
        if ((lb != null) && (lb.getCluster() != null) &&
            (lb.getCluster().get(0) != null)) {
            return new Boolean(lb.getCluster().get(0).getSelfLoadbalance()).booleanValue();
        }

        return false;
    }

    public ControllerInitializer getSipControllerInitializer() {
        return sipControllerInitializer;
    }

    public ControllerInitializer getHttpControllerInitializer() {
        return httpControllerInitializer;
    }

    /**
     * Method to check if the converged-loadbalancer.xml has changed
     *
     * @return a boolean indicating whether the configuration has changed
     */
    protected boolean checkIfConfigChanged() {
        if (lastModifiedDate.before(new Date(this.lbConfig.lastModified()))) {
            return true;
        }

        return false;
    }

    public static CLBConfigurator getInstance() {
        return sigletonCLBConfigurator;
    }
    
    public static CLBConfigurator createInstance() {
        if (sigletonCLBConfigurator == null) {
            sigletonCLBConfigurator = new CLBConfigurator();
        }

        return sigletonCLBConfigurator;
    }

    public static void registerAdminEventListeners() {
        if (adminEventListenersRegistered) {
            return;
        }

        adminEventListenersRegistered = true;

        //Register the event listeners for  Dynamic reconfig  
        //If CLBConfigurator is null, then no need to register admin event listener
        //for converged load-balancer xml updates
        if (CLBConfigurator.getInstance() != null) {
            AdminEventListenerRegistry.registerEventListener(ConvergedLbEvent.eventType,
                new ConvergedLBEventListenerImpl(), ConvergedLbEventListener.class);
        }

        AdminEventListenerRegistry.registerEventListener(ConvergedLbPolicyEvent.eventType,
            new ConvergedLBPolicyEventListenerImpl(), ConvergedLbPolicyEventListener.class);
    }
}
